/*
 * Copyright 2010-2020 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.expressions.impl

import ksp.org.jetbrains.kotlin.KtFakeSourceElementKind
import ksp.org.jetbrains.kotlin.KtSourceElement
import ksp.org.jetbrains.kotlin.fakeElement
import ksp.org.jetbrains.kotlin.fir.MutableOrEmptyList
import ksp.org.jetbrains.kotlin.fir.builder.toMutableOrEmpty
import ksp.org.jetbrains.kotlin.fir.expressions.FirAnnotation
import ksp.org.jetbrains.kotlin.fir.expressions.FirBlock
import ksp.org.jetbrains.kotlin.fir.expressions.FirStatement
import ksp.org.jetbrains.kotlin.fir.expressions.UnresolvedExpressionTypeAccess
import ksp.org.jetbrains.kotlin.fir.types.ConeKotlinType
import ksp.org.jetbrains.kotlin.fir.visitors.FirTransformer
import ksp.org.jetbrains.kotlin.fir.visitors.FirVisitor
import ksp.org.jetbrains.kotlin.fir.visitors.transformInplace
import ksp.org.jetbrains.kotlin.fir.visitors.transformSingle

@OptIn(UnresolvedExpressionTypeAccess::class)
class FirSingleExpressionBlock(
    var statement: FirStatement
) : FirBlock() {
    override val source: KtSourceElement?
        get() = statement.source?.fakeElement(KtFakeSourceElementKind.SingleExpressionBlock)
    override var annotations: MutableOrEmptyList<FirAnnotation> = MutableOrEmptyList.empty()
    override val statements: List<FirStatement> get() = listOf(statement)

    @UnresolvedExpressionTypeAccess
    override var coneTypeOrNull: ConeKotlinType? = null
    override var isUnitCoerced: Boolean = false

    override fun <R, D> acceptChildren(visitor: FirVisitor<R, D>, data: D) {
        annotations.forEach { it.accept(visitor, data) }
        statement.accept(visitor, data)
    }

    override fun <D> transformChildren(transformer: FirTransformer<D>, data: D): FirSingleExpressionBlock {
        transformStatements(transformer, data)
        transformOtherChildren(transformer, data)
        return this
    }

    override fun replaceConeTypeOrNull(newConeTypeOrNull: ConeKotlinType?) {
        coneTypeOrNull = newConeTypeOrNull
    }

    override fun <D> transformStatements(transformer: FirTransformer<D>, data: D): FirBlock {
        statement = statement.transformSingle(transformer, data)
        return this
    }

    override fun <D> transformOtherChildren(transformer: FirTransformer<D>, data: D): FirBlock {
        transformAnnotations(transformer, data)
        return this
    }

    override fun replaceAnnotations(newAnnotations: List<FirAnnotation>) {
        annotations = newAnnotations.toMutableOrEmpty()
    }

    override fun <D> transformAnnotations(transformer: FirTransformer<D>, data: D): FirBlock {
        annotations.transformInplace(transformer, data)
        return this
    }

    override fun replaceIsUnitCoerced(newIsUnitCoerced: Boolean) {
        isUnitCoerced = newIsUnitCoerced
    }
}

@Suppress("NOTHING_TO_INLINE")
inline fun buildSingleExpressionBlock(statement: FirStatement): FirBlock {
    return FirSingleExpressionBlock(statement)
}
