/*
 * Copyright 2010-2025 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.light.classes.symbol.classes

import ksp.com.intellij.psi.PsiManager
import ksp.com.intellij.psi.PsiMethod
import ksp.com.intellij.psi.PsiReferenceList
import ksp.org.jetbrains.kotlin.analysis.api.projectStructure.KaModule
import ksp.org.jetbrains.kotlin.analysis.api.symbols.*
import ksp.org.jetbrains.kotlin.analysis.api.symbols.pointers.KaSymbolPointer
import ksp.org.jetbrains.kotlin.asJava.classes.lazyPub
import ksp.org.jetbrains.kotlin.light.classes.symbol.cachedValue
import ksp.org.jetbrains.kotlin.psi.KtClass
import ksp.org.jetbrains.kotlin.psi.KtClassOrObject

internal open class SymbolLightClassForInterface : SymbolLightClassForInterfaceOrAnnotationClass {
    constructor(
        ktModule: KaModule,
        classSymbol: KaNamedClassSymbol,
        manager: PsiManager
    ) : super(
        ktModule = ktModule,
        classSymbol = classSymbol,
        manager = manager,
    ) {
        require(classSymbol.classKind == KaClassKind.INTERFACE)
    }

    constructor(classOrObject: KtClassOrObject, ktModule: KaModule) : super(classOrObject, ktModule) {
        require(classOrObject is KtClass && classOrObject.isInterface())
    }

    protected constructor(
        classOrObjectDeclaration: KtClassOrObject?,
        classSymbolPointer: KaSymbolPointer<KaNamedClassSymbol>,
        ktModule: KaModule,
        manager: PsiManager,
    ) : super(
        classOrObjectDeclaration = classOrObjectDeclaration,
        classSymbolPointer = classSymbolPointer,
        ktModule = ktModule,
        manager = manager,
    )

    override fun getOwnMethods(): List<PsiMethod> = cachedValue {
        withClassSymbol { classSymbol ->
            val result = mutableListOf<PsiMethod>()

            val visibleDeclarations = classSymbol.declaredMemberScope.callables.filter { acceptCallableSymbol(it) }

            createMethods(this@SymbolLightClassForInterface, visibleDeclarations, result)
            addMethodsFromCompanionIfNeeded(result, classSymbol)

            result
        }
    }

    protected open fun acceptCallableSymbol(symbol: KaCallableSymbol): Boolean {
        return symbol !is KaNamedFunctionSymbol || symbol.visibility != KaSymbolVisibility.PRIVATE
    }

    override fun copy(): SymbolLightClassForInterface =
        SymbolLightClassForInterface(classOrObjectDeclaration, classSymbolPointer, ktModule, manager)

    private val _extendsList: PsiReferenceList by lazyPub {
        withClassSymbol { classSymbol ->
            createInheritanceList(this@SymbolLightClassForInterface, forExtendsList = true, classSymbol.superTypes)
        }
    }

    override fun getExtendsList(): PsiReferenceList? = _extendsList
    override fun classKind(): KaClassKind = KaClassKind.INTERFACE
}
