/*
 * Copyright 2010-2025 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.psi;

import ksp.com.intellij.lang.ASTNode;
import ksp.com.intellij.psi.PsiElement;
import ksp.com.intellij.psi.tree.TokenSet;
import ksp.com.intellij.psi.util.PsiTreeUtil;
import ksp.org.jetbrains.annotations.NotNull;
import ksp.org.jetbrains.annotations.Nullable;
import ksp.org.jetbrains.kotlin.KtNodeTypes;
import ksp.org.jetbrains.kotlin.lexer.KtTokens;
import ksp.org.jetbrains.kotlin.psi.psiUtil.KtPsiUtilKt;

import java.util.List;

import static org.jetbrains.kotlin.lexer.KtTokens.*;

public class KtDestructuringDeclaration extends KtDeclarationImpl
        implements KtValVarKeywordOwner, KtDeclarationWithInitializer, KtDeclarationWithReturnType {

    public KtDestructuringDeclaration(@NotNull ASTNode node) {
        super(node);
    }

    @Override
    public <R, D> R accept(@NotNull KtVisitor<R, D> visitor, D data) {
        return visitor.visitDestructuringDeclaration(this, data);
    }

    @NotNull
    public List<KtDestructuringDeclarationEntry> getEntries() {
        return findChildrenByType(KtNodeTypes.DESTRUCTURING_DECLARATION_ENTRY);
    }

    @Nullable
    @Override
    public KtExpression getInitializer() {
        ASTNode eqNode = getNode().findChildByType(EQ);
        if (eqNode == null) {
            return null;
        }
        return PsiTreeUtil.getNextSiblingOfType(eqNode.getPsi(), KtExpression.class);
    }

    @Override
    public boolean hasInitializer() {
        return getInitializer() != null;
    }

    public boolean isVar() {
        return getNode().findChildByType(KtTokens.VAR_KEYWORD) != null;
    }

    @Override
    @Nullable
    public PsiElement getValOrVarKeyword() {
        return findChildByType(VAL_VAR);
    }

    /**
     * Example:
     * <code>
     *     (val x, var y) = ...
     * </code>
     *
     * @return true when this destructuring declaration uses the full form with the val or var keywords inside the entries.
     * See <a href="https://kotl.in/name-based-destructuring">KEEP</a>.
     */
    public boolean isFullForm() {
        List<KtDestructuringDeclarationEntry> entries = getEntries();
        return !entries.isEmpty() && entries.get(0).getOwnValOrVarKeyword() != null;
    }

    private static final TokenSet OPENING_BRACES = TokenSet.create(LPAR, LBRACKET);
    private static final TokenSet CLOSING_BRACES = TokenSet.create(RPAR, RBRACKET);

    @Nullable
    public PsiElement getRPar() {
        return findChildByType(CLOSING_BRACES);
    }

    @Nullable
    public PsiElement getLPar() {
        return findChildByType(OPENING_BRACES);
    }

    /**
     * @return true when this destructuring declaration uses square brackets.
     */
    public boolean hasSquareBrackets() {
        return getNode().findChildByType(LBRACKET) != null;
    }

    @Nullable
    public PsiElement getTrailingComma() {
        return KtPsiUtilKt.getTrailingCommaByClosingElement(getRPar());
    }

    @Nullable
    @Override
    public KtTypeReference getTypeReference() {
        return null;
    }
}
