/*
 * Copyright 2010-2023 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.wasm.resolve.diagnostics

import ksp.org.jetbrains.kotlin.descriptors.DeclarationDescriptor
import ksp.org.jetbrains.kotlin.descriptors.FunctionDescriptor
import ksp.org.jetbrains.kotlin.descriptors.MemberDescriptor
import ksp.org.jetbrains.kotlin.descriptors.PropertyAccessorDescriptor
import ksp.org.jetbrains.kotlin.js.resolve.diagnostics.findPsi
import ksp.org.jetbrains.kotlin.name.FqName
import ksp.org.jetbrains.kotlin.psi.KtDeclaration
import ksp.org.jetbrains.kotlin.resolve.DescriptorUtils
import ksp.org.jetbrains.kotlin.resolve.checkers.DeclarationChecker
import ksp.org.jetbrains.kotlin.resolve.checkers.DeclarationCheckerContext
import ksp.org.jetbrains.kotlin.resolve.descriptorUtil.isEffectivelyExternal

object WasmWasiExternalDeclarationChecker : DeclarationChecker {
    override fun check(declaration: KtDeclaration, descriptor: DeclarationDescriptor, context: DeclarationCheckerContext) {
        if (descriptor !is MemberDescriptor) return
        if (descriptor is PropertyAccessorDescriptor) return
        if (!DescriptorUtils.isTopLevelDeclaration(descriptor)) return
        if (!descriptor.isEffectivelyExternal()) return

        if (descriptor is FunctionDescriptor) {
            if (!descriptor.annotations.hasAnnotation(FqName("kotlin.wasm.WasmImport"))) {
                val reportOn = descriptor.findPsi() ?: declaration
                context.trace.report(ErrorsWasm.WASI_EXTERNAL_FUNCTION_WITHOUT_IMPORT.on(reportOn))
            }
        } else {
            val reportOn = descriptor.findPsi() ?: declaration
            context.trace.report(ErrorsWasm.WASI_EXTERNAL_NOT_TOP_LEVEL_FUNCTION.on(reportOn))
        }
    }
}