/*
 * Decompiled with CFR 0.152.
 */
package com.google.firebase.database.android;

import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteDatabaseLockedException;
import android.database.sqlite.SQLiteException;
import android.database.sqlite.SQLiteOpenHelper;
import com.google.firebase.database.DatabaseException;
import com.google.firebase.database.core.CompoundWrite;
import com.google.firebase.database.core.Path;
import com.google.firebase.database.core.UserWriteRecord;
import com.google.firebase.database.core.persistence.PersistenceStorageEngine;
import com.google.firebase.database.core.persistence.PruneForest;
import com.google.firebase.database.core.persistence.TrackedQuery;
import com.google.firebase.database.core.utilities.ImmutableTree;
import com.google.firebase.database.core.utilities.NodeSizeEstimator;
import com.google.firebase.database.core.utilities.Pair;
import com.google.firebase.database.core.utilities.Utilities;
import com.google.firebase.database.core.view.QuerySpec;
import com.google.firebase.database.logging.LogWrapper;
import com.google.firebase.database.snapshot.ChildKey;
import com.google.firebase.database.snapshot.ChildrenNode;
import com.google.firebase.database.snapshot.EmptyNode;
import com.google.firebase.database.snapshot.NamedNode;
import com.google.firebase.database.snapshot.Node;
import com.google.firebase.database.snapshot.NodeUtilities;
import com.google.firebase.database.util.JsonMapper;
import java.io.IOException;
import java.net.URLEncoder;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;

public class SqlPersistenceStorageEngine
implements PersistenceStorageEngine {
    private static final String CREATE_SERVER_CACHE = "CREATE TABLE serverCache (path TEXT PRIMARY KEY, value BLOB);";
    private static final String SERVER_CACHE_TABLE = "serverCache";
    private static final String PATH_COLUMN_NAME = "path";
    private static final String VALUE_COLUMN_NAME = "value";
    private static final String CREATE_WRITES = "CREATE TABLE writes (id INTEGER, path TEXT, type TEXT, part INTEGER, node BLOB, UNIQUE (id, part));";
    private static final String WRITES_TABLE = "writes";
    private static final String WRITE_ID_COLUMN_NAME = "id";
    private static final String WRITE_NODE_COLUMN_NAME = "node";
    private static final String WRITE_PART_COLUMN_NAME = "part";
    private static final String WRITE_TYPE_COLUMN_NAME = "type";
    private static final String WRITE_TYPE_OVERWRITE = "o";
    private static final String WRITE_TYPE_MERGE = "m";
    private static final String CREATE_TRACKED_QUERIES = "CREATE TABLE trackedQueries (id INTEGER PRIMARY KEY, path TEXT, queryParams TEXT, lastUse INTEGER, complete INTEGER, active INTEGER);";
    private static final String TRACKED_QUERY_TABLE = "trackedQueries";
    private static final String TRACKED_QUERY_ID_COLUMN_NAME = "id";
    private static final String TRACKED_QUERY_PATH_COLUMN_NAME = "path";
    private static final String TRACKED_QUERY_PARAMS_COLUMN_NAME = "queryParams";
    private static final String TRACKED_QUERY_LAST_USE_COLUMN_NAME = "lastUse";
    private static final String TRACKED_QUERY_COMPLETE_COLUMN_NAME = "complete";
    private static final String TRACKED_QUERY_ACTIVE_COLUMN_NAME = "active";
    private static final String CREATE_TRACKED_KEYS = "CREATE TABLE trackedKeys (id INTEGER, key TEXT);";
    private static final String TRACKED_KEYS_TABLE = "trackedKeys";
    private static final String TRACKED_KEYS_ID_COLUMN_NAME = "id";
    private static final String TRACKED_KEYS_KEY_COLUMN_NAME = "key";
    private static final String ROW_ID_COLUMN_NAME = "rowid";
    private static final int CHILDREN_NODE_SPLIT_SIZE_THRESHOLD = 16384;
    private static final int ROW_SPLIT_SIZE = 262144;
    private static final String PART_KEY_FORMAT = ".part-%04d";
    private static final String FIRST_PART_KEY = ".part-0000";
    private static final String PART_KEY_PREFIX = ".part-";
    private static final Charset UTF8_CHARSET = Charset.forName("UTF-8");
    private static final String LOGGER_COMPONENT = "Persistence";
    private final SQLiteDatabase database;
    private final LogWrapper logger;
    private boolean insideTransaction;
    private long transactionStart = 0L;

    public SqlPersistenceStorageEngine(Context context, com.google.firebase.database.core.Context firebaseContext, String cacheId) {
        String sanitizedCacheId;
        try {
            sanitizedCacheId = URLEncoder.encode(cacheId, "utf-8");
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
        this.logger = firebaseContext.getLogger(LOGGER_COMPONENT);
        this.database = this.openDatabase(context, sanitizedCacheId);
    }

    @Override
    public void saveUserOverwrite(Path path, Node node, long writeId) {
        this.verifyInsideTransaction();
        long start = System.currentTimeMillis();
        byte[] serializedNode = this.serializeObject(node.getValue(true));
        this.saveWrite(path, writeId, WRITE_TYPE_OVERWRITE, serializedNode);
        long duration = System.currentTimeMillis() - start;
        if (this.logger.logsDebug()) {
            this.logger.debug(String.format(Locale.US, "Persisted user overwrite in %dms", duration), new Object[0]);
        }
    }

    @Override
    public void saveUserMerge(Path path, CompoundWrite children, long writeId) {
        this.verifyInsideTransaction();
        long start = System.currentTimeMillis();
        byte[] serializedNode = this.serializeObject(children.getValue(true));
        this.saveWrite(path, writeId, WRITE_TYPE_MERGE, serializedNode);
        long duration = System.currentTimeMillis() - start;
        if (this.logger.logsDebug()) {
            this.logger.debug(String.format(Locale.US, "Persisted user merge in %dms", duration), new Object[0]);
        }
    }

    @Override
    public void removeUserWrite(long writeId) {
        this.verifyInsideTransaction();
        long start = System.currentTimeMillis();
        int count = this.database.delete(WRITES_TABLE, "id = ?", new String[]{String.valueOf(writeId)});
        long duration = System.currentTimeMillis() - start;
        if (this.logger.logsDebug()) {
            this.logger.debug(String.format(Locale.US, "Deleted %d write(s) with writeId %d in %dms", count, writeId, duration), new Object[0]);
        }
    }

    @Override
    public List<UserWriteRecord> loadUserWrites() {
        String[] columns = new String[]{"id", "path", WRITE_TYPE_COLUMN_NAME, WRITE_PART_COLUMN_NAME, WRITE_NODE_COLUMN_NAME};
        long start = System.currentTimeMillis();
        ArrayList<UserWriteRecord> writes = new ArrayList<UserWriteRecord>();
        try (Cursor cursor = this.database.query(WRITES_TABLE, columns, null, null, null, null, "id, part");){
            while (cursor.moveToNext()) {
                UserWriteRecord record;
                byte[] serialized;
                long writeId = cursor.getLong(0);
                Path path = new Path(cursor.getString(1));
                String type = cursor.getString(2);
                if (cursor.isNull(3)) {
                    serialized = cursor.getBlob(4);
                } else {
                    ArrayList<byte[]> parts = new ArrayList<byte[]>();
                    do {
                        parts.add(cursor.getBlob(4));
                    } while (cursor.moveToNext() && cursor.getLong(0) == writeId);
                    cursor.moveToPrevious();
                    serialized = this.joinBytes(parts);
                }
                String serializedString = new String(serialized, UTF8_CHARSET);
                Object writeValue = JsonMapper.parseJsonValue(serializedString);
                if (WRITE_TYPE_OVERWRITE.equals(type)) {
                    Node set = NodeUtilities.NodeFromJSON(writeValue);
                    record = new UserWriteRecord(writeId, path, set, true);
                } else if (WRITE_TYPE_MERGE.equals(type)) {
                    CompoundWrite merge = CompoundWrite.fromValue((Map)writeValue);
                    record = new UserWriteRecord(writeId, path, merge);
                } else {
                    throw new IllegalStateException("Got invalid write type: " + type);
                }
                writes.add(record);
            }
            long duration = System.currentTimeMillis() - start;
            if (this.logger.logsDebug()) {
                this.logger.debug(String.format(Locale.US, "Loaded %d writes in %dms", writes.size(), duration), new Object[0]);
            }
            ArrayList<UserWriteRecord> arrayList = writes;
            return arrayList;
        }
    }

    private void saveWrite(Path path, long writeId, String type, byte[] serializedWrite) {
        this.verifyInsideTransaction();
        this.database.delete(WRITES_TABLE, "id = ?", new String[]{String.valueOf(writeId)});
        if (serializedWrite.length >= 262144) {
            List<byte[]> parts = SqlPersistenceStorageEngine.splitBytes(serializedWrite, 262144);
            for (int i = 0; i < parts.size(); ++i) {
                ContentValues values = new ContentValues();
                values.put("id", Long.valueOf(writeId));
                values.put("path", SqlPersistenceStorageEngine.pathToKey(path));
                values.put(WRITE_TYPE_COLUMN_NAME, type);
                values.put(WRITE_PART_COLUMN_NAME, Integer.valueOf(i));
                values.put(WRITE_NODE_COLUMN_NAME, parts.get(i));
                this.database.insertWithOnConflict(WRITES_TABLE, null, values, 5);
            }
        } else {
            ContentValues values = new ContentValues();
            values.put("id", Long.valueOf(writeId));
            values.put("path", SqlPersistenceStorageEngine.pathToKey(path));
            values.put(WRITE_TYPE_COLUMN_NAME, type);
            values.put(WRITE_PART_COLUMN_NAME, (Integer)null);
            values.put(WRITE_NODE_COLUMN_NAME, serializedWrite);
            this.database.insertWithOnConflict(WRITES_TABLE, null, values, 5);
        }
    }

    @Override
    public Node serverCache(Path path) {
        return this.loadNested(path);
    }

    @Override
    public void overwriteServerCache(Path path, Node node) {
        this.verifyInsideTransaction();
        this.updateServerCache(path, node, false);
    }

    @Override
    public void mergeIntoServerCache(Path path, Node node) {
        this.verifyInsideTransaction();
        this.updateServerCache(path, node, true);
    }

    private void updateServerCache(Path path, Node node, boolean merge) {
        int savedRows;
        int removedRows;
        long start = System.currentTimeMillis();
        if (!merge) {
            removedRows = this.removeNested(SERVER_CACHE_TABLE, path);
            savedRows = this.saveNested(path, node);
        } else {
            removedRows = 0;
            savedRows = 0;
            for (NamedNode child : node) {
                removedRows += this.removeNested(SERVER_CACHE_TABLE, path.child(child.getName()));
                savedRows += this.saveNested(path.child(child.getName()), child.getNode());
            }
        }
        long duration = System.currentTimeMillis() - start;
        if (this.logger.logsDebug()) {
            this.logger.debug(String.format(Locale.US, "Persisted a total of %d rows and deleted %d rows for a set at %s in %dms", savedRows, removedRows, path.toString(), duration), new Object[0]);
        }
    }

    @Override
    public void mergeIntoServerCache(Path path, CompoundWrite children) {
        this.verifyInsideTransaction();
        long start = System.currentTimeMillis();
        int savedRows = 0;
        int removedRows = 0;
        for (Map.Entry<Path, Node> entry : children) {
            removedRows += this.removeNested(SERVER_CACHE_TABLE, path.child(entry.getKey()));
            savedRows += this.saveNested(path.child(entry.getKey()), entry.getValue());
        }
        long duration = System.currentTimeMillis() - start;
        if (this.logger.logsDebug()) {
            this.logger.debug(String.format(Locale.US, "Persisted a total of %d rows and deleted %d rows for a merge at %s in %dms", savedRows, removedRows, path.toString(), duration), new Object[0]);
        }
    }

    @Override
    public long serverCacheEstimatedSizeInBytes() {
        String query = String.format("SELECT sum(length(%s) + length(%s)) FROM %s", VALUE_COLUMN_NAME, "path", SERVER_CACHE_TABLE);
        try (Cursor cursor = this.database.rawQuery(query, null);){
            if (cursor.moveToFirst()) {
                long l = cursor.getLong(0);
                return l;
            }
            throw new IllegalStateException("Couldn't read database result!");
        }
    }

    @Override
    public void saveTrackedQuery(TrackedQuery trackedQuery) {
        this.verifyInsideTransaction();
        long start = System.currentTimeMillis();
        ContentValues values = new ContentValues();
        values.put("id", Long.valueOf(trackedQuery.id));
        values.put("path", SqlPersistenceStorageEngine.pathToKey(trackedQuery.querySpec.getPath()));
        values.put(TRACKED_QUERY_PARAMS_COLUMN_NAME, trackedQuery.querySpec.getParams().toJSON());
        values.put(TRACKED_QUERY_LAST_USE_COLUMN_NAME, Long.valueOf(trackedQuery.lastUse));
        values.put(TRACKED_QUERY_COMPLETE_COLUMN_NAME, Boolean.valueOf(trackedQuery.complete));
        values.put(TRACKED_QUERY_ACTIVE_COLUMN_NAME, Boolean.valueOf(trackedQuery.active));
        this.database.insertWithOnConflict(TRACKED_QUERY_TABLE, null, values, 5);
        long duration = System.currentTimeMillis() - start;
        if (this.logger.logsDebug()) {
            this.logger.debug(String.format(Locale.US, "Saved new tracked query in %dms", duration), new Object[0]);
        }
    }

    @Override
    public void deleteTrackedQuery(long trackedQueryId) {
        this.verifyInsideTransaction();
        String trackedQueryIdStr = String.valueOf(trackedQueryId);
        String queriesWhereClause = "id = ?";
        this.database.delete(TRACKED_QUERY_TABLE, queriesWhereClause, new String[]{trackedQueryIdStr});
        String keysWhereClause = "id = ?";
        this.database.delete(TRACKED_KEYS_TABLE, keysWhereClause, new String[]{trackedQueryIdStr});
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public List<TrackedQuery> loadTrackedQueries() {
        String[] columns = new String[]{"id", "path", TRACKED_QUERY_PARAMS_COLUMN_NAME, TRACKED_QUERY_LAST_USE_COLUMN_NAME, TRACKED_QUERY_COMPLETE_COLUMN_NAME, TRACKED_QUERY_ACTIVE_COLUMN_NAME};
        long start = System.currentTimeMillis();
        ArrayList<TrackedQuery> queries = new ArrayList<TrackedQuery>();
        try (Cursor cursor = this.database.query(TRACKED_QUERY_TABLE, columns, null, null, null, null, "id");){
            while (cursor.moveToNext()) {
                Map<String, Object> paramsObject;
                long id = cursor.getLong(0);
                Path path = new Path(cursor.getString(1));
                String paramsStr = cursor.getString(2);
                try {
                    paramsObject = JsonMapper.parseJson(paramsStr);
                }
                catch (IOException e) {
                    throw new RuntimeException(e);
                }
                QuerySpec query = QuerySpec.fromPathAndQueryObject(path, paramsObject);
                long lastUse = cursor.getLong(3);
                boolean complete = cursor.getInt(4) != 0;
                boolean active = cursor.getInt(5) != 0;
                TrackedQuery trackedQuery = new TrackedQuery(id, query, lastUse, complete, active);
                queries.add(trackedQuery);
            }
            long duration = System.currentTimeMillis() - start;
            if (this.logger.logsDebug()) {
                this.logger.debug(String.format(Locale.US, "Loaded %d tracked queries in %dms", queries.size(), duration), new Object[0]);
            }
            ArrayList<TrackedQuery> arrayList = queries;
            return arrayList;
        }
    }

    @Override
    public void resetPreviouslyActiveTrackedQueries(long lastUse) {
        this.verifyInsideTransaction();
        long start = System.currentTimeMillis();
        String whereClause = "active = 1";
        ContentValues values = new ContentValues();
        values.put(TRACKED_QUERY_ACTIVE_COLUMN_NAME, Boolean.valueOf(false));
        values.put(TRACKED_QUERY_LAST_USE_COLUMN_NAME, Long.valueOf(lastUse));
        this.database.updateWithOnConflict(TRACKED_QUERY_TABLE, values, whereClause, new String[0], 5);
        long duration = System.currentTimeMillis() - start;
        if (this.logger.logsDebug()) {
            this.logger.debug(String.format(Locale.US, "Reset active tracked queries in %dms", duration), new Object[0]);
        }
    }

    @Override
    public void saveTrackedQueryKeys(long trackedQueryId, Set<ChildKey> keys) {
        this.verifyInsideTransaction();
        long start = System.currentTimeMillis();
        String trackedQueryIdStr = String.valueOf(trackedQueryId);
        String keysWhereClause = "id = ?";
        this.database.delete(TRACKED_KEYS_TABLE, keysWhereClause, new String[]{trackedQueryIdStr});
        for (ChildKey addedKey : keys) {
            ContentValues values = new ContentValues();
            values.put("id", Long.valueOf(trackedQueryId));
            values.put(TRACKED_KEYS_KEY_COLUMN_NAME, addedKey.asString());
            this.database.insertWithOnConflict(TRACKED_KEYS_TABLE, null, values, 5);
        }
        long duration = System.currentTimeMillis() - start;
        if (this.logger.logsDebug()) {
            this.logger.debug(String.format(Locale.US, "Set %d tracked query keys for tracked query %d in %dms", keys.size(), trackedQueryId, duration), new Object[0]);
        }
    }

    @Override
    public void updateTrackedQueryKeys(long trackedQueryId, Set<ChildKey> added, Set<ChildKey> removed) {
        this.verifyInsideTransaction();
        long start = System.currentTimeMillis();
        String whereClause = "id = ? AND key = ?";
        String trackedQueryIdStr = String.valueOf(trackedQueryId);
        for (ChildKey removedKey : removed) {
            this.database.delete(TRACKED_KEYS_TABLE, whereClause, new String[]{trackedQueryIdStr, removedKey.asString()});
        }
        for (ChildKey addedKey : added) {
            ContentValues values = new ContentValues();
            values.put("id", Long.valueOf(trackedQueryId));
            values.put(TRACKED_KEYS_KEY_COLUMN_NAME, addedKey.asString());
            this.database.insertWithOnConflict(TRACKED_KEYS_TABLE, null, values, 5);
        }
        long duration = System.currentTimeMillis() - start;
        if (this.logger.logsDebug()) {
            this.logger.debug(String.format(Locale.US, "Updated tracked query keys (%d added, %d removed) for tracked query id %d in %dms", added.size(), removed.size(), trackedQueryId, duration), new Object[0]);
        }
    }

    @Override
    public Set<ChildKey> loadTrackedQueryKeys(long trackedQueryId) {
        return this.loadTrackedQueryKeys(Collections.singleton(trackedQueryId));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public Set<ChildKey> loadTrackedQueryKeys(Set<Long> trackedQueryIds) {
        String[] columns = new String[]{TRACKED_KEYS_KEY_COLUMN_NAME};
        long start = System.currentTimeMillis();
        String whereClause = "id IN (" + this.commaSeparatedList(trackedQueryIds) + ")";
        HashSet<ChildKey> keys = new HashSet<ChildKey>();
        try (Cursor cursor = this.database.query(true, TRACKED_KEYS_TABLE, columns, whereClause, null, null, null, null, null);){
            while (cursor.moveToNext()) {
                String key = cursor.getString(0);
                keys.add(ChildKey.fromString(key));
            }
            long duration = System.currentTimeMillis() - start;
            if (this.logger.logsDebug()) {
                this.logger.debug(String.format(Locale.US, "Loaded %d tracked queries keys for tracked queries %s in %dms", keys.size(), trackedQueryIds.toString(), duration), new Object[0]);
            }
            HashSet<ChildKey> hashSet = keys;
            return hashSet;
        }
    }

    @Override
    public void pruneCache(Path root, PruneForest pruneForest) {
        if (!pruneForest.prunesAnything()) {
            return;
        }
        this.verifyInsideTransaction();
        long start = System.currentTimeMillis();
        Cursor cursor = this.loadNestedQuery(root, new String[]{ROW_ID_COLUMN_NAME, "path"});
        ImmutableTree<Object> rowIdsToPrune = new ImmutableTree<Object>(null);
        ImmutableTree<Object> rowIdsToKeep = new ImmutableTree<Object>(null);
        while (cursor.moveToNext()) {
            long rowId = cursor.getLong(0);
            Path rowPath = new Path(cursor.getString(1));
            if (!root.contains(rowPath)) {
                this.logger.warn("We are pruning at " + root + " but we have data stored higher up at " + rowPath + ". Ignoring.");
                continue;
            }
            Path relativePath = Path.getRelative(root, rowPath);
            if (pruneForest.shouldPruneUnkeptDescendants(relativePath)) {
                rowIdsToPrune = rowIdsToPrune.set(relativePath, rowId);
                continue;
            }
            if (pruneForest.shouldKeep(relativePath)) {
                rowIdsToKeep = rowIdsToKeep.set(relativePath, rowId);
                continue;
            }
            this.logger.warn("We are pruning at " + root + " and have data at " + rowPath + " that isn't marked for pruning or keeping. Ignoring.");
        }
        int prunedCount = 0;
        int resavedCount = 0;
        if (!rowIdsToPrune.isEmpty()) {
            ArrayList<Pair<Path, Node>> rowsToResave = new ArrayList<Pair<Path, Node>>();
            this.pruneTreeRecursive(root, Path.getEmptyPath(), rowIdsToPrune, rowIdsToKeep, pruneForest, rowsToResave);
            Collection<Object> rowIdsToDelete = rowIdsToPrune.values();
            String whereClause = "rowid IN (" + this.commaSeparatedList(rowIdsToDelete) + ")";
            this.database.delete(SERVER_CACHE_TABLE, whereClause, null);
            for (Pair pair : rowsToResave) {
                this.saveNested(root.child((Path)pair.getFirst()), (Node)pair.getSecond());
            }
            prunedCount = rowIdsToDelete.size();
            resavedCount = rowsToResave.size();
        }
        long duration = System.currentTimeMillis() - start;
        if (this.logger.logsDebug()) {
            this.logger.debug(String.format(Locale.US, "Pruned %d rows with %d nodes resaved in %dms", prunedCount, resavedCount, duration), new Object[0]);
        }
    }

    private void pruneTreeRecursive(Path pruneRoot, final Path relativePath, ImmutableTree<Long> rowIdsToPrune, final ImmutableTree<Long> rowIdsToKeep, PruneForest pruneForest, final List<Pair<Path, Node>> rowsToResaveAccumulator) {
        if (rowIdsToPrune.getValue() != null) {
            int nodesToResave = pruneForest.foldKeptNodes(0, new ImmutableTree.TreeVisitor<Void, Integer>(){

                @Override
                public Integer onNodeValue(Path keepPath, Void ignore, Integer nodesToResave) {
                    return rowIdsToKeep.get(keepPath) == null ? nodesToResave + 1 : nodesToResave;
                }
            });
            if (nodesToResave > 0) {
                Path absolutePath = pruneRoot.child(relativePath);
                if (this.logger.logsDebug()) {
                    this.logger.debug(String.format(Locale.US, "Need to rewrite %d nodes below path %s", nodesToResave, absolutePath), new Object[0]);
                }
                final Node currentNode = this.loadNested(absolutePath);
                pruneForest.foldKeptNodes(null, new ImmutableTree.TreeVisitor<Void, Void>(){

                    @Override
                    public Void onNodeValue(Path keepPath, Void ignore, Void ignore2) {
                        if (rowIdsToKeep.get(keepPath) == null) {
                            rowsToResaveAccumulator.add(new Pair<Path, Node>(relativePath.child(keepPath), currentNode.getChild(keepPath)));
                        }
                        return null;
                    }
                });
            }
        } else {
            for (Map.Entry entry : rowIdsToPrune.getChildren()) {
                ChildKey childKey = (ChildKey)entry.getKey();
                PruneForest childPruneForest = pruneForest.child((ChildKey)entry.getKey());
                this.pruneTreeRecursive(pruneRoot, relativePath.child(childKey), (ImmutableTree)entry.getValue(), rowIdsToKeep.getChild(childKey), childPruneForest, rowsToResaveAccumulator);
            }
        }
    }

    @Override
    public void removeAllUserWrites() {
        this.verifyInsideTransaction();
        long start = System.currentTimeMillis();
        int count = this.database.delete(WRITES_TABLE, null, null);
        long duration = System.currentTimeMillis() - start;
        if (this.logger.logsDebug()) {
            this.logger.debug(String.format(Locale.US, "Deleted %d (all) write(s) in %dms", count, duration), new Object[0]);
        }
    }

    public void purgeCache() {
        this.verifyInsideTransaction();
        this.database.delete(SERVER_CACHE_TABLE, null, null);
        this.database.delete(WRITES_TABLE, null, null);
        this.database.delete(TRACKED_QUERY_TABLE, null, null);
        this.database.delete(TRACKED_KEYS_TABLE, null, null);
    }

    @Override
    public void beginTransaction() {
        Utilities.hardAssert(!this.insideTransaction, "runInTransaction called when an existing transaction is already in progress.");
        if (this.logger.logsDebug()) {
            this.logger.debug("Starting transaction.", new Object[0]);
        }
        this.database.beginTransaction();
        this.insideTransaction = true;
        this.transactionStart = System.currentTimeMillis();
    }

    @Override
    public void endTransaction() {
        this.database.endTransaction();
        this.insideTransaction = false;
        long elapsed = System.currentTimeMillis() - this.transactionStart;
        if (this.logger.logsDebug()) {
            this.logger.debug(String.format(Locale.US, "Transaction completed. Elapsed: %dms", elapsed), new Object[0]);
        }
    }

    @Override
    public void setTransactionSuccessful() {
        this.database.setTransactionSuccessful();
    }

    @Override
    public void close() {
        this.database.close();
    }

    private SQLiteDatabase openDatabase(Context context, String cacheId) {
        PersistentCacheOpenHelper helper = new PersistentCacheOpenHelper(context, cacheId);
        try {
            SQLiteDatabase database = helper.getWritableDatabase();
            database.rawQuery("PRAGMA locking_mode = EXCLUSIVE", null).close();
            database.beginTransaction();
            database.endTransaction();
            return database;
        }
        catch (SQLiteException e) {
            if (e instanceof SQLiteDatabaseLockedException) {
                String msg = "Failed to gain exclusive lock to Firebase Database's offline persistence. This generally means you are using Firebase Database from multiple processes in your app. Keep in mind that multi-process Android apps execute the code in your Application class in all processes, so you may need to avoid initializing FirebaseDatabase in your Application class. If you are intentionally using Firebase Database from multiple processes, you can only enable offline persistence (i.e. call setPersistenceEnabled(true)) in one of them.";
                throw new DatabaseException(msg, e);
            }
            throw e;
        }
    }

    private void verifyInsideTransaction() {
        Utilities.hardAssert(this.insideTransaction, "Transaction expected to already be in progress.");
    }

    private int saveNested(Path path, Node node) {
        long estimatedSize = NodeSizeEstimator.estimateSerializedNodeSize(node);
        if (node instanceof ChildrenNode && estimatedSize > 16384L) {
            if (this.logger.logsDebug()) {
                this.logger.debug(String.format(Locale.US, "Node estimated serialized size at path %s of %d bytes exceeds limit of %d bytes. Splitting up.", path, estimatedSize, 16384), new Object[0]);
            }
            int sum = 0;
            for (NamedNode child : node) {
                sum += this.saveNested(path.child(child.getName()), child.getNode());
            }
            if (!node.getPriority().isEmpty()) {
                this.saveNode(path.child(ChildKey.getPriorityKey()), node.getPriority());
                ++sum;
            }
            this.saveNode(path, EmptyNode.Empty());
            return ++sum;
        }
        this.saveNode(path, node);
        return 1;
    }

    private String partKey(Path path, int i) {
        return SqlPersistenceStorageEngine.pathToKey(path) + String.format(Locale.US, PART_KEY_FORMAT, i);
    }

    private void saveNode(Path path, Node node) {
        byte[] serialized = this.serializeObject(node.getValue(true));
        if (serialized.length >= 262144) {
            List<byte[]> parts = SqlPersistenceStorageEngine.splitBytes(serialized, 262144);
            if (this.logger.logsDebug()) {
                this.logger.debug("Saving huge leaf node with " + parts.size() + " parts.", new Object[0]);
            }
            for (int i = 0; i < parts.size(); ++i) {
                ContentValues values = new ContentValues();
                values.put("path", this.partKey(path, i));
                values.put(VALUE_COLUMN_NAME, parts.get(i));
                this.database.insertWithOnConflict(SERVER_CACHE_TABLE, null, values, 5);
            }
        } else {
            ContentValues values = new ContentValues();
            values.put("path", SqlPersistenceStorageEngine.pathToKey(path));
            values.put(VALUE_COLUMN_NAME, serialized);
            this.database.insertWithOnConflict(SERVER_CACHE_TABLE, null, values, 5);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Node loadNested(Path path) {
        ArrayList<String> pathStrings = new ArrayList<String>();
        ArrayList<byte[]> payloads = new ArrayList<byte[]>();
        long queryStart = System.currentTimeMillis();
        Cursor cursor = this.loadNestedQuery(path, new String[]{"path", VALUE_COLUMN_NAME});
        long queryDuration = System.currentTimeMillis() - queryStart;
        long loadingStart = System.currentTimeMillis();
        try {
            while (cursor.moveToNext()) {
                pathStrings.add(cursor.getString(0));
                payloads.add(cursor.getBlob(1));
            }
        }
        finally {
            cursor.close();
        }
        long loadingDuration = System.currentTimeMillis() - loadingStart;
        long serializingStart = System.currentTimeMillis();
        Node node = EmptyNode.Empty();
        boolean sawDescendant = false;
        HashMap<Path, Node> priorities = new HashMap<Path, Node>();
        for (int i = 0; i < payloads.size(); ++i) {
            Node savedNode;
            Path savedPath;
            if (((String)pathStrings.get(i)).endsWith(FIRST_PART_KEY)) {
                String pathString = (String)pathStrings.get(i);
                savedPath = new Path(pathString.substring(0, pathString.length() - FIRST_PART_KEY.length()));
                int splitNodeRunLength = this.splitNodeRunLength(savedPath, pathStrings, i);
                if (this.logger.logsDebug()) {
                    this.logger.debug("Loading split node with " + splitNodeRunLength + " parts.", new Object[0]);
                }
                savedNode = this.deserializeNode(this.joinBytes(payloads.subList(i, i + splitNodeRunLength)));
                i = i + splitNodeRunLength - 1;
            } else {
                savedNode = this.deserializeNode((byte[])payloads.get(i));
                savedPath = new Path((String)pathStrings.get(i));
            }
            if (savedPath.getBack() != null && savedPath.getBack().isPriorityChildName()) {
                priorities.put(savedPath, savedNode);
                continue;
            }
            if (savedPath.contains(path)) {
                Utilities.hardAssert(!sawDescendant, "Descendants of path must come after ancestors.");
                node = savedNode.getChild(Path.getRelative(savedPath, path));
                continue;
            }
            if (path.contains(savedPath)) {
                sawDescendant = true;
                Path childPath = Path.getRelative(path, savedPath);
                node = node.updateChild(childPath, savedNode);
                continue;
            }
            throw new IllegalStateException(String.format("Loading an unrelated row with path %s for %s", savedPath, path));
        }
        for (Map.Entry entry : priorities.entrySet()) {
            Path priorityPath = (Path)entry.getKey();
            node = node.updateChild(Path.getRelative(path, priorityPath), (Node)entry.getValue());
        }
        long serializeDuration = System.currentTimeMillis() - serializingStart;
        long duration = System.currentTimeMillis() - queryStart;
        if (this.logger.logsDebug()) {
            this.logger.debug(String.format(Locale.US, "Loaded a total of %d rows for a total of %d nodes at %s in %dms (Query: %dms, Loading: %dms, Serializing: %dms)", payloads.size(), NodeSizeEstimator.nodeCount(node), path, duration, queryDuration, loadingDuration, serializeDuration), new Object[0]);
        }
        return node;
    }

    private int splitNodeRunLength(Path path, List<String> pathStrings, int startPosition) {
        int endPosition;
        String pathPrefix = SqlPersistenceStorageEngine.pathToKey(path);
        if (!pathStrings.get(startPosition).startsWith(pathPrefix)) {
            throw new IllegalStateException("Extracting split nodes needs to start with path prefix");
        }
        for (endPosition = startPosition + 1; endPosition < pathStrings.size() && pathStrings.get(endPosition).equals(this.partKey(path, endPosition - startPosition)); ++endPosition) {
        }
        if (endPosition < pathStrings.size() && pathStrings.get(endPosition).startsWith(pathPrefix + PART_KEY_PREFIX)) {
            throw new IllegalStateException("Run did not finish with all parts");
        }
        return endPosition - startPosition;
    }

    private Cursor loadNestedQuery(Path path, String[] columns) {
        String pathPrefixStart = SqlPersistenceStorageEngine.pathToKey(path);
        String pathPrefixEnd = SqlPersistenceStorageEngine.pathPrefixStartToPrefixEnd(pathPrefixStart);
        String[] arguments = new String[path.size() + 3];
        String whereClause = SqlPersistenceStorageEngine.buildAncestorWhereClause(path, arguments);
        whereClause = whereClause + " OR (path > ? AND path < ?)";
        arguments[path.size() + 1] = pathPrefixStart;
        arguments[path.size() + 2] = pathPrefixEnd;
        String orderBy = "path";
        return this.database.query(SERVER_CACHE_TABLE, columns, whereClause, arguments, null, null, orderBy);
    }

    private static String pathToKey(Path path) {
        if (path.isEmpty()) {
            return "/";
        }
        return path.toString() + "/";
    }

    private static String pathPrefixStartToPrefixEnd(String prefix) {
        Utilities.hardAssert(prefix.endsWith("/"), "Path keys must end with a '/'");
        return prefix.substring(0, prefix.length() - 1) + '0';
    }

    private static String buildAncestorWhereClause(Path path, String[] arguments) {
        Utilities.hardAssert(arguments.length >= path.size() + 1);
        int count = 0;
        StringBuilder whereClause = new StringBuilder("(");
        while (!path.isEmpty()) {
            whereClause.append("path");
            whereClause.append(" = ? OR ");
            arguments[count] = SqlPersistenceStorageEngine.pathToKey(path);
            path = path.getParent();
            ++count;
        }
        whereClause.append("path");
        whereClause.append(" = ?)");
        arguments[count] = SqlPersistenceStorageEngine.pathToKey(Path.getEmptyPath());
        return whereClause.toString();
    }

    private int removeNested(String table, Path path) {
        String pathPrefixQuery = "path >= ? AND path < ?";
        String pathPrefixStart = SqlPersistenceStorageEngine.pathToKey(path);
        String pathPrefixEnd = SqlPersistenceStorageEngine.pathPrefixStartToPrefixEnd(pathPrefixStart);
        return this.database.delete(table, pathPrefixQuery, new String[]{pathPrefixStart, pathPrefixEnd});
    }

    private static List<byte[]> splitBytes(byte[] bytes, int size) {
        int parts = (bytes.length - 1) / size + 1;
        ArrayList<byte[]> partList = new ArrayList<byte[]>(parts);
        for (int i = 0; i < parts; ++i) {
            int length = Math.min(size, bytes.length - i * size);
            byte[] part = new byte[length];
            System.arraycopy(bytes, i * size, part, 0, length);
            partList.add(part);
        }
        return partList;
    }

    private byte[] joinBytes(List<byte[]> payloads) {
        int totalSize = 0;
        for (byte[] payload : payloads) {
            totalSize += payload.length;
        }
        byte[] buffer = new byte[totalSize];
        int currentBytePosition = 0;
        for (byte[] payload : payloads) {
            System.arraycopy(payload, 0, buffer, currentBytePosition, payload.length);
            currentBytePosition += payload.length;
        }
        return buffer;
    }

    private byte[] serializeObject(Object object) {
        try {
            return JsonMapper.serializeJsonValue(object).getBytes(UTF8_CHARSET);
        }
        catch (IOException e) {
            throw new RuntimeException("Could not serialize leaf node", e);
        }
    }

    private Node deserializeNode(byte[] value) {
        try {
            Object o = JsonMapper.parseJsonValue(new String(value, UTF8_CHARSET));
            return NodeUtilities.NodeFromJSON(o);
        }
        catch (IOException e) {
            String stringValue = new String(value, UTF8_CHARSET);
            throw new RuntimeException("Could not deserialize node: " + stringValue, e);
        }
    }

    private String commaSeparatedList(Collection<Long> items) {
        StringBuilder list = new StringBuilder();
        boolean first = true;
        for (long item : items) {
            if (!first) {
                list.append(",");
            }
            first = false;
            list.append(item);
        }
        return list.toString();
    }

    private static class PersistentCacheOpenHelper
    extends SQLiteOpenHelper {
        private static final int DATABASE_VERSION = 2;

        public PersistentCacheOpenHelper(Context context, String cacheId) {
            super(context, cacheId, null, 2);
        }

        public void onCreate(SQLiteDatabase db) {
            db.execSQL(SqlPersistenceStorageEngine.CREATE_SERVER_CACHE);
            db.execSQL(SqlPersistenceStorageEngine.CREATE_WRITES);
            db.execSQL(SqlPersistenceStorageEngine.CREATE_TRACKED_QUERIES);
            db.execSQL(SqlPersistenceStorageEngine.CREATE_TRACKED_KEYS);
        }

        public void onUpgrade(SQLiteDatabase db, int oldVersion, int newVersion) {
            Utilities.hardAssert(newVersion == 2, "Why is onUpgrade() called with a different version?");
            if (oldVersion > 1) {
                throw new AssertionError((Object)("We don't handle upgrading to " + newVersion));
            }
            this.dropTable(db, SqlPersistenceStorageEngine.SERVER_CACHE_TABLE);
            db.execSQL(SqlPersistenceStorageEngine.CREATE_SERVER_CACHE);
            this.dropTable(db, SqlPersistenceStorageEngine.TRACKED_QUERY_COMPLETE_COLUMN_NAME);
            db.execSQL(SqlPersistenceStorageEngine.CREATE_TRACKED_KEYS);
            db.execSQL(SqlPersistenceStorageEngine.CREATE_TRACKED_QUERIES);
        }

        private void dropTable(SQLiteDatabase db, String table) {
            db.execSQL("DROP TABLE IF EXISTS " + table);
        }
    }
}

