/*
 * Decompiled with CFR 0.152.
 */
package com.google.firebase.perf.transport;

import android.content.Context;
import androidx.annotation.NonNull;
import com.google.android.gms.common.util.VisibleForTesting;
import com.google.firebase.perf.config.ConfigResolver;
import com.google.firebase.perf.logging.AndroidLogger;
import com.google.firebase.perf.metrics.resource.ResourceType;
import com.google.firebase.perf.util.Clock;
import com.google.firebase.perf.util.Constants;
import com.google.firebase.perf.util.Rate;
import com.google.firebase.perf.util.Timer;
import com.google.firebase.perf.util.Utils;
import com.google.firebase.perf.v1.PerfMetric;
import com.google.firebase.perf.v1.PerfSession;
import com.google.firebase.perf.v1.SessionVerbosity;
import java.util.List;
import java.util.Random;
import java.util.concurrent.TimeUnit;

final class RateLimiter {
    private final ConfigResolver configResolver;
    private final float samplingBucketId;
    private RateLimiterImpl traceLimiter = null;
    private RateLimiterImpl networkLimiter = null;
    private boolean isLogcatEnabled = false;

    public RateLimiter(@NonNull Context appContext, Rate rate, long capacity) {
        this(rate, capacity, new Clock(), RateLimiter.getSamplingBucketId(), ConfigResolver.getInstance());
        this.isLogcatEnabled = Utils.isDebugLoggingEnabled(appContext);
    }

    @VisibleForTesting
    static float getSamplingBucketId() {
        return new Random().nextFloat();
    }

    RateLimiter(Rate rate, long capacity, Clock clock, float samplingBucketId, ConfigResolver configResolver) {
        Utils.checkArgument(0.0f <= samplingBucketId && samplingBucketId < 1.0f, "Sampling bucket ID should be in range [0.0f, 1.0f).");
        this.samplingBucketId = samplingBucketId;
        this.configResolver = configResolver;
        this.traceLimiter = new RateLimiterImpl(rate, capacity, clock, configResolver, "Trace", this.isLogcatEnabled);
        this.networkLimiter = new RateLimiterImpl(rate, capacity, clock, configResolver, "Network", this.isLogcatEnabled);
    }

    private boolean isDeviceAllowedToSendTraces() {
        float validTraceSamplingBucketIdThreshold = this.configResolver.getTraceSamplingRate();
        return this.samplingBucketId < validTraceSamplingBucketIdThreshold;
    }

    private boolean isDeviceAllowedToSendNetworkEvents() {
        float validNetworkSamplingBucketIdThreshold = this.configResolver.getNetworkRequestSamplingRate();
        return this.samplingBucketId < validNetworkSamplingBucketIdThreshold;
    }

    boolean check(PerfMetric metric) {
        if (metric.hasTraceMetric() && !this.isDeviceAllowedToSendTraces() && !this.hasVerboseSessions(metric.getTraceMetric().getPerfSessionsList())) {
            return false;
        }
        if (metric.hasNetworkRequestMetric() && !this.isDeviceAllowedToSendNetworkEvents() && !this.hasVerboseSessions(metric.getNetworkRequestMetric().getPerfSessionsList())) {
            return false;
        }
        if (!this.isRateLimited(metric)) {
            return true;
        }
        if (metric.hasNetworkRequestMetric()) {
            return this.networkLimiter.check(metric);
        }
        if (metric.hasTraceMetric()) {
            return this.traceLimiter.check(metric);
        }
        return false;
    }

    private boolean hasVerboseSessions(List<PerfSession> perfSessions) {
        if (perfSessions.size() > 0 && perfSessions.get(0).getSessionVerbosityCount() > 0) {
            return perfSessions.get(0).getSessionVerbosity(0) == SessionVerbosity.GAUGES_AND_SYSTEM_EVENTS;
        }
        return false;
    }

    boolean isRateLimited(@NonNull PerfMetric metric) {
        if (metric.hasTraceMetric() && (metric.getTraceMetric().getName().equals(Constants.TraceNames.FOREGROUND_TRACE_NAME.toString()) || metric.getTraceMetric().getName().equals(Constants.TraceNames.BACKGROUND_TRACE_NAME.toString())) && metric.getTraceMetric().getCountersCount() > 0) {
            return false;
        }
        return !metric.hasGaugeMetric();
    }

    void changeRate(boolean isForeground) {
        this.traceLimiter.changeRate(isForeground);
        this.networkLimiter.changeRate(isForeground);
    }

    @VisibleForTesting
    boolean getIsDeviceAllowedToSendTraces() {
        return this.isDeviceAllowedToSendTraces();
    }

    @VisibleForTesting
    boolean getIsDeviceAllowedToSendNetworkEvents() {
        return this.isDeviceAllowedToSendNetworkEvents();
    }

    static class RateLimiterImpl {
        private static final AndroidLogger logger = AndroidLogger.getInstance();
        private static final long MICROS_IN_A_SECOND = TimeUnit.SECONDS.toMicros(1L);
        private final Clock clock;
        private final boolean isLogcatEnabled;
        private Timer lastTimeTokenReplenished;
        private Rate rate;
        private long capacity;
        private long tokenCount;
        private Rate foregroundRate;
        private Rate backgroundRate;
        private long foregroundCapacity;
        private long backgroundCapacity;

        RateLimiterImpl(Rate rate, long capacity, Clock clock, ConfigResolver configResolver, @ResourceType String type, boolean isLogcatEnabled) {
            this.clock = clock;
            this.capacity = capacity;
            this.rate = rate;
            this.tokenCount = capacity;
            this.lastTimeTokenReplenished = this.clock.getTime();
            this.setRateByReadingRemoteConfigValues(configResolver, type, isLogcatEnabled);
            this.isLogcatEnabled = isLogcatEnabled;
        }

        synchronized boolean check(@NonNull PerfMetric metric) {
            Timer now = this.clock.getTime();
            long newTokens = Math.max(0L, (long)((double)this.lastTimeTokenReplenished.getDurationMicros(now) * this.rate.getTokensPerSeconds() / (double)MICROS_IN_A_SECOND));
            this.tokenCount = Math.min(this.tokenCount + newTokens, this.capacity);
            if (newTokens > 0L) {
                this.lastTimeTokenReplenished = new Timer(this.lastTimeTokenReplenished.getMicros() + (long)((double)(newTokens * MICROS_IN_A_SECOND) / this.rate.getTokensPerSeconds()));
            }
            if (this.tokenCount > 0L) {
                --this.tokenCount;
                return true;
            }
            if (this.isLogcatEnabled) {
                logger.warn("Exceeded log rate limit, dropping the log.");
            }
            return false;
        }

        synchronized void changeRate(boolean isForeground) {
            this.rate = isForeground ? this.foregroundRate : this.backgroundRate;
            this.capacity = isForeground ? this.foregroundCapacity : this.backgroundCapacity;
        }

        private void setRateByReadingRemoteConfigValues(ConfigResolver configResolver, @ResourceType String type, boolean isLogcatEnabled) {
            long fLimitTime = RateLimiterImpl.getFlimitSec(configResolver, type);
            long fLimitEvents = RateLimiterImpl.getFlimitEvents(configResolver, type);
            this.foregroundRate = new Rate(fLimitEvents, fLimitTime, TimeUnit.SECONDS);
            this.foregroundCapacity = fLimitEvents;
            if (isLogcatEnabled) {
                logger.debug("Foreground %s logging rate:%f, burst capacity:%d", type, this.foregroundRate, this.foregroundCapacity);
            }
            long bLimitTime = RateLimiterImpl.getBlimitSec(configResolver, type);
            long bLimitEvents = RateLimiterImpl.getBlimitEvents(configResolver, type);
            this.backgroundRate = new Rate(bLimitEvents, bLimitTime, TimeUnit.SECONDS);
            this.backgroundCapacity = bLimitEvents;
            if (isLogcatEnabled) {
                logger.debug("Background %s logging rate:%f, capacity:%d", type, this.backgroundRate, this.backgroundCapacity);
            }
        }

        private static long getFlimitSec(ConfigResolver configResolver, @ResourceType String type) {
            if (type == "Trace") {
                return configResolver.getRateLimitSec();
            }
            return configResolver.getRateLimitSec();
        }

        private static long getFlimitEvents(ConfigResolver configResolver, @ResourceType String type) {
            if (type == "Trace") {
                return configResolver.getTraceEventCountForeground();
            }
            return configResolver.getNetworkEventCountForeground();
        }

        private static long getBlimitSec(ConfigResolver configResolver, @ResourceType String type) {
            if (type == "Trace") {
                return configResolver.getRateLimitSec();
            }
            return configResolver.getRateLimitSec();
        }

        private static long getBlimitEvents(ConfigResolver configResolver, @ResourceType String type) {
            if (type == "Trace") {
                return configResolver.getTraceEventCountBackground();
            }
            return configResolver.getNetworkEventCountBackground();
        }

        @VisibleForTesting
        Rate getForegroundRate() {
            return this.foregroundRate;
        }

        @VisibleForTesting
        long getForegroundCapacity() {
            return this.foregroundCapacity;
        }

        @VisibleForTesting
        Rate getBackgroundRate() {
            return this.backgroundRate;
        }

        @VisibleForTesting
        long getBackgroundCapacity() {
            return this.backgroundCapacity;
        }

        @VisibleForTesting
        Rate getRate() {
            return this.rate;
        }

        @VisibleForTesting
        void setRate(Rate newRate) {
            this.rate = newRate;
        }
    }
}

