/**
 * SPDX-FileCopyrightText: (c) 2024 Liferay, Inc. https://liferay.com
 * SPDX-License-Identifier: LGPL-2.1-or-later OR LicenseRef-Liferay-DXP-EULA-2.0.0-2023-06
 */

package com.liferay.portal.kernel.upgrade.recorder;

import com.liferay.petra.lang.HashUtil;
import com.liferay.petra.string.StringBundler;
import com.liferay.petra.string.StringPool;
import com.liferay.portal.kernel.dao.jdbc.util.CallableStatementWrapper;
import com.liferay.portal.kernel.dao.jdbc.util.ConnectionWrapper;
import com.liferay.portal.kernel.dao.jdbc.util.PreparedStatementWrapper;
import com.liferay.portal.kernel.dao.jdbc.util.StatementWrapper;
import com.liferay.portal.kernel.db.partition.DBPartition;
import com.liferay.portal.kernel.security.auth.CompanyThreadLocal;
import com.liferay.portal.kernel.util.GetterUtil;
import com.liferay.portal.kernel.util.PropsKeys;
import com.liferay.portal.kernel.util.PropsUtil;
import com.liferay.portal.kernel.util.StringUtil;
import com.liferay.portal.kernel.util.Validator;

import java.sql.CallableStatement;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;

import java.util.List;
import java.util.Set;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.CopyOnWriteArraySet;

/**
 * @author István András Dézsi
 */
public class UpgradeSQLRecorder {

	public static Connection getConnectionWrapper(Connection connection) {
		return getConnectionWrapper(connection, StringPool.BLANK);
	}

	public static Connection getConnectionWrapper(
		Connection connection, String upgradeProcessClassName) {

		_upgradeProcessClassName = upgradeProcessClassName;

		if (!_enabled) {
			return connection;
		}

		return new ConnectionWrapper(connection) {

			@Override
			public Statement createStatement() throws SQLException {
				return _wrap(super.createStatement());
			}

			@Override
			public Statement createStatement(
					int resultSetType, int resultSetConcurrency)
				throws SQLException {

				return _wrap(
					super.createStatement(resultSetType, resultSetConcurrency));
			}

			@Override
			public Statement createStatement(
					int resultSetType, int resultSetConcurrency,
					int resultSetHoldability)
				throws SQLException {

				return _wrap(
					super.createStatement(
						resultSetType, resultSetConcurrency,
						resultSetHoldability));
			}

			@Override
			public CallableStatement prepareCall(String sql)
				throws SQLException {

				return (CallableStatement)_wrap(super.prepareCall(sql));
			}

			@Override
			public CallableStatement prepareCall(
					String sql, int resultSetType, int resultSetConcurrency)
				throws SQLException {

				return (CallableStatement)_wrap(
					super.prepareCall(
						sql, resultSetType, resultSetConcurrency));
			}

			@Override
			public CallableStatement prepareCall(
					String sql, int resultSetType, int resultSetConcurrency,
					int resultSetHoldability)
				throws SQLException {

				return (CallableStatement)_wrap(
					super.prepareCall(
						sql, resultSetType, resultSetConcurrency,
						resultSetHoldability));
			}

			@Override
			public PreparedStatement prepareStatement(String sql)
				throws SQLException {

				return (PreparedStatement)_wrap(super.prepareStatement(sql));
			}

			@Override
			public PreparedStatement prepareStatement(
					String sql, int autoGeneratedKeys)
				throws SQLException {

				return (PreparedStatement)_wrap(
					super.prepareStatement(sql, autoGeneratedKeys));
			}

			@Override
			public PreparedStatement prepareStatement(
					String sql, int resultSetType, int resultSetConcurrency)
				throws SQLException {

				return (PreparedStatement)_wrap(
					super.prepareStatement(
						sql, resultSetType, resultSetConcurrency));
			}

			@Override
			public PreparedStatement prepareStatement(
					String sql, int resultSetType, int resultSetConcurrency,
					int resultSetHoldability)
				throws SQLException {

				return (PreparedStatement)_wrap(
					super.prepareStatement(
						sql, resultSetType, resultSetConcurrency,
						resultSetHoldability));
			}

			@Override
			public PreparedStatement prepareStatement(
					String sql, int[] columnIndexes)
				throws SQLException {

				return (PreparedStatement)_wrap(
					super.prepareStatement(sql, columnIndexes));
			}

			@Override
			public PreparedStatement prepareStatement(
					String sql, String[] columnNames)
				throws SQLException {

				return (PreparedStatement)_wrap(
					super.prepareStatement(sql, columnNames));
			}

		};
	}

	public static List<FailedSQL> getFailedSQLs() {
		return _failedSQLs;
	}

	public static Set<RunningSQL> getRunningSQLs() {
		return _runningSQLs;
	}

	public static void start() {
		_enabled = true;

		_failedSQLs.clear();

		_runningSQLs.clear();
	}

	public static void stop() {
		_enabled = false;
	}

	public static class FailedSQL {

		public FailedSQL(String sql) {
			this(StringPool.BLANK, sql);
		}

		public FailedSQL(String message, String sql) {
			_message = message;
			_sql = sql;
		}

		public String getMessage() {
			return _message;
		}

		public String getSQL() {
			return _sql;
		}

		@Override
		public String toString() {
			return StringBundler.concat(
				"SQL: ", _sql, "\nError: ", _message, "\n");
		}

		private final String _message;
		private final String _sql;

	}

	public static class RunningSQL {

		public RunningSQL(long duration, String sql) {
			this(duration, sql, StringPool.BLANK);
		}

		public RunningSQL(
			long duration, String sql, String upgradeProcessClassName) {

			_duration = duration;
			_sql = sql;
			_upgradeProcessClassName = upgradeProcessClassName;
		}

		@Override
		public boolean equals(Object object) {
			if (this == object) {
				return true;
			}

			if (!(object instanceof RunningSQL)) {
				return false;
			}

			RunningSQL runningSQL = (RunningSQL)object;

			if (Validator.isBlank(_upgradeProcessClassName)) {
				return _sql.equals(runningSQL._sql);
			}

			if (_sql.equals(runningSQL._sql) &&
				_upgradeProcessClassName.equals(
					runningSQL._upgradeProcessClassName)) {

				return true;
			}

			return false;
		}

		public long getDuration() {
			return _duration;
		}

		public String getSQL() {
			return _sql;
		}

		public String getUpgradeProcessClassName() {
			return _upgradeProcessClassName;
		}

		@Override
		public int hashCode() {
			if (Validator.isBlank(_upgradeProcessClassName)) {
				return _sql.hashCode();
			}

			int hashCode = HashUtil.hash(0, _sql);

			return HashUtil.hash(hashCode, _upgradeProcessClassName);
		}

		@Override
		public String toString() {
			return StringBundler.concat(
				"Upgrade Process: ", _upgradeProcessClassName, "\nSQL: ", _sql,
				"\nDuration: ", _duration, " ms\n");
		}

		private final long _duration;
		private final String _sql;
		private final String _upgradeProcessClassName;

	}

	private static <T> T _execute(SQLCallable<T> sqlCallable, Object object)
		throws SQLException {

		long startTime = System.currentTimeMillis();

		try {
			return sqlCallable.call();
		}
		catch (SQLException sqlException) {
			String sql = _extractSQL(object);

			if (sql != null) {
				sql += StringPool.SEMICOLON;

				String message = sqlException.getMessage();

				if (Validator.isBlank(message)) {
					_failedSQLs.add(new FailedSQL(sql));
				}
				else {
					_failedSQLs.add(new FailedSQL(message, sql));
				}
			}

			throw sqlException;
		}
		finally {
			_executeFinally(object, startTime);
		}
	}

	private static void _executeFinally(Object object, long startTime) {
		String sql = _extractSQL(object);

		if (sql == null) {
			return;
		}

		sql += StringPool.SEMICOLON;

		long duration = System.currentTimeMillis() - startTime;

		if (duration < _UPGRADE_REPORT_SQL_STATEMENT_THRESHOLD) {
			return;
		}

		if (Validator.isBlank(_upgradeProcessClassName)) {
			_runningSQLs.add(new RunningSQL(duration, sql));
		}
		else if (DBPartition.isPartitionEnabled()) {
			_runningSQLs.add(
				new RunningSQL(
					duration, sql,
					_upgradeProcessClassName + StringPool.AT +
						String.valueOf(CompanyThreadLocal.getCompanyId())));
		}
		else {
			_runningSQLs.add(
				new RunningSQL(duration, sql, _upgradeProcessClassName));
		}
	}

	private static String _extractSQL(Object object) {
		if (object instanceof String) {
			return (String)object;
		}
		else if (object instanceof Statement) {
			String string = object.toString();

			int index = string.indexOf(':');

			if (index >= 0) {
				return StringUtil.trim(string.substring(index + 1));
			}
		}

		return null;
	}

	private static <T extends Statement> T _wrap(T statement) {
		if (statement instanceof CallableStatement) {
			return (T)new CallableStatementWrapper(
				(CallableStatement)statement) {

				@Override
				public boolean execute() throws SQLException {
					return _execute(() -> super.execute(), statement);
				}

				@Override
				public int[] executeBatch() throws SQLException {
					return _execute(() -> super.executeBatch(), statement);
				}

				@Override
				public ResultSet executeQuery() throws SQLException {
					return _execute(() -> super.executeQuery(), statement);
				}

				@Override
				public int executeUpdate() throws SQLException {
					return _execute(() -> super.executeUpdate(), statement);
				}

			};
		}
		else if (statement instanceof PreparedStatement) {
			return (T)new PreparedStatementWrapper(
				(PreparedStatement)statement) {

				@Override
				public boolean execute() throws SQLException {
					return _execute(() -> super.execute(), statement);
				}

				@Override
				public int[] executeBatch() throws SQLException {
					return _execute(() -> super.executeBatch(), statement);
				}

				@Override
				public ResultSet executeQuery() throws SQLException {
					return _execute(() -> super.executeQuery(), statement);
				}

				@Override
				public int executeUpdate() throws SQLException {
					return _execute(() -> super.executeUpdate(), statement);
				}

			};
		}

		return (T)new StatementWrapper(statement) {

			@Override
			public boolean execute(String sql) throws SQLException {
				return _execute(() -> super.execute(sql), sql);
			}

			@Override
			public int[] executeBatch() throws SQLException {
				return _execute(() -> super.executeBatch(), statement);
			}

			@Override
			public ResultSet executeQuery(String sql) throws SQLException {
				return _execute(() -> super.executeQuery(sql), sql);
			}

			@Override
			public int executeUpdate(String sql) throws SQLException {
				return _execute(() -> super.executeUpdate(sql), sql);
			}

			@Override
			public boolean execute(String sql, int autoGeneratedKeys)
				throws SQLException {

				return _execute(
					() -> super.execute(sql, autoGeneratedKeys), sql);
			}

			@Override
			public boolean execute(String sql, int[] columnIndexes)
				throws SQLException {

				return _execute(() -> super.execute(sql, columnIndexes), sql);
			}

			@Override
			public boolean execute(String sql, String[] columnNames)
				throws SQLException {

				return _execute(() -> super.execute(sql, columnNames), sql);
			}

			@Override
			public int executeUpdate(String sql, int autoGeneratedKeys)
				throws SQLException {

				return _execute(
					() -> super.executeUpdate(sql, autoGeneratedKeys), sql);
			}

			@Override
			public int executeUpdate(String sql, int[] columnIndexes)
				throws SQLException {

				return _execute(
					() -> super.executeUpdate(sql, columnIndexes), sql);
			}

			@Override
			public int executeUpdate(String sql, String[] columnNames)
				throws SQLException {

				return _execute(
					() -> super.executeUpdate(sql, columnNames), sql);
			}

		};
	}

	private static final long _UPGRADE_REPORT_SQL_STATEMENT_THRESHOLD =
		GetterUtil.getLong(
			PropsUtil.get(PropsKeys.UPGRADE_REPORT_SQL_STATEMENT_THRESHOLD));

	private static boolean _enabled;
	private static final List<FailedSQL> _failedSQLs =
		new CopyOnWriteArrayList<>();
	private static final Set<RunningSQL> _runningSQLs =
		new CopyOnWriteArraySet<>();
	private static volatile String _upgradeProcessClassName = StringPool.BLANK;

	@FunctionalInterface
	private interface SQLCallable<R> {

		public R call() throws SQLException;

	}

}