/*
 * Copyright (C) 2016 - present by OpenGamma Inc. and the OpenGamma group of companies
 *
 * Please see distribution for license.
 */
package com.opengamma.strata.calc.runner;

import java.lang.invoke.MethodHandles;
import java.util.Optional;

import org.joda.beans.ImmutableBean;
import org.joda.beans.JodaBeanUtils;
import org.joda.beans.MetaBean;
import org.joda.beans.TypedMetaBean;
import org.joda.beans.gen.BeanDefinition;
import org.joda.beans.gen.PropertyDefinition;
import org.joda.beans.impl.light.LightMetaBean;

import com.opengamma.strata.basics.CalculationTarget;

/**
 * A set of calculation functions which combines the functions in two other sets of functions.
 */
@BeanDefinition(style = "light")
final class CompositeCalculationFunctions implements CalculationFunctions, ImmutableBean {

  /** The first set of calculation functions. */
  @PropertyDefinition(validate = "notNull")
  private final CalculationFunctions functions1;

  /** The second set of calculation functions. */
  @PropertyDefinition(validate = "notNull")
  private final CalculationFunctions functions2;

  /**
   * Returns a set of calculation functions composed of two other sets of functions.
   * <p>
   * If both sets of functions contain a function for a target then the function from {@code functions1} is returned.
   *
   * @param functions1  the higher priority set of functions
   * @param functions2  the lower priority set of functions
   * @return a set of calculation functions composed of two other sets of functions
   */
  static CompositeCalculationFunctions of(CalculationFunctions functions1, CalculationFunctions functions2) {
    return new CompositeCalculationFunctions(functions1, functions2);
  }

  @Override
  public <T extends CalculationTarget> Optional<CalculationFunction<? super T>> findFunction(T target) {
    Optional<CalculationFunction<? super T>> function = functions1.findFunction(target);
    return function.isPresent() ? function : functions2.findFunction(target);
  }

  //------------------------- AUTOGENERATED START -------------------------
  /**
   * The meta-bean for {@code CompositeCalculationFunctions}.
   */
  private static final TypedMetaBean<CompositeCalculationFunctions> META_BEAN =
      LightMetaBean.of(
          CompositeCalculationFunctions.class,
          MethodHandles.lookup(),
          new String[] {
              "functions1",
              "functions2"},
          new Object[0]);

  /**
   * The meta-bean for {@code CompositeCalculationFunctions}.
   * @return the meta-bean, not null
   */
  public static TypedMetaBean<CompositeCalculationFunctions> meta() {
    return META_BEAN;
  }

  static {
    MetaBean.register(META_BEAN);
  }

  private CompositeCalculationFunctions(
      CalculationFunctions functions1,
      CalculationFunctions functions2) {
    JodaBeanUtils.notNull(functions1, "functions1");
    JodaBeanUtils.notNull(functions2, "functions2");
    this.functions1 = functions1;
    this.functions2 = functions2;
  }

  @Override
  public TypedMetaBean<CompositeCalculationFunctions> metaBean() {
    return META_BEAN;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the first set of calculation functions.
   * @return the value of the property, not null
   */
  public CalculationFunctions getFunctions1() {
    return functions1;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the second set of calculation functions.
   * @return the value of the property, not null
   */
  public CalculationFunctions getFunctions2() {
    return functions2;
  }

  //-----------------------------------------------------------------------
  @Override
  public boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (obj != null && obj.getClass() == this.getClass()) {
      CompositeCalculationFunctions other = (CompositeCalculationFunctions) obj;
      return JodaBeanUtils.equal(functions1, other.functions1) &&
          JodaBeanUtils.equal(functions2, other.functions2);
    }
    return false;
  }

  @Override
  public int hashCode() {
    int hash = getClass().hashCode();
    hash = hash * 31 + JodaBeanUtils.hashCode(functions1);
    hash = hash * 31 + JodaBeanUtils.hashCode(functions2);
    return hash;
  }

  @Override
  public String toString() {
    StringBuilder buf = new StringBuilder(96);
    buf.append("CompositeCalculationFunctions{");
    buf.append("functions1").append('=').append(JodaBeanUtils.toString(functions1)).append(',').append(' ');
    buf.append("functions2").append('=').append(JodaBeanUtils.toString(functions2));
    buf.append('}');
    return buf.toString();
  }

  //-------------------------- AUTOGENERATED END --------------------------
}
