/*
 * Copyright (C) 2009 - present by OpenGamma Inc. and the OpenGamma group of companies
 *
 * Please see distribution for license.
 */
package com.opengamma.strata.collect.tuple;

import java.io.Serializable;
import java.util.Map;
import java.util.NoSuchElementException;

import org.joda.beans.Bean;
import org.joda.beans.BeanBuilder;
import org.joda.beans.ImmutableBean;
import org.joda.beans.JodaBeanUtils;
import org.joda.beans.MetaBean;
import org.joda.beans.MetaProperty;
import org.joda.beans.gen.BeanDefinition;
import org.joda.beans.gen.PropertyDefinition;
import org.joda.beans.impl.direct.DirectMetaBean;
import org.joda.beans.impl.direct.DirectMetaProperty;
import org.joda.beans.impl.direct.DirectMetaPropertyMap;
import org.joda.beans.impl.direct.DirectPrivateBeanBuilder;

import com.google.common.collect.ComparisonChain;
import com.google.common.collect.ImmutableList;
import com.opengamma.strata.collect.ArgChecker;

/**
 * An immutable pair consisting of an {@code Object} and a {@code double}.
 * <p>
 * This class is similar to {@link Pair} but includes a primitive element.
 * <p>
 * This class is immutable and thread-safe.
 * 
 * @param <A> the type of the object
 */
@BeanDefinition(builderScope = "private")
public final class ObjDoublePair<A>
    implements ImmutableBean, Tuple, Comparable<ObjDoublePair<A>>, Serializable {

  /**
   * The first element in this pair.
   */
  @PropertyDefinition(validate = "notNull")
  private final A first;
  /**
   * The second element in this pair.
   */
  @PropertyDefinition
  private final double second;

  //-------------------------------------------------------------------------
  /**
   * Obtains an instance from an {@code Object} and a {@code double}.
   * 
   * @param <A> the first element type
   * @param first  the first element
   * @param second  the second element
   * @return a pair formed from the two parameters
   */
  public static <A> ObjDoublePair<A> of(A first, double second) {
    return new ObjDoublePair<A>(first, second);
  }

  /**
   * Obtains an instance from a {@code Pair}.
   * 
   * @param <A> the first element type
   * @param pair  the pair to convert
   * @return a pair formed by extracting values from the pair
   */
  public static <A> ObjDoublePair<A> ofPair(Pair<A, Double> pair) {
    ArgChecker.notNull(pair, "pair");
    return new ObjDoublePair<A>(pair.getFirst(), pair.getSecond());
  }

  //-------------------------------------------------------------------------
  /**
   * Gets the number of elements held by this pair.
   * 
   * @return size 2
   */
  @Override
  public int size() {
    return 2;
  }

  /**
   * Gets the elements from this pair as a list.
   * <p>
   * The list returns each element in the pair in order.
   * 
   * @return the elements as an immutable list
   */
  @Override
  public ImmutableList<Object> elements() {
    return ImmutableList.of(first, second);
  }

  //-------------------------------------------------------------------------
  /**
   * Converts this pair to an object-based {@code Pair}.
   * 
   * @return the object-based pair
   */
  public Pair<A, Double> toPair() {
    return Pair.of(first, second);
  }

  //-------------------------------------------------------------------------
  /**
   * Compares the pair based on the first element followed by the second element.
   * <p>
   * The first element must be {@code Comparable}.
   * 
   * @param other  the other pair
   * @return negative if this is less, zero if equal, positive if greater
   * @throws ClassCastException if the object is not comparable
   */
  @Override
  public int compareTo(ObjDoublePair<A> other) {
    return ComparisonChain.start()
        .compare((Comparable<?>) first, (Comparable<?>) other.first)
        .compare(second, other.second)
        .result();
  }

  //-------------------------------------------------------------------------
  /**
   * Gets the pair using a standard string format.
   * <p>
   * The standard format is '[$first, $second]'. Spaces around the values are trimmed.
   * 
   * @return the pair as a string
   */
  @Override
  public String toString() {
    return new StringBuilder()
        .append('[')
        .append(first)
        .append(", ")
        .append(second)
        .append(']')
        .toString();
  }

  //------------------------- AUTOGENERATED START -------------------------
  /**
   * The meta-bean for {@code ObjDoublePair}.
   * @return the meta-bean, not null
   */
  @SuppressWarnings("rawtypes")
  public static ObjDoublePair.Meta meta() {
    return ObjDoublePair.Meta.INSTANCE;
  }

  /**
   * The meta-bean for {@code ObjDoublePair}.
   * @param <R>  the bean's generic type
   * @param cls  the bean's generic type
   * @return the meta-bean, not null
   */
  @SuppressWarnings("unchecked")
  public static <R> ObjDoublePair.Meta<R> metaObjDoublePair(Class<R> cls) {
    return ObjDoublePair.Meta.INSTANCE;
  }

  static {
    MetaBean.register(ObjDoublePair.Meta.INSTANCE);
  }

  /**
   * The serialization version id.
   */
  private static final long serialVersionUID = 1L;

  private ObjDoublePair(
      A first,
      double second) {
    JodaBeanUtils.notNull(first, "first");
    this.first = first;
    this.second = second;
  }

  @SuppressWarnings("unchecked")
  @Override
  public ObjDoublePair.Meta<A> metaBean() {
    return ObjDoublePair.Meta.INSTANCE;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the first element in this pair.
   * @return the value of the property, not null
   */
  public A getFirst() {
    return first;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the second element in this pair.
   * @return the value of the property
   */
  public double getSecond() {
    return second;
  }

  //-----------------------------------------------------------------------
  @Override
  public boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (obj != null && obj.getClass() == this.getClass()) {
      ObjDoublePair<?> other = (ObjDoublePair<?>) obj;
      return JodaBeanUtils.equal(first, other.first) &&
          JodaBeanUtils.equal(second, other.second);
    }
    return false;
  }

  @Override
  public int hashCode() {
    int hash = getClass().hashCode();
    hash = hash * 31 + JodaBeanUtils.hashCode(first);
    hash = hash * 31 + JodaBeanUtils.hashCode(second);
    return hash;
  }

  //-----------------------------------------------------------------------
  /**
   * The meta-bean for {@code ObjDoublePair}.
   * @param <A>  the type
   */
  public static final class Meta<A> extends DirectMetaBean {
    /**
     * The singleton instance of the meta-bean.
     */
    @SuppressWarnings("rawtypes")
    static final Meta INSTANCE = new Meta();

    /**
     * The meta-property for the {@code first} property.
     */
    @SuppressWarnings({"unchecked", "rawtypes" })
    private final MetaProperty<A> first = (DirectMetaProperty) DirectMetaProperty.ofImmutable(
        this, "first", ObjDoublePair.class, Object.class);
    /**
     * The meta-property for the {@code second} property.
     */
    private final MetaProperty<Double> second = DirectMetaProperty.ofImmutable(
        this, "second", ObjDoublePair.class, Double.TYPE);
    /**
     * The meta-properties.
     */
    private final Map<String, MetaProperty<?>> metaPropertyMap$ = new DirectMetaPropertyMap(
        this, null,
        "first",
        "second");

    /**
     * Restricted constructor.
     */
    private Meta() {
    }

    @Override
    protected MetaProperty<?> metaPropertyGet(String propertyName) {
      switch (propertyName.hashCode()) {
        case 97440432:  // first
          return first;
        case -906279820:  // second
          return second;
      }
      return super.metaPropertyGet(propertyName);
    }

    @Override
    public BeanBuilder<? extends ObjDoublePair<A>> builder() {
      return new ObjDoublePair.Builder<>();
    }

    @SuppressWarnings({"unchecked", "rawtypes" })
    @Override
    public Class<? extends ObjDoublePair<A>> beanType() {
      return (Class) ObjDoublePair.class;
    }

    @Override
    public Map<String, MetaProperty<?>> metaPropertyMap() {
      return metaPropertyMap$;
    }

    //-----------------------------------------------------------------------
    /**
     * The meta-property for the {@code first} property.
     * @return the meta-property, not null
     */
    public MetaProperty<A> first() {
      return first;
    }

    /**
     * The meta-property for the {@code second} property.
     * @return the meta-property, not null
     */
    public MetaProperty<Double> second() {
      return second;
    }

    //-----------------------------------------------------------------------
    @Override
    protected Object propertyGet(Bean bean, String propertyName, boolean quiet) {
      switch (propertyName.hashCode()) {
        case 97440432:  // first
          return ((ObjDoublePair<?>) bean).getFirst();
        case -906279820:  // second
          return ((ObjDoublePair<?>) bean).getSecond();
      }
      return super.propertyGet(bean, propertyName, quiet);
    }

    @Override
    protected void propertySet(Bean bean, String propertyName, Object newValue, boolean quiet) {
      metaProperty(propertyName);
      if (quiet) {
        return;
      }
      throw new UnsupportedOperationException("Property cannot be written: " + propertyName);
    }

  }

  //-----------------------------------------------------------------------
  /**
   * The bean-builder for {@code ObjDoublePair}.
   * @param <A>  the type
   */
  private static final class Builder<A> extends DirectPrivateBeanBuilder<ObjDoublePair<A>> {

    private A first;
    private double second;

    /**
     * Restricted constructor.
     */
    private Builder() {
    }

    //-----------------------------------------------------------------------
    @Override
    public Object get(String propertyName) {
      switch (propertyName.hashCode()) {
        case 97440432:  // first
          return first;
        case -906279820:  // second
          return second;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
    }

    @SuppressWarnings("unchecked")
    @Override
    public Builder<A> set(String propertyName, Object newValue) {
      switch (propertyName.hashCode()) {
        case 97440432:  // first
          this.first = (A) newValue;
          break;
        case -906279820:  // second
          this.second = (Double) newValue;
          break;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
      return this;
    }

    @Override
    public ObjDoublePair<A> build() {
      return new ObjDoublePair<>(
          first,
          second);
    }

    //-----------------------------------------------------------------------
    @Override
    public String toString() {
      StringBuilder buf = new StringBuilder(96);
      buf.append("ObjDoublePair.Builder{");
      buf.append("first").append('=').append(JodaBeanUtils.toString(first)).append(',').append(' ');
      buf.append("second").append('=').append(JodaBeanUtils.toString(second));
      buf.append('}');
      return buf.toString();
    }

  }

  //-------------------------- AUTOGENERATED END --------------------------
}
