/*
 * Copyright (C) 2017 - present by OpenGamma Inc. and the OpenGamma group of companies
 *
 * Please see distribution for license.
 */
package com.opengamma.strata.market;

import java.io.Serializable;
import java.util.Map;
import java.util.NoSuchElementException;

import org.joda.beans.Bean;
import org.joda.beans.BeanBuilder;
import org.joda.beans.ImmutableBean;
import org.joda.beans.JodaBeanUtils;
import org.joda.beans.MetaBean;
import org.joda.beans.MetaProperty;
import org.joda.beans.gen.BeanDefinition;
import org.joda.beans.gen.PropertyDefinition;
import org.joda.beans.impl.direct.DirectMetaBean;
import org.joda.beans.impl.direct.DirectMetaProperty;
import org.joda.beans.impl.direct.DirectMetaPropertyMap;
import org.joda.beans.impl.direct.DirectPrivateBeanBuilder;

import com.opengamma.strata.basics.ReferenceData;
import com.opengamma.strata.collect.array.DoubleArray;
import com.opengamma.strata.data.scenario.MarketDataBox;
import com.opengamma.strata.data.scenario.ScenarioPerturbation;

/**
 * A perturbation that applies different shifts to a double value.
 * <p>
 * This class contains shifts, each of which is associated with a scenario and applied to a value based on the shift type.
 * A constant spread can be added for all of the scenarios. 
 * <p>
 * A typical application of this is to generate scenario perturbation to market quotes, 
 * where the market quote is represented in terms of {@code QuoteId} and {@code Double}.
 */
@BeanDefinition(builderScope = "private", constructorScope = "package")
public final class GenericDoubleShifts
    implements ScenarioPerturbation<Double>, ImmutableBean, Serializable {

  /**
   * The type of shift applied to a {@code Double} value.
   */
  @PropertyDefinition(validate = "notNull")
  private final ShiftType shiftType;

  /**
   * The shifts to apply to a {@code Double} value.
   * <p>
   * Each element in the array corresponds to each scenario.
   */
  @PropertyDefinition(validate = "notNull")
  private final DoubleArray shiftAmount;

  /**
   * The constant spread.
   */
  @PropertyDefinition(validate = "notNull")
  private final double spread;

  //-------------------------------------------------------------------------
  /**
   * Creates an instance with zero spread. 
   * 
   * @param shiftType  the shift type
   * @param shiftAmount  the shift amount
   * @return the instance
   */
  public static GenericDoubleShifts of(ShiftType shiftType, DoubleArray shiftAmount) {
    return new GenericDoubleShifts(shiftType, shiftAmount, 0d);
  }

  /**
   * Creates an instance with spread.
   * 
   * @param shiftType  the shift type
   * @param shiftAmount  the shift amount
   * @param spread  the spread
   * @return the instance
   */
  public static GenericDoubleShifts of(ShiftType shiftType, DoubleArray shiftAmount, double spread) {
    return new GenericDoubleShifts(shiftType, shiftAmount, spread);
  }

  //-------------------------------------------------------------------------
  @Override
  public MarketDataBox<Double> applyTo(MarketDataBox<Double> marketData, ReferenceData refData) {
    return marketData.mapWithIndex(
        getScenarioCount(),
        (value, scenarioIndex) -> shiftType.applyShift(value + spread, shiftAmount.get(scenarioIndex)) - spread);
  }

  @Override
  public int getScenarioCount() {
    return shiftAmount.size();
  }

  @Override
  public Class<Double> getMarketDataType() {
    return Double.class;
  }

  //------------------------- AUTOGENERATED START -------------------------
  /**
   * The meta-bean for {@code GenericDoubleShifts}.
   * @return the meta-bean, not null
   */
  public static GenericDoubleShifts.Meta meta() {
    return GenericDoubleShifts.Meta.INSTANCE;
  }

  static {
    MetaBean.register(GenericDoubleShifts.Meta.INSTANCE);
  }

  /**
   * The serialization version id.
   */
  private static final long serialVersionUID = 1L;

  /**
   * Creates an instance.
   * @param shiftType  the value of the property, not null
   * @param shiftAmount  the value of the property, not null
   * @param spread  the value of the property, not null
   */
  GenericDoubleShifts(
      ShiftType shiftType,
      DoubleArray shiftAmount,
      double spread) {
    JodaBeanUtils.notNull(shiftType, "shiftType");
    JodaBeanUtils.notNull(shiftAmount, "shiftAmount");
    JodaBeanUtils.notNull(spread, "spread");
    this.shiftType = shiftType;
    this.shiftAmount = shiftAmount;
    this.spread = spread;
  }

  @Override
  public GenericDoubleShifts.Meta metaBean() {
    return GenericDoubleShifts.Meta.INSTANCE;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the type of shift applied to a {@code Double} value.
   * @return the value of the property, not null
   */
  public ShiftType getShiftType() {
    return shiftType;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the shifts to apply to a {@code Double} value.
   * <p>
   * Each element in the array corresponds to each scenario.
   * @return the value of the property, not null
   */
  public DoubleArray getShiftAmount() {
    return shiftAmount;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the constant spread.
   * @return the value of the property, not null
   */
  public double getSpread() {
    return spread;
  }

  //-----------------------------------------------------------------------
  @Override
  public boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (obj != null && obj.getClass() == this.getClass()) {
      GenericDoubleShifts other = (GenericDoubleShifts) obj;
      return JodaBeanUtils.equal(shiftType, other.shiftType) &&
          JodaBeanUtils.equal(shiftAmount, other.shiftAmount) &&
          JodaBeanUtils.equal(spread, other.spread);
    }
    return false;
  }

  @Override
  public int hashCode() {
    int hash = getClass().hashCode();
    hash = hash * 31 + JodaBeanUtils.hashCode(shiftType);
    hash = hash * 31 + JodaBeanUtils.hashCode(shiftAmount);
    hash = hash * 31 + JodaBeanUtils.hashCode(spread);
    return hash;
  }

  @Override
  public String toString() {
    StringBuilder buf = new StringBuilder(128);
    buf.append("GenericDoubleShifts{");
    buf.append("shiftType").append('=').append(JodaBeanUtils.toString(shiftType)).append(',').append(' ');
    buf.append("shiftAmount").append('=').append(JodaBeanUtils.toString(shiftAmount)).append(',').append(' ');
    buf.append("spread").append('=').append(JodaBeanUtils.toString(spread));
    buf.append('}');
    return buf.toString();
  }

  //-----------------------------------------------------------------------
  /**
   * The meta-bean for {@code GenericDoubleShifts}.
   */
  public static final class Meta extends DirectMetaBean {
    /**
     * The singleton instance of the meta-bean.
     */
    static final Meta INSTANCE = new Meta();

    /**
     * The meta-property for the {@code shiftType} property.
     */
    private final MetaProperty<ShiftType> shiftType = DirectMetaProperty.ofImmutable(
        this, "shiftType", GenericDoubleShifts.class, ShiftType.class);
    /**
     * The meta-property for the {@code shiftAmount} property.
     */
    private final MetaProperty<DoubleArray> shiftAmount = DirectMetaProperty.ofImmutable(
        this, "shiftAmount", GenericDoubleShifts.class, DoubleArray.class);
    /**
     * The meta-property for the {@code spread} property.
     */
    private final MetaProperty<Double> spread = DirectMetaProperty.ofImmutable(
        this, "spread", GenericDoubleShifts.class, Double.TYPE);
    /**
     * The meta-properties.
     */
    private final Map<String, MetaProperty<?>> metaPropertyMap$ = new DirectMetaPropertyMap(
        this, null,
        "shiftType",
        "shiftAmount",
        "spread");

    /**
     * Restricted constructor.
     */
    private Meta() {
    }

    @Override
    protected MetaProperty<?> metaPropertyGet(String propertyName) {
      switch (propertyName.hashCode()) {
        case 893345500:  // shiftType
          return shiftType;
        case -1043480710:  // shiftAmount
          return shiftAmount;
        case -895684237:  // spread
          return spread;
      }
      return super.metaPropertyGet(propertyName);
    }

    @Override
    public BeanBuilder<? extends GenericDoubleShifts> builder() {
      return new GenericDoubleShifts.Builder();
    }

    @Override
    public Class<? extends GenericDoubleShifts> beanType() {
      return GenericDoubleShifts.class;
    }

    @Override
    public Map<String, MetaProperty<?>> metaPropertyMap() {
      return metaPropertyMap$;
    }

    //-----------------------------------------------------------------------
    /**
     * The meta-property for the {@code shiftType} property.
     * @return the meta-property, not null
     */
    public MetaProperty<ShiftType> shiftType() {
      return shiftType;
    }

    /**
     * The meta-property for the {@code shiftAmount} property.
     * @return the meta-property, not null
     */
    public MetaProperty<DoubleArray> shiftAmount() {
      return shiftAmount;
    }

    /**
     * The meta-property for the {@code spread} property.
     * @return the meta-property, not null
     */
    public MetaProperty<Double> spread() {
      return spread;
    }

    //-----------------------------------------------------------------------
    @Override
    protected Object propertyGet(Bean bean, String propertyName, boolean quiet) {
      switch (propertyName.hashCode()) {
        case 893345500:  // shiftType
          return ((GenericDoubleShifts) bean).getShiftType();
        case -1043480710:  // shiftAmount
          return ((GenericDoubleShifts) bean).getShiftAmount();
        case -895684237:  // spread
          return ((GenericDoubleShifts) bean).getSpread();
      }
      return super.propertyGet(bean, propertyName, quiet);
    }

    @Override
    protected void propertySet(Bean bean, String propertyName, Object newValue, boolean quiet) {
      metaProperty(propertyName);
      if (quiet) {
        return;
      }
      throw new UnsupportedOperationException("Property cannot be written: " + propertyName);
    }

  }

  //-----------------------------------------------------------------------
  /**
   * The bean-builder for {@code GenericDoubleShifts}.
   */
  private static final class Builder extends DirectPrivateBeanBuilder<GenericDoubleShifts> {

    private ShiftType shiftType;
    private DoubleArray shiftAmount;
    private double spread;

    /**
     * Restricted constructor.
     */
    private Builder() {
    }

    //-----------------------------------------------------------------------
    @Override
    public Object get(String propertyName) {
      switch (propertyName.hashCode()) {
        case 893345500:  // shiftType
          return shiftType;
        case -1043480710:  // shiftAmount
          return shiftAmount;
        case -895684237:  // spread
          return spread;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
    }

    @Override
    public Builder set(String propertyName, Object newValue) {
      switch (propertyName.hashCode()) {
        case 893345500:  // shiftType
          this.shiftType = (ShiftType) newValue;
          break;
        case -1043480710:  // shiftAmount
          this.shiftAmount = (DoubleArray) newValue;
          break;
        case -895684237:  // spread
          this.spread = (Double) newValue;
          break;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
      return this;
    }

    @Override
    public GenericDoubleShifts build() {
      return new GenericDoubleShifts(
          shiftType,
          shiftAmount,
          spread);
    }

    //-----------------------------------------------------------------------
    @Override
    public String toString() {
      StringBuilder buf = new StringBuilder(128);
      buf.append("GenericDoubleShifts.Builder{");
      buf.append("shiftType").append('=').append(JodaBeanUtils.toString(shiftType)).append(',').append(' ');
      buf.append("shiftAmount").append('=').append(JodaBeanUtils.toString(shiftAmount)).append(',').append(' ');
      buf.append("spread").append('=').append(JodaBeanUtils.toString(spread));
      buf.append('}');
      return buf.toString();
    }

  }

  //-------------------------- AUTOGENERATED END --------------------------
}
