/*
 * Copyright (C) 2018 - present by OpenGamma Inc. and the OpenGamma group of companies
 *
 * Please see distribution for license.
 */
package com.opengamma.strata.market.curve;

import java.io.Serializable;
import java.lang.invoke.MethodHandles;

import org.joda.beans.ImmutableBean;
import org.joda.beans.JodaBeanUtils;
import org.joda.beans.MetaBean;
import org.joda.beans.TypedMetaBean;
import org.joda.beans.gen.BeanDefinition;
import org.joda.beans.gen.PropertyDefinition;
import org.joda.beans.impl.light.LightMetaBean;

import com.opengamma.strata.data.MarketDataId;
import com.opengamma.strata.data.ObservableSource;

/**
 * An identifier used to access a curve group by name.
 * <p>
 * This is used when there is a need to obtain an instance of {@link LegalEntityCurveGroup}.
 */
@BeanDefinition(style = "light", cacheHashCode = true)
public final class LegalEntityCurveGroupId
    implements MarketDataId<LegalEntityCurveGroup>, ImmutableBean, Serializable {

  /**
   * The curve group name.
   */
  @PropertyDefinition(validate = "notNull")
  private final CurveGroupName curveGroupName;
  /**
   * The source of observable market data.
   */
  @PropertyDefinition(validate = "notNull")
  private final ObservableSource observableSource;

  //-------------------------------------------------------------------------
  /**
   * Obtains an instance used to obtain a curve group by name.
   *
   * @param groupName  the curve group name
   * @return the identifier
   */
  public static LegalEntityCurveGroupId of(String groupName) {
    return new LegalEntityCurveGroupId(CurveGroupName.of(groupName), ObservableSource.NONE);
  }

  /**
   * Obtains an instance used to obtain a curve group by name.
   *
   * @param groupName  the curve group name
   * @return the identifier
   */
  public static LegalEntityCurveGroupId of(CurveGroupName groupName) {
    return new LegalEntityCurveGroupId(groupName, ObservableSource.NONE);
  }

  /**
   * Obtains an instance used to obtain a curve group by name, specifying the source of observable market data.
   *
   * @param groupName  the curve group name
   * @param obsSource  source of observable market data
   * @return the identifier
   */
  public static LegalEntityCurveGroupId of(CurveGroupName groupName, ObservableSource obsSource) {
    return new LegalEntityCurveGroupId(groupName, obsSource);
  }

  //-------------------------------------------------------------------------
  @Override
  public Class<LegalEntityCurveGroup> getMarketDataType() {
    return LegalEntityCurveGroup.class;
  }

  @Override
  public String toString() {
    return new StringBuilder(32)
        .append("LegalEntityCurveGroupId:")
        .append(curveGroupName)
        .append(observableSource.equals(ObservableSource.NONE) ? "" : "/" + observableSource)
        .toString();
  }

  //------------------------- AUTOGENERATED START -------------------------
  /**
   * The meta-bean for {@code LegalEntityCurveGroupId}.
   */
  private static final TypedMetaBean<LegalEntityCurveGroupId> META_BEAN =
      LightMetaBean.of(
          LegalEntityCurveGroupId.class,
          MethodHandles.lookup(),
          new String[] {
              "curveGroupName",
              "observableSource"},
          new Object[0]);

  /**
   * The meta-bean for {@code LegalEntityCurveGroupId}.
   * @return the meta-bean, not null
   */
  public static TypedMetaBean<LegalEntityCurveGroupId> meta() {
    return META_BEAN;
  }

  static {
    MetaBean.register(META_BEAN);
  }

  /**
   * The serialization version id.
   */
  private static final long serialVersionUID = 1L;

  /**
   * The cached hash code, using the racy single-check idiom.
   */
  private transient int cacheHashCode;

  private LegalEntityCurveGroupId(
      CurveGroupName curveGroupName,
      ObservableSource observableSource) {
    JodaBeanUtils.notNull(curveGroupName, "curveGroupName");
    JodaBeanUtils.notNull(observableSource, "observableSource");
    this.curveGroupName = curveGroupName;
    this.observableSource = observableSource;
  }

  @Override
  public TypedMetaBean<LegalEntityCurveGroupId> metaBean() {
    return META_BEAN;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the curve group name.
   * @return the value of the property, not null
   */
  public CurveGroupName getCurveGroupName() {
    return curveGroupName;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the source of observable market data.
   * @return the value of the property, not null
   */
  public ObservableSource getObservableSource() {
    return observableSource;
  }

  //-----------------------------------------------------------------------
  @Override
  public boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (obj != null && obj.getClass() == this.getClass()) {
      LegalEntityCurveGroupId other = (LegalEntityCurveGroupId) obj;
      return JodaBeanUtils.equal(curveGroupName, other.curveGroupName) &&
          JodaBeanUtils.equal(observableSource, other.observableSource);
    }
    return false;
  }

  @Override
  public int hashCode() {
    int hash = cacheHashCode;
    if (hash == 0) {
      hash = getClass().hashCode();
      hash = hash * 31 + JodaBeanUtils.hashCode(curveGroupName);
      hash = hash * 31 + JodaBeanUtils.hashCode(observableSource);
      cacheHashCode = hash;
    }
    return hash;
  }

  //-------------------------- AUTOGENERATED END --------------------------
}
