/*
 * Copyright (C) 2015 - present by OpenGamma Inc. and the OpenGamma group of companies
 *
 * Please see distribution for license.
 */
package com.opengamma.strata.market.curve;

import java.io.Serializable;
import java.util.Map;
import java.util.NoSuchElementException;

import org.joda.beans.Bean;
import org.joda.beans.ImmutableBean;
import org.joda.beans.JodaBeanUtils;
import org.joda.beans.MetaBean;
import org.joda.beans.MetaProperty;
import org.joda.beans.gen.BeanDefinition;
import org.joda.beans.gen.PropertyDefinition;
import org.joda.beans.impl.direct.DirectFieldsBeanBuilder;
import org.joda.beans.impl.direct.DirectMetaBean;
import org.joda.beans.impl.direct.DirectMetaProperty;
import org.joda.beans.impl.direct.DirectMetaPropertyMap;

import com.google.common.collect.ImmutableMap;
import com.opengamma.strata.data.MarketDataId;

/**
 * The input data used when calibrating a curve.
 * <p>
 * This class contains the current market value of a set of instruments used when calibrating a curve.
 */
@BeanDefinition
public final class RatesCurveInputs
    implements ImmutableBean, Serializable {

  /**
   * The market data.
   * <p>
   * There will typically be at least one entry for each node on the curve.
   */
  @PropertyDefinition(validate = "notNull", builderType = "Map<? extends MarketDataId<?>, ?>")
  private final ImmutableMap<? extends MarketDataId<?>, ?> marketData;
  /**
   * The metadata for the curve.
   * <p>
   * This is used to identify the curve and the necessary pieces of market data.
   */
  @PropertyDefinition(validate = "notNull")
  private final CurveMetadata curveMetadata;

  //-------------------------------------------------------------------------
  /**
   * Returns a {@code CurveInputs} instance containing the specified market data.
   *
   * @param marketData  a map of market data, keyed by the ID of the data
   * @param metadata  the metadata for the curve
   * @return a {@code CurveInputs} instance containing the specified market data
   */
  public static RatesCurveInputs of(Map<? extends MarketDataId<?>, ?> marketData, CurveMetadata metadata) {
    return new RatesCurveInputs(marketData, metadata);
  }

  //------------------------- AUTOGENERATED START -------------------------
  /**
   * The meta-bean for {@code RatesCurveInputs}.
   * @return the meta-bean, not null
   */
  public static RatesCurveInputs.Meta meta() {
    return RatesCurveInputs.Meta.INSTANCE;
  }

  static {
    MetaBean.register(RatesCurveInputs.Meta.INSTANCE);
  }

  /**
   * The serialization version id.
   */
  private static final long serialVersionUID = 1L;

  /**
   * Returns a builder used to create an instance of the bean.
   * @return the builder, not null
   */
  public static RatesCurveInputs.Builder builder() {
    return new RatesCurveInputs.Builder();
  }

  private RatesCurveInputs(
      Map<? extends MarketDataId<?>, ?> marketData,
      CurveMetadata curveMetadata) {
    JodaBeanUtils.notNull(marketData, "marketData");
    JodaBeanUtils.notNull(curveMetadata, "curveMetadata");
    this.marketData = ImmutableMap.copyOf(marketData);
    this.curveMetadata = curveMetadata;
  }

  @Override
  public RatesCurveInputs.Meta metaBean() {
    return RatesCurveInputs.Meta.INSTANCE;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the market data.
   * <p>
   * There will typically be at least one entry for each node on the curve.
   * @return the value of the property, not null
   */
  public ImmutableMap<? extends MarketDataId<?>, ?> getMarketData() {
    return marketData;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the metadata for the curve.
   * <p>
   * This is used to identify the curve and the necessary pieces of market data.
   * @return the value of the property, not null
   */
  public CurveMetadata getCurveMetadata() {
    return curveMetadata;
  }

  //-----------------------------------------------------------------------
  /**
   * Returns a builder that allows this bean to be mutated.
   * @return the mutable builder, not null
   */
  public Builder toBuilder() {
    return new Builder(this);
  }

  @Override
  public boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (obj != null && obj.getClass() == this.getClass()) {
      RatesCurveInputs other = (RatesCurveInputs) obj;
      return JodaBeanUtils.equal(marketData, other.marketData) &&
          JodaBeanUtils.equal(curveMetadata, other.curveMetadata);
    }
    return false;
  }

  @Override
  public int hashCode() {
    int hash = getClass().hashCode();
    hash = hash * 31 + JodaBeanUtils.hashCode(marketData);
    hash = hash * 31 + JodaBeanUtils.hashCode(curveMetadata);
    return hash;
  }

  @Override
  public String toString() {
    StringBuilder buf = new StringBuilder(96);
    buf.append("RatesCurveInputs{");
    buf.append("marketData").append('=').append(JodaBeanUtils.toString(marketData)).append(',').append(' ');
    buf.append("curveMetadata").append('=').append(JodaBeanUtils.toString(curveMetadata));
    buf.append('}');
    return buf.toString();
  }

  //-----------------------------------------------------------------------
  /**
   * The meta-bean for {@code RatesCurveInputs}.
   */
  public static final class Meta extends DirectMetaBean {
    /**
     * The singleton instance of the meta-bean.
     */
    static final Meta INSTANCE = new Meta();

    /**
     * The meta-property for the {@code marketData} property.
     */
    @SuppressWarnings({"unchecked", "rawtypes" })
    private final MetaProperty<ImmutableMap<? extends MarketDataId<?>, ?>> marketData = DirectMetaProperty.ofImmutable(
        this, "marketData", RatesCurveInputs.class, (Class) ImmutableMap.class);
    /**
     * The meta-property for the {@code curveMetadata} property.
     */
    private final MetaProperty<CurveMetadata> curveMetadata = DirectMetaProperty.ofImmutable(
        this, "curveMetadata", RatesCurveInputs.class, CurveMetadata.class);
    /**
     * The meta-properties.
     */
    private final Map<String, MetaProperty<?>> metaPropertyMap$ = new DirectMetaPropertyMap(
        this, null,
        "marketData",
        "curveMetadata");

    /**
     * Restricted constructor.
     */
    private Meta() {
    }

    @Override
    protected MetaProperty<?> metaPropertyGet(String propertyName) {
      switch (propertyName.hashCode()) {
        case 1116764678:  // marketData
          return marketData;
        case 278233406:  // curveMetadata
          return curveMetadata;
      }
      return super.metaPropertyGet(propertyName);
    }

    @Override
    public RatesCurveInputs.Builder builder() {
      return new RatesCurveInputs.Builder();
    }

    @Override
    public Class<? extends RatesCurveInputs> beanType() {
      return RatesCurveInputs.class;
    }

    @Override
    public Map<String, MetaProperty<?>> metaPropertyMap() {
      return metaPropertyMap$;
    }

    //-----------------------------------------------------------------------
    /**
     * The meta-property for the {@code marketData} property.
     * @return the meta-property, not null
     */
    public MetaProperty<ImmutableMap<? extends MarketDataId<?>, ?>> marketData() {
      return marketData;
    }

    /**
     * The meta-property for the {@code curveMetadata} property.
     * @return the meta-property, not null
     */
    public MetaProperty<CurveMetadata> curveMetadata() {
      return curveMetadata;
    }

    //-----------------------------------------------------------------------
    @Override
    protected Object propertyGet(Bean bean, String propertyName, boolean quiet) {
      switch (propertyName.hashCode()) {
        case 1116764678:  // marketData
          return ((RatesCurveInputs) bean).getMarketData();
        case 278233406:  // curveMetadata
          return ((RatesCurveInputs) bean).getCurveMetadata();
      }
      return super.propertyGet(bean, propertyName, quiet);
    }

    @Override
    protected void propertySet(Bean bean, String propertyName, Object newValue, boolean quiet) {
      metaProperty(propertyName);
      if (quiet) {
        return;
      }
      throw new UnsupportedOperationException("Property cannot be written: " + propertyName);
    }

  }

  //-----------------------------------------------------------------------
  /**
   * The bean-builder for {@code RatesCurveInputs}.
   */
  public static final class Builder extends DirectFieldsBeanBuilder<RatesCurveInputs> {

    private Map<? extends MarketDataId<?>, ?> marketData = ImmutableMap.of();
    private CurveMetadata curveMetadata;

    /**
     * Restricted constructor.
     */
    private Builder() {
    }

    /**
     * Restricted copy constructor.
     * @param beanToCopy  the bean to copy from, not null
     */
    private Builder(RatesCurveInputs beanToCopy) {
      this.marketData = beanToCopy.getMarketData();
      this.curveMetadata = beanToCopy.getCurveMetadata();
    }

    //-----------------------------------------------------------------------
    @Override
    public Object get(String propertyName) {
      switch (propertyName.hashCode()) {
        case 1116764678:  // marketData
          return marketData;
        case 278233406:  // curveMetadata
          return curveMetadata;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
    }

    @SuppressWarnings("unchecked")
    @Override
    public Builder set(String propertyName, Object newValue) {
      switch (propertyName.hashCode()) {
        case 1116764678:  // marketData
          this.marketData = (Map<? extends MarketDataId<?>, ?>) newValue;
          break;
        case 278233406:  // curveMetadata
          this.curveMetadata = (CurveMetadata) newValue;
          break;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
      return this;
    }

    @Override
    public Builder set(MetaProperty<?> property, Object value) {
      super.set(property, value);
      return this;
    }

    @Override
    public RatesCurveInputs build() {
      return new RatesCurveInputs(
          marketData,
          curveMetadata);
    }

    //-----------------------------------------------------------------------
    /**
     * Sets the market data.
     * <p>
     * There will typically be at least one entry for each node on the curve.
     * @param marketData  the new value, not null
     * @return this, for chaining, not null
     */
    public Builder marketData(Map<? extends MarketDataId<?>, ?> marketData) {
      JodaBeanUtils.notNull(marketData, "marketData");
      this.marketData = marketData;
      return this;
    }

    /**
     * Sets the metadata for the curve.
     * <p>
     * This is used to identify the curve and the necessary pieces of market data.
     * @param curveMetadata  the new value, not null
     * @return this, for chaining, not null
     */
    public Builder curveMetadata(CurveMetadata curveMetadata) {
      JodaBeanUtils.notNull(curveMetadata, "curveMetadata");
      this.curveMetadata = curveMetadata;
      return this;
    }

    //-----------------------------------------------------------------------
    @Override
    public String toString() {
      StringBuilder buf = new StringBuilder(96);
      buf.append("RatesCurveInputs.Builder{");
      buf.append("marketData").append('=').append(JodaBeanUtils.toString(marketData)).append(',').append(' ');
      buf.append("curveMetadata").append('=').append(JodaBeanUtils.toString(curveMetadata));
      buf.append('}');
      return buf.toString();
    }

  }

  //-------------------------- AUTOGENERATED END --------------------------
}
