/*
 * Copyright (C) 2016 - present by OpenGamma Inc. and the OpenGamma group of companies
 *
 * Please see distribution for license.
 */
package com.opengamma.strata.market.curve;

import java.io.Serializable;
import java.util.Map;
import java.util.NoSuchElementException;

import org.joda.beans.Bean;
import org.joda.beans.BeanBuilder;
import org.joda.beans.ImmutableBean;
import org.joda.beans.JodaBeanUtils;
import org.joda.beans.MetaBean;
import org.joda.beans.MetaProperty;
import org.joda.beans.gen.BeanDefinition;
import org.joda.beans.gen.PropertyDefinition;
import org.joda.beans.impl.direct.DirectMetaBean;
import org.joda.beans.impl.direct.DirectMetaProperty;
import org.joda.beans.impl.direct.DirectMetaPropertyMap;
import org.joda.beans.impl.direct.DirectPrivateBeanBuilder;

import com.opengamma.strata.collect.ArgChecker;
import com.opengamma.strata.collect.array.DoubleArray;
import com.opengamma.strata.market.ShiftType;

/**
 * Provides the definition of seasonality for a price index curve.
 * <p>
 * The seasonality is describe by a adjustment type and the month on month adjustments.
 * The adjustment type is usually {@link ShiftType#SCALED} (multiplicative) or {@link ShiftType#ABSOLUTE} (additive).
 * The month on month adjustment is an array of length 12 with the first element being the
 * adjustment from January to February, the second element being the adjustment from February to March,
 * and so on to the 12th element being the adjustment from  December to January.
 */
@BeanDefinition(builderScope = "private")
public final class SeasonalityDefinition
    implements ImmutableBean, Serializable {

  /**
   * The month on month adjustment.
   * <p>
   * This is an array of length 12, with the first element being the adjustment from
   * January to February, the second element being the adjustment from February to March,
   * and so on to the 12th element being the adjustment from December to January.
   */
  @PropertyDefinition(validate = "notNull")
  private final DoubleArray seasonalityMonthOnMonth;
  /**
   * The shift type applied to the unadjusted value and the adjustment.
   * (value, seasonality) -> adjustmentType.applyShift(value, seasonality).
   */
  @PropertyDefinition(validate = "notNull")
  private final ShiftType adjustmentType;

  //-------------------------------------------------------------------------
  /**
   * Obtains an instance of the seasonality.
   * 
   * @param seasonalityMonthOnMonth  the month-on-month seasonality
   * @param adjustmentType  the adjustment type
   * @return the instance
   */
  public static SeasonalityDefinition of(DoubleArray seasonalityMonthOnMonth, ShiftType adjustmentType) {
    ArgChecker.isTrue(seasonalityMonthOnMonth.size() == 12, "seasonality must be of length 12");
    return new SeasonalityDefinition(seasonalityMonthOnMonth, adjustmentType);
  }

  //------------------------- AUTOGENERATED START -------------------------
  /**
   * The meta-bean for {@code SeasonalityDefinition}.
   * @return the meta-bean, not null
   */
  public static SeasonalityDefinition.Meta meta() {
    return SeasonalityDefinition.Meta.INSTANCE;
  }

  static {
    MetaBean.register(SeasonalityDefinition.Meta.INSTANCE);
  }

  /**
   * The serialization version id.
   */
  private static final long serialVersionUID = 1L;

  private SeasonalityDefinition(
      DoubleArray seasonalityMonthOnMonth,
      ShiftType adjustmentType) {
    JodaBeanUtils.notNull(seasonalityMonthOnMonth, "seasonalityMonthOnMonth");
    JodaBeanUtils.notNull(adjustmentType, "adjustmentType");
    this.seasonalityMonthOnMonth = seasonalityMonthOnMonth;
    this.adjustmentType = adjustmentType;
  }

  @Override
  public SeasonalityDefinition.Meta metaBean() {
    return SeasonalityDefinition.Meta.INSTANCE;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the month on month adjustment.
   * <p>
   * This is an array of length 12, with the first element being the adjustment from
   * January to February, the second element being the adjustment from February to March,
   * and so on to the 12th element being the adjustment from December to January.
   * @return the value of the property, not null
   */
  public DoubleArray getSeasonalityMonthOnMonth() {
    return seasonalityMonthOnMonth;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the shift type applied to the unadjusted value and the adjustment.
   * (value, seasonality) -> adjustmentType.applyShift(value, seasonality).
   * @return the value of the property, not null
   */
  public ShiftType getAdjustmentType() {
    return adjustmentType;
  }

  //-----------------------------------------------------------------------
  @Override
  public boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (obj != null && obj.getClass() == this.getClass()) {
      SeasonalityDefinition other = (SeasonalityDefinition) obj;
      return JodaBeanUtils.equal(seasonalityMonthOnMonth, other.seasonalityMonthOnMonth) &&
          JodaBeanUtils.equal(adjustmentType, other.adjustmentType);
    }
    return false;
  }

  @Override
  public int hashCode() {
    int hash = getClass().hashCode();
    hash = hash * 31 + JodaBeanUtils.hashCode(seasonalityMonthOnMonth);
    hash = hash * 31 + JodaBeanUtils.hashCode(adjustmentType);
    return hash;
  }

  @Override
  public String toString() {
    StringBuilder buf = new StringBuilder(96);
    buf.append("SeasonalityDefinition{");
    buf.append("seasonalityMonthOnMonth").append('=').append(JodaBeanUtils.toString(seasonalityMonthOnMonth)).append(',').append(' ');
    buf.append("adjustmentType").append('=').append(JodaBeanUtils.toString(adjustmentType));
    buf.append('}');
    return buf.toString();
  }

  //-----------------------------------------------------------------------
  /**
   * The meta-bean for {@code SeasonalityDefinition}.
   */
  public static final class Meta extends DirectMetaBean {
    /**
     * The singleton instance of the meta-bean.
     */
    static final Meta INSTANCE = new Meta();

    /**
     * The meta-property for the {@code seasonalityMonthOnMonth} property.
     */
    private final MetaProperty<DoubleArray> seasonalityMonthOnMonth = DirectMetaProperty.ofImmutable(
        this, "seasonalityMonthOnMonth", SeasonalityDefinition.class, DoubleArray.class);
    /**
     * The meta-property for the {@code adjustmentType} property.
     */
    private final MetaProperty<ShiftType> adjustmentType = DirectMetaProperty.ofImmutable(
        this, "adjustmentType", SeasonalityDefinition.class, ShiftType.class);
    /**
     * The meta-properties.
     */
    private final Map<String, MetaProperty<?>> metaPropertyMap$ = new DirectMetaPropertyMap(
        this, null,
        "seasonalityMonthOnMonth",
        "adjustmentType");

    /**
     * Restricted constructor.
     */
    private Meta() {
    }

    @Override
    protected MetaProperty<?> metaPropertyGet(String propertyName) {
      switch (propertyName.hashCode()) {
        case -731183871:  // seasonalityMonthOnMonth
          return seasonalityMonthOnMonth;
        case -1002343865:  // adjustmentType
          return adjustmentType;
      }
      return super.metaPropertyGet(propertyName);
    }

    @Override
    public BeanBuilder<? extends SeasonalityDefinition> builder() {
      return new SeasonalityDefinition.Builder();
    }

    @Override
    public Class<? extends SeasonalityDefinition> beanType() {
      return SeasonalityDefinition.class;
    }

    @Override
    public Map<String, MetaProperty<?>> metaPropertyMap() {
      return metaPropertyMap$;
    }

    //-----------------------------------------------------------------------
    /**
     * The meta-property for the {@code seasonalityMonthOnMonth} property.
     * @return the meta-property, not null
     */
    public MetaProperty<DoubleArray> seasonalityMonthOnMonth() {
      return seasonalityMonthOnMonth;
    }

    /**
     * The meta-property for the {@code adjustmentType} property.
     * @return the meta-property, not null
     */
    public MetaProperty<ShiftType> adjustmentType() {
      return adjustmentType;
    }

    //-----------------------------------------------------------------------
    @Override
    protected Object propertyGet(Bean bean, String propertyName, boolean quiet) {
      switch (propertyName.hashCode()) {
        case -731183871:  // seasonalityMonthOnMonth
          return ((SeasonalityDefinition) bean).getSeasonalityMonthOnMonth();
        case -1002343865:  // adjustmentType
          return ((SeasonalityDefinition) bean).getAdjustmentType();
      }
      return super.propertyGet(bean, propertyName, quiet);
    }

    @Override
    protected void propertySet(Bean bean, String propertyName, Object newValue, boolean quiet) {
      metaProperty(propertyName);
      if (quiet) {
        return;
      }
      throw new UnsupportedOperationException("Property cannot be written: " + propertyName);
    }

  }

  //-----------------------------------------------------------------------
  /**
   * The bean-builder for {@code SeasonalityDefinition}.
   */
  private static final class Builder extends DirectPrivateBeanBuilder<SeasonalityDefinition> {

    private DoubleArray seasonalityMonthOnMonth;
    private ShiftType adjustmentType;

    /**
     * Restricted constructor.
     */
    private Builder() {
    }

    //-----------------------------------------------------------------------
    @Override
    public Object get(String propertyName) {
      switch (propertyName.hashCode()) {
        case -731183871:  // seasonalityMonthOnMonth
          return seasonalityMonthOnMonth;
        case -1002343865:  // adjustmentType
          return adjustmentType;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
    }

    @Override
    public Builder set(String propertyName, Object newValue) {
      switch (propertyName.hashCode()) {
        case -731183871:  // seasonalityMonthOnMonth
          this.seasonalityMonthOnMonth = (DoubleArray) newValue;
          break;
        case -1002343865:  // adjustmentType
          this.adjustmentType = (ShiftType) newValue;
          break;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
      return this;
    }

    @Override
    public SeasonalityDefinition build() {
      return new SeasonalityDefinition(
          seasonalityMonthOnMonth,
          adjustmentType);
    }

    //-----------------------------------------------------------------------
    @Override
    public String toString() {
      StringBuilder buf = new StringBuilder(96);
      buf.append("SeasonalityDefinition.Builder{");
      buf.append("seasonalityMonthOnMonth").append('=').append(JodaBeanUtils.toString(seasonalityMonthOnMonth)).append(',').append(' ');
      buf.append("adjustmentType").append('=').append(JodaBeanUtils.toString(adjustmentType));
      buf.append('}');
      return buf.toString();
    }

  }

  //-------------------------- AUTOGENERATED END --------------------------
}
