/*
 * Copyright (C) 2015 - present by OpenGamma Inc. and the OpenGamma group of companies
 *
 * Please see distribution for license.
 */
package com.opengamma.strata.market.observable;

import java.io.Serializable;
import java.util.Map;
import java.util.NoSuchElementException;

import org.joda.beans.Bean;
import org.joda.beans.BeanBuilder;
import org.joda.beans.ImmutableBean;
import org.joda.beans.JodaBeanUtils;
import org.joda.beans.MetaBean;
import org.joda.beans.MetaProperty;
import org.joda.beans.gen.BeanDefinition;
import org.joda.beans.gen.PropertyDefinition;
import org.joda.beans.impl.direct.DirectMetaBean;
import org.joda.beans.impl.direct.DirectMetaProperty;
import org.joda.beans.impl.direct.DirectMetaPropertyMap;
import org.joda.beans.impl.direct.DirectPrivateBeanBuilder;

import com.opengamma.strata.basics.StandardId;
import com.opengamma.strata.collect.array.DoubleArray;
import com.opengamma.strata.data.FieldName;
import com.opengamma.strata.data.scenario.MarketDataBox;
import com.opengamma.strata.data.scenario.ScenarioMarketDataId;

/**
 * An identifier identifying a {@link QuoteScenarioArray} containing values for a piece
 * of quoted market data in multiple scenarios.
 */
@BeanDefinition(builderScope = "private", cacheHashCode = true)
public final class QuoteScenarioArrayId
    implements ScenarioMarketDataId<Double, QuoteScenarioArray>, ImmutableBean, Serializable {

  /** The market data key identifying the quote. */
  @PropertyDefinition(validate = "notNull")
  private final QuoteId id;

  /**
   * Returns a key identifying the market data with the specified ID and field name.
   *
   * @param id  the ID of the quote
   * @param fieldName  the field name of the market data record containing the quote data
   * @return a key identifying the market data with the specified ID and field name
   */
  public static QuoteScenarioArrayId of(StandardId id, FieldName fieldName) {
    return new QuoteScenarioArrayId(QuoteId.of(id, fieldName));
  }

  /**
   * Returns a key identifying the same market data as the quote key.
   *
   * @param quoteId  key identifying the quote
   * @return a key identifying the same market data as the quote key
   */
  public static QuoteScenarioArrayId of(QuoteId quoteId) {
    return new QuoteScenarioArrayId(quoteId);
  }

  @Override
  public QuoteId getMarketDataId() {
    return id;
  }

  @Override
  public Class<QuoteScenarioArray> getScenarioMarketDataType() {
    return QuoteScenarioArray.class;
  }

  @Override
  public QuoteScenarioArray createScenarioValue(MarketDataBox<Double> marketDataBox, int scenarioCount) {
    return QuoteScenarioArray.of(DoubleArray.of(scenarioCount, i -> marketDataBox.getValue(i)));
  }

  //------------------------- AUTOGENERATED START -------------------------
  /**
   * The meta-bean for {@code QuoteScenarioArrayId}.
   * @return the meta-bean, not null
   */
  public static QuoteScenarioArrayId.Meta meta() {
    return QuoteScenarioArrayId.Meta.INSTANCE;
  }

  static {
    MetaBean.register(QuoteScenarioArrayId.Meta.INSTANCE);
  }

  /**
   * The serialization version id.
   */
  private static final long serialVersionUID = 1L;

  /**
   * The cached hash code, using the racy single-check idiom.
   */
  private transient int cacheHashCode;

  private QuoteScenarioArrayId(
      QuoteId id) {
    JodaBeanUtils.notNull(id, "id");
    this.id = id;
  }

  @Override
  public QuoteScenarioArrayId.Meta metaBean() {
    return QuoteScenarioArrayId.Meta.INSTANCE;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the market data key identifying the quote.
   * @return the value of the property, not null
   */
  public QuoteId getId() {
    return id;
  }

  //-----------------------------------------------------------------------
  @Override
  public boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (obj != null && obj.getClass() == this.getClass()) {
      QuoteScenarioArrayId other = (QuoteScenarioArrayId) obj;
      return JodaBeanUtils.equal(id, other.id);
    }
    return false;
  }

  @Override
  public int hashCode() {
    int hash = cacheHashCode;
    if (hash == 0) {
      hash = getClass().hashCode();
      hash = hash * 31 + JodaBeanUtils.hashCode(id);
      cacheHashCode = hash;
    }
    return hash;
  }

  @Override
  public String toString() {
    StringBuilder buf = new StringBuilder(64);
    buf.append("QuoteScenarioArrayId{");
    buf.append("id").append('=').append(JodaBeanUtils.toString(id));
    buf.append('}');
    return buf.toString();
  }

  //-----------------------------------------------------------------------
  /**
   * The meta-bean for {@code QuoteScenarioArrayId}.
   */
  public static final class Meta extends DirectMetaBean {
    /**
     * The singleton instance of the meta-bean.
     */
    static final Meta INSTANCE = new Meta();

    /**
     * The meta-property for the {@code id} property.
     */
    private final MetaProperty<QuoteId> id = DirectMetaProperty.ofImmutable(
        this, "id", QuoteScenarioArrayId.class, QuoteId.class);
    /**
     * The meta-properties.
     */
    private final Map<String, MetaProperty<?>> metaPropertyMap$ = new DirectMetaPropertyMap(
        this, null,
        "id");

    /**
     * Restricted constructor.
     */
    private Meta() {
    }

    @Override
    protected MetaProperty<?> metaPropertyGet(String propertyName) {
      switch (propertyName.hashCode()) {
        case 3355:  // id
          return id;
      }
      return super.metaPropertyGet(propertyName);
    }

    @Override
    public BeanBuilder<? extends QuoteScenarioArrayId> builder() {
      return new QuoteScenarioArrayId.Builder();
    }

    @Override
    public Class<? extends QuoteScenarioArrayId> beanType() {
      return QuoteScenarioArrayId.class;
    }

    @Override
    public Map<String, MetaProperty<?>> metaPropertyMap() {
      return metaPropertyMap$;
    }

    //-----------------------------------------------------------------------
    /**
     * The meta-property for the {@code id} property.
     * @return the meta-property, not null
     */
    public MetaProperty<QuoteId> id() {
      return id;
    }

    //-----------------------------------------------------------------------
    @Override
    protected Object propertyGet(Bean bean, String propertyName, boolean quiet) {
      switch (propertyName.hashCode()) {
        case 3355:  // id
          return ((QuoteScenarioArrayId) bean).getId();
      }
      return super.propertyGet(bean, propertyName, quiet);
    }

    @Override
    protected void propertySet(Bean bean, String propertyName, Object newValue, boolean quiet) {
      metaProperty(propertyName);
      if (quiet) {
        return;
      }
      throw new UnsupportedOperationException("Property cannot be written: " + propertyName);
    }

  }

  //-----------------------------------------------------------------------
  /**
   * The bean-builder for {@code QuoteScenarioArrayId}.
   */
  private static final class Builder extends DirectPrivateBeanBuilder<QuoteScenarioArrayId> {

    private QuoteId id;

    /**
     * Restricted constructor.
     */
    private Builder() {
    }

    //-----------------------------------------------------------------------
    @Override
    public Object get(String propertyName) {
      switch (propertyName.hashCode()) {
        case 3355:  // id
          return id;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
    }

    @Override
    public Builder set(String propertyName, Object newValue) {
      switch (propertyName.hashCode()) {
        case 3355:  // id
          this.id = (QuoteId) newValue;
          break;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
      return this;
    }

    @Override
    public QuoteScenarioArrayId build() {
      return new QuoteScenarioArrayId(
          id);
    }

    //-----------------------------------------------------------------------
    @Override
    public String toString() {
      StringBuilder buf = new StringBuilder(64);
      buf.append("QuoteScenarioArrayId.Builder{");
      buf.append("id").append('=').append(JodaBeanUtils.toString(id));
      buf.append('}');
      return buf.toString();
    }

  }

  //-------------------------- AUTOGENERATED END --------------------------
}
