/*
 * Copyright (C) 2024 - present by OpenGamma Inc. and the OpenGamma group of companies
 *
 * Please see distribution for license.
 */
package com.opengamma.strata.market.param;

import java.io.Serializable;
import java.util.Map;
import java.util.NoSuchElementException;

import org.joda.beans.Bean;
import org.joda.beans.BeanBuilder;
import org.joda.beans.ImmutableBean;
import org.joda.beans.JodaBeanUtils;
import org.joda.beans.MetaBean;
import org.joda.beans.MetaProperty;
import org.joda.beans.gen.BeanDefinition;
import org.joda.beans.gen.ImmutablePreBuild;
import org.joda.beans.gen.PropertyDefinition;
import org.joda.beans.impl.direct.DirectMetaBean;
import org.joda.beans.impl.direct.DirectMetaProperty;
import org.joda.beans.impl.direct.DirectMetaPropertyMap;
import org.joda.beans.impl.direct.DirectPrivateBeanBuilder;

import com.opengamma.strata.basics.date.Tenor;
import com.opengamma.strata.collect.tuple.Triple;

/**
 * Parameter metadata based on an expiry tenor, an underlying tenor and strike value.
 */
@BeanDefinition(builderScope = "private")
public final class TenorTenorStrikeParameterMetadata
    implements ParameterMetadata, ImmutableBean, Serializable {

  /**
   * The expiry tenor associated with the parameter.
   */
  @PropertyDefinition(validate = "notNull")
  private final Tenor expiryTenor;
  /**
   * The underlying tenor associated with the parameter.
   */
  @PropertyDefinition(validate = "notNull")
  private final Tenor underlyingTenor;
  /**
   * The strike value associated with the parameter.
   */
  @PropertyDefinition
  private final double strike;
  /**
   * The label that describes the parameter, defaulted to the both tenors and strike.
   */
  @PropertyDefinition(validate = "notEmpty", overrideGet = true)
  private final String label;

  //-------------------------------------------------------------------------

  /**
   * Creates node metadata with expiry tenor, underlying tenor and strike.
   *
   * @param expiryTenor the expiry tenor
   * @param underlyingTenor the underlying
   * @param strike the strike
   * @return the metadata
   */
  public static TenorTenorStrikeParameterMetadata of(Tenor expiryTenor, Tenor underlyingTenor, double strike) {
    String label = Triple.of(expiryTenor, underlyingTenor, strike).toString();
    return new TenorTenorStrikeParameterMetadata(expiryTenor, underlyingTenor, strike, label);
  }

  /**
   * Creates node metadata with expiry tenor, underlying tenor, strike and label.
   *
   * @param expiryTenor the expiry tenor
   * @param underlyingTenor the underlying
   * @param strike the strike
   * @param label the label
   * @return the metadata
   */
  public static TenorTenorStrikeParameterMetadata of(
      Tenor expiryTenor,
      Tenor underlyingTenor,
      double strike,
      String label) {

    return new TenorTenorStrikeParameterMetadata(expiryTenor, underlyingTenor, strike, label);
  }

  @ImmutablePreBuild
  private static void preBuild(Builder builder) {
    if (builder.label == null) {
      builder.label = Triple.of(builder.expiryTenor, builder.underlyingTenor, builder.strike).toString();
    }
  }

  @Override
  public Triple<Tenor, Tenor, Double> getIdentifier() {
    return Triple.of(expiryTenor, underlyingTenor, strike);
  }

  //------------------------- AUTOGENERATED START -------------------------
  /**
   * The meta-bean for {@code TenorTenorStrikeParameterMetadata}.
   * @return the meta-bean, not null
   */
  public static TenorTenorStrikeParameterMetadata.Meta meta() {
    return TenorTenorStrikeParameterMetadata.Meta.INSTANCE;
  }

  static {
    MetaBean.register(TenorTenorStrikeParameterMetadata.Meta.INSTANCE);
  }

  /**
   * The serialization version id.
   */
  private static final long serialVersionUID = 1L;

  private TenorTenorStrikeParameterMetadata(
      Tenor expiryTenor,
      Tenor underlyingTenor,
      double strike,
      String label) {
    JodaBeanUtils.notNull(expiryTenor, "expiryTenor");
    JodaBeanUtils.notNull(underlyingTenor, "underlyingTenor");
    JodaBeanUtils.notEmpty(label, "label");
    this.expiryTenor = expiryTenor;
    this.underlyingTenor = underlyingTenor;
    this.strike = strike;
    this.label = label;
  }

  @Override
  public TenorTenorStrikeParameterMetadata.Meta metaBean() {
    return TenorTenorStrikeParameterMetadata.Meta.INSTANCE;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the expiry tenor associated with the parameter.
   * @return the value of the property, not null
   */
  public Tenor getExpiryTenor() {
    return expiryTenor;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the underlying tenor associated with the parameter.
   * @return the value of the property, not null
   */
  public Tenor getUnderlyingTenor() {
    return underlyingTenor;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the strike value associated with the parameter.
   * @return the value of the property
   */
  public double getStrike() {
    return strike;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the label that describes the parameter, defaulted to the both tenors and strike.
   * @return the value of the property, not empty
   */
  @Override
  public String getLabel() {
    return label;
  }

  //-----------------------------------------------------------------------
  @Override
  public boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (obj != null && obj.getClass() == this.getClass()) {
      TenorTenorStrikeParameterMetadata other = (TenorTenorStrikeParameterMetadata) obj;
      return JodaBeanUtils.equal(expiryTenor, other.expiryTenor) &&
          JodaBeanUtils.equal(underlyingTenor, other.underlyingTenor) &&
          JodaBeanUtils.equal(strike, other.strike) &&
          JodaBeanUtils.equal(label, other.label);
    }
    return false;
  }

  @Override
  public int hashCode() {
    int hash = getClass().hashCode();
    hash = hash * 31 + JodaBeanUtils.hashCode(expiryTenor);
    hash = hash * 31 + JodaBeanUtils.hashCode(underlyingTenor);
    hash = hash * 31 + JodaBeanUtils.hashCode(strike);
    hash = hash * 31 + JodaBeanUtils.hashCode(label);
    return hash;
  }

  @Override
  public String toString() {
    StringBuilder buf = new StringBuilder(160);
    buf.append("TenorTenorStrikeParameterMetadata{");
    buf.append("expiryTenor").append('=').append(JodaBeanUtils.toString(expiryTenor)).append(',').append(' ');
    buf.append("underlyingTenor").append('=').append(JodaBeanUtils.toString(underlyingTenor)).append(',').append(' ');
    buf.append("strike").append('=').append(JodaBeanUtils.toString(strike)).append(',').append(' ');
    buf.append("label").append('=').append(JodaBeanUtils.toString(label));
    buf.append('}');
    return buf.toString();
  }

  //-----------------------------------------------------------------------
  /**
   * The meta-bean for {@code TenorTenorStrikeParameterMetadata}.
   */
  public static final class Meta extends DirectMetaBean {
    /**
     * The singleton instance of the meta-bean.
     */
    static final Meta INSTANCE = new Meta();

    /**
     * The meta-property for the {@code expiryTenor} property.
     */
    private final MetaProperty<Tenor> expiryTenor = DirectMetaProperty.ofImmutable(
        this, "expiryTenor", TenorTenorStrikeParameterMetadata.class, Tenor.class);
    /**
     * The meta-property for the {@code underlyingTenor} property.
     */
    private final MetaProperty<Tenor> underlyingTenor = DirectMetaProperty.ofImmutable(
        this, "underlyingTenor", TenorTenorStrikeParameterMetadata.class, Tenor.class);
    /**
     * The meta-property for the {@code strike} property.
     */
    private final MetaProperty<Double> strike = DirectMetaProperty.ofImmutable(
        this, "strike", TenorTenorStrikeParameterMetadata.class, Double.TYPE);
    /**
     * The meta-property for the {@code label} property.
     */
    private final MetaProperty<String> label = DirectMetaProperty.ofImmutable(
        this, "label", TenorTenorStrikeParameterMetadata.class, String.class);
    /**
     * The meta-properties.
     */
    private final Map<String, MetaProperty<?>> metaPropertyMap$ = new DirectMetaPropertyMap(
        this, null,
        "expiryTenor",
        "underlyingTenor",
        "strike",
        "label");

    /**
     * Restricted constructor.
     */
    private Meta() {
    }

    @Override
    protected MetaProperty<?> metaPropertyGet(String propertyName) {
      switch (propertyName.hashCode()) {
        case 465802573:  // expiryTenor
          return expiryTenor;
        case -824175261:  // underlyingTenor
          return underlyingTenor;
        case -891985998:  // strike
          return strike;
        case 102727412:  // label
          return label;
      }
      return super.metaPropertyGet(propertyName);
    }

    @Override
    public BeanBuilder<? extends TenorTenorStrikeParameterMetadata> builder() {
      return new TenorTenorStrikeParameterMetadata.Builder();
    }

    @Override
    public Class<? extends TenorTenorStrikeParameterMetadata> beanType() {
      return TenorTenorStrikeParameterMetadata.class;
    }

    @Override
    public Map<String, MetaProperty<?>> metaPropertyMap() {
      return metaPropertyMap$;
    }

    //-----------------------------------------------------------------------
    /**
     * The meta-property for the {@code expiryTenor} property.
     * @return the meta-property, not null
     */
    public MetaProperty<Tenor> expiryTenor() {
      return expiryTenor;
    }

    /**
     * The meta-property for the {@code underlyingTenor} property.
     * @return the meta-property, not null
     */
    public MetaProperty<Tenor> underlyingTenor() {
      return underlyingTenor;
    }

    /**
     * The meta-property for the {@code strike} property.
     * @return the meta-property, not null
     */
    public MetaProperty<Double> strike() {
      return strike;
    }

    /**
     * The meta-property for the {@code label} property.
     * @return the meta-property, not null
     */
    public MetaProperty<String> label() {
      return label;
    }

    //-----------------------------------------------------------------------
    @Override
    protected Object propertyGet(Bean bean, String propertyName, boolean quiet) {
      switch (propertyName.hashCode()) {
        case 465802573:  // expiryTenor
          return ((TenorTenorStrikeParameterMetadata) bean).getExpiryTenor();
        case -824175261:  // underlyingTenor
          return ((TenorTenorStrikeParameterMetadata) bean).getUnderlyingTenor();
        case -891985998:  // strike
          return ((TenorTenorStrikeParameterMetadata) bean).getStrike();
        case 102727412:  // label
          return ((TenorTenorStrikeParameterMetadata) bean).getLabel();
      }
      return super.propertyGet(bean, propertyName, quiet);
    }

    @Override
    protected void propertySet(Bean bean, String propertyName, Object newValue, boolean quiet) {
      metaProperty(propertyName);
      if (quiet) {
        return;
      }
      throw new UnsupportedOperationException("Property cannot be written: " + propertyName);
    }

  }

  //-----------------------------------------------------------------------
  /**
   * The bean-builder for {@code TenorTenorStrikeParameterMetadata}.
   */
  private static final class Builder extends DirectPrivateBeanBuilder<TenorTenorStrikeParameterMetadata> {

    private Tenor expiryTenor;
    private Tenor underlyingTenor;
    private double strike;
    private String label;

    /**
     * Restricted constructor.
     */
    private Builder() {
    }

    //-----------------------------------------------------------------------
    @Override
    public Object get(String propertyName) {
      switch (propertyName.hashCode()) {
        case 465802573:  // expiryTenor
          return expiryTenor;
        case -824175261:  // underlyingTenor
          return underlyingTenor;
        case -891985998:  // strike
          return strike;
        case 102727412:  // label
          return label;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
    }

    @Override
    public Builder set(String propertyName, Object newValue) {
      switch (propertyName.hashCode()) {
        case 465802573:  // expiryTenor
          this.expiryTenor = (Tenor) newValue;
          break;
        case -824175261:  // underlyingTenor
          this.underlyingTenor = (Tenor) newValue;
          break;
        case -891985998:  // strike
          this.strike = (Double) newValue;
          break;
        case 102727412:  // label
          this.label = (String) newValue;
          break;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
      return this;
    }

    @Override
    public TenorTenorStrikeParameterMetadata build() {
      preBuild(this);
      return new TenorTenorStrikeParameterMetadata(
          expiryTenor,
          underlyingTenor,
          strike,
          label);
    }

    //-----------------------------------------------------------------------
    @Override
    public String toString() {
      StringBuilder buf = new StringBuilder(160);
      buf.append("TenorTenorStrikeParameterMetadata.Builder{");
      buf.append("expiryTenor").append('=').append(JodaBeanUtils.toString(expiryTenor)).append(',').append(' ');
      buf.append("underlyingTenor").append('=').append(JodaBeanUtils.toString(underlyingTenor)).append(',').append(' ');
      buf.append("strike").append('=').append(JodaBeanUtils.toString(strike)).append(',').append(' ');
      buf.append("label").append('=').append(JodaBeanUtils.toString(label));
      buf.append('}');
      return buf.toString();
    }

  }

  //-------------------------- AUTOGENERATED END --------------------------
}
