/*
 * Copyright (C) 2016 - present by OpenGamma Inc. and the OpenGamma group of companies
 *
 * Please see distribution for license.
 */
package com.opengamma.strata.product;

import java.io.Serializable;
import java.util.Map;
import java.util.NoSuchElementException;

import org.joda.beans.Bean;
import org.joda.beans.BeanBuilder;
import org.joda.beans.ImmutableBean;
import org.joda.beans.JodaBeanUtils;
import org.joda.beans.MetaBean;
import org.joda.beans.MetaProperty;
import org.joda.beans.gen.BeanDefinition;
import org.joda.beans.gen.PropertyDefinition;
import org.joda.beans.impl.direct.DirectMetaBean;
import org.joda.beans.impl.direct.DirectMetaProperty;
import org.joda.beans.impl.direct.DirectMetaPropertyMap;
import org.joda.beans.impl.direct.DirectPrivateBeanBuilder;

import com.google.common.collect.ImmutableSet;
import com.opengamma.strata.basics.ReferenceData;
import com.opengamma.strata.basics.currency.Currency;
import com.opengamma.strata.product.bond.BondFutureOptionSecurity;
import com.opengamma.strata.product.bond.FixedCouponBondSecurity;
import com.opengamma.strata.product.index.IborFutureSecurity;

/**
 * A generic security, defined in terms of the value of each tick.
 * <p>
 * In most cases, applications will choose to represent information about securities
 * using the relevant type, such as {@link FixedCouponBondSecurity},
 * {@link BondFutureOptionSecurity} or {@link IborFutureSecurity}.
 * Sometimes however, it can be useful to store minimal information about the security,
 * expressing just the tick size and tick value.
 */
@BeanDefinition(builderScope = "private")
public final class GenericSecurity
    implements Security, SecuritizedProduct, ImmutableBean, Serializable {

  /**
   * The standard security information.
   * <p>
   * This includes the security identifier.
   */
  @PropertyDefinition(validate = "notNull", overrideGet = true)
  private final SecurityInfo info;

  //-------------------------------------------------------------------------
  /**
   * Obtains an instance from security information, tick size and tick value.
   * 
   * @param securityInfo  the security information
   * @return the security
   */
  public static GenericSecurity of(SecurityInfo securityInfo) {
    return new GenericSecurity(securityInfo);
  }

  //-------------------------------------------------------------------------
  @Override
  public SecurityId getSecurityId() {
    return Security.super.getSecurityId();
  }

  @Override
  public Currency getCurrency() {
    return Security.super.getCurrency();
  }

  @Override
  public ImmutableSet<SecurityId> getUnderlyingIds() {
    return ImmutableSet.of();
  }

  //-------------------------------------------------------------------------
  @Override
  public GenericSecurity withInfo(SecurityInfo info) {
    return new GenericSecurity(info);
  }

  //-------------------------------------------------------------------------
  /**
   * Creates the associated product, which simply returns {@code this}.
   * <p>
   * The product associated with a security normally returns the financial model used for pricing.
   * In the case of a {@code GenericSecurity}, no underlying financial model is available.
   * As such, the {@code GenericSecurity} is the product.
   * 
   * @param refData  the reference data to use
   * @return this security
   */
  @Override
  public GenericSecurity createProduct(ReferenceData refData) {
    return this;
  }

  @Override
  public GenericSecurityTrade createTrade(TradeInfo info, double quantity, double tradePrice, ReferenceData refData) {
    return new GenericSecurityTrade(info, this, quantity, tradePrice);
  }

  @Override
  public GenericSecurityPosition createPosition(PositionInfo tradeInfo, double quantity, ReferenceData refData) {
    return GenericSecurityPosition.ofNet(tradeInfo, this, quantity);
  }

  @Override
  public GenericSecurityPosition createPosition(
      PositionInfo positionInfo,
      double longQuantity,
      double shortQuantity,
      ReferenceData refData) {

    return GenericSecurityPosition.ofLongShort(positionInfo, this, longQuantity, shortQuantity);
  }

  //------------------------- AUTOGENERATED START -------------------------
  /**
   * The meta-bean for {@code GenericSecurity}.
   * @return the meta-bean, not null
   */
  public static GenericSecurity.Meta meta() {
    return GenericSecurity.Meta.INSTANCE;
  }

  static {
    MetaBean.register(GenericSecurity.Meta.INSTANCE);
  }

  /**
   * The serialization version id.
   */
  private static final long serialVersionUID = 1L;

  private GenericSecurity(
      SecurityInfo info) {
    JodaBeanUtils.notNull(info, "info");
    this.info = info;
  }

  @Override
  public GenericSecurity.Meta metaBean() {
    return GenericSecurity.Meta.INSTANCE;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the standard security information.
   * <p>
   * This includes the security identifier.
   * @return the value of the property, not null
   */
  @Override
  public SecurityInfo getInfo() {
    return info;
  }

  //-----------------------------------------------------------------------
  @Override
  public boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (obj != null && obj.getClass() == this.getClass()) {
      GenericSecurity other = (GenericSecurity) obj;
      return JodaBeanUtils.equal(info, other.info);
    }
    return false;
  }

  @Override
  public int hashCode() {
    int hash = getClass().hashCode();
    hash = hash * 31 + JodaBeanUtils.hashCode(info);
    return hash;
  }

  @Override
  public String toString() {
    StringBuilder buf = new StringBuilder(64);
    buf.append("GenericSecurity{");
    buf.append("info").append('=').append(JodaBeanUtils.toString(info));
    buf.append('}');
    return buf.toString();
  }

  //-----------------------------------------------------------------------
  /**
   * The meta-bean for {@code GenericSecurity}.
   */
  public static final class Meta extends DirectMetaBean {
    /**
     * The singleton instance of the meta-bean.
     */
    static final Meta INSTANCE = new Meta();

    /**
     * The meta-property for the {@code info} property.
     */
    private final MetaProperty<SecurityInfo> info = DirectMetaProperty.ofImmutable(
        this, "info", GenericSecurity.class, SecurityInfo.class);
    /**
     * The meta-properties.
     */
    private final Map<String, MetaProperty<?>> metaPropertyMap$ = new DirectMetaPropertyMap(
        this, null,
        "info");

    /**
     * Restricted constructor.
     */
    private Meta() {
    }

    @Override
    protected MetaProperty<?> metaPropertyGet(String propertyName) {
      switch (propertyName.hashCode()) {
        case 3237038:  // info
          return info;
      }
      return super.metaPropertyGet(propertyName);
    }

    @Override
    public BeanBuilder<? extends GenericSecurity> builder() {
      return new GenericSecurity.Builder();
    }

    @Override
    public Class<? extends GenericSecurity> beanType() {
      return GenericSecurity.class;
    }

    @Override
    public Map<String, MetaProperty<?>> metaPropertyMap() {
      return metaPropertyMap$;
    }

    //-----------------------------------------------------------------------
    /**
     * The meta-property for the {@code info} property.
     * @return the meta-property, not null
     */
    public MetaProperty<SecurityInfo> info() {
      return info;
    }

    //-----------------------------------------------------------------------
    @Override
    protected Object propertyGet(Bean bean, String propertyName, boolean quiet) {
      switch (propertyName.hashCode()) {
        case 3237038:  // info
          return ((GenericSecurity) bean).getInfo();
      }
      return super.propertyGet(bean, propertyName, quiet);
    }

    @Override
    protected void propertySet(Bean bean, String propertyName, Object newValue, boolean quiet) {
      metaProperty(propertyName);
      if (quiet) {
        return;
      }
      throw new UnsupportedOperationException("Property cannot be written: " + propertyName);
    }

  }

  //-----------------------------------------------------------------------
  /**
   * The bean-builder for {@code GenericSecurity}.
   */
  private static final class Builder extends DirectPrivateBeanBuilder<GenericSecurity> {

    private SecurityInfo info;

    /**
     * Restricted constructor.
     */
    private Builder() {
    }

    //-----------------------------------------------------------------------
    @Override
    public Object get(String propertyName) {
      switch (propertyName.hashCode()) {
        case 3237038:  // info
          return info;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
    }

    @Override
    public Builder set(String propertyName, Object newValue) {
      switch (propertyName.hashCode()) {
        case 3237038:  // info
          this.info = (SecurityInfo) newValue;
          break;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
      return this;
    }

    @Override
    public GenericSecurity build() {
      return new GenericSecurity(
          info);
    }

    //-----------------------------------------------------------------------
    @Override
    public String toString() {
      StringBuilder buf = new StringBuilder(64);
      buf.append("GenericSecurity.Builder{");
      buf.append("info").append('=').append(JodaBeanUtils.toString(info));
      buf.append('}');
      return buf.toString();
    }

  }

  //-------------------------- AUTOGENERATED END --------------------------
}
