/*
 * Copyright (C) 2022 - present by OpenGamma Inc. and the OpenGamma group of companies
 *
 * Please see distribution for license.
 */
package com.opengamma.strata.product.bond;

import java.io.Serializable;
import java.time.ZonedDateTime;
import java.util.Map;
import java.util.NoSuchElementException;

import org.joda.beans.Bean;
import org.joda.beans.ImmutableBean;
import org.joda.beans.JodaBeanUtils;
import org.joda.beans.MetaBean;
import org.joda.beans.MetaProperty;
import org.joda.beans.gen.BeanDefinition;
import org.joda.beans.gen.ImmutableValidator;
import org.joda.beans.gen.PropertyDefinition;
import org.joda.beans.impl.direct.DirectFieldsBeanBuilder;
import org.joda.beans.impl.direct.DirectMetaBean;
import org.joda.beans.impl.direct.DirectMetaProperty;
import org.joda.beans.impl.direct.DirectMetaPropertyMap;

import com.opengamma.strata.basics.currency.Currency;
import com.opengamma.strata.collect.ArgChecker;
import com.opengamma.strata.product.ResolvedProduct;
import com.opengamma.strata.product.common.LongShort;

/**
 * An option on a {@link FixedCouponBond} resolved for pricing.
 * <p>
 * The option strike is expressed as "Clean price". The "clean" price excludes any accrued interest.
 * <p>
 * The call/put is provided by the quantity's sign. If positive, it indicates the right to buy the bond (call), 
 * if negative it indicates the right to sell the bond (put).
 */
@BeanDefinition
public final class ResolvedFixedCouponBondOption
    implements ResolvedProduct, ImmutableBean, Serializable {

  /**
   * Whether the option is long or short.
   * <p>
   * Long indicates that the owner has the right to exercise the option at expiry.
   */
  @PropertyDefinition(validate = "notNull")
  private final LongShort longShort;
  /**
   * The bond underlying the option.
   */
  @PropertyDefinition(validate = "notNull")
  private final ResolvedFixedCouponBond underlying;
  /**
   * The expiry date-time of the option.
   * <p>
   * The option is European, and can only be exercised on the expiry date.
   */
  @PropertyDefinition(validate = "notNull")
  private final ZonedDateTime expiry;
  /**
   * The quantity that was traded.
   * <p>
   * This will be positive if buying (call) and negative if selling (put).
   */
  @PropertyDefinition
  private final double quantity;
  /**
   * The bond's settlement details.
   */
  @PropertyDefinition(validate = "notNull")
  private final ResolvedFixedCouponBondSettlement settlement;

  @ImmutableValidator
  private void validate() {
    ArgChecker.inOrderOrEqual(expiry.toLocalDate(), settlement.getSettlementDate(),
        "expiry date", "settlement date");
  }
  
  /**
   * Returns the bond option currency.
   * 
   * @return the currency
   */
  public Currency getCurrency() {
    return underlying.getCurrency();
  }

  //------------------------- AUTOGENERATED START -------------------------
  /**
   * The meta-bean for {@code ResolvedFixedCouponBondOption}.
   * @return the meta-bean, not null
   */
  public static ResolvedFixedCouponBondOption.Meta meta() {
    return ResolvedFixedCouponBondOption.Meta.INSTANCE;
  }

  static {
    MetaBean.register(ResolvedFixedCouponBondOption.Meta.INSTANCE);
  }

  /**
   * The serialization version id.
   */
  private static final long serialVersionUID = 1L;

  /**
   * Returns a builder used to create an instance of the bean.
   * @return the builder, not null
   */
  public static ResolvedFixedCouponBondOption.Builder builder() {
    return new ResolvedFixedCouponBondOption.Builder();
  }

  private ResolvedFixedCouponBondOption(
      LongShort longShort,
      ResolvedFixedCouponBond underlying,
      ZonedDateTime expiry,
      double quantity,
      ResolvedFixedCouponBondSettlement settlement) {
    JodaBeanUtils.notNull(longShort, "longShort");
    JodaBeanUtils.notNull(underlying, "underlying");
    JodaBeanUtils.notNull(expiry, "expiry");
    JodaBeanUtils.notNull(settlement, "settlement");
    this.longShort = longShort;
    this.underlying = underlying;
    this.expiry = expiry;
    this.quantity = quantity;
    this.settlement = settlement;
    validate();
  }

  @Override
  public ResolvedFixedCouponBondOption.Meta metaBean() {
    return ResolvedFixedCouponBondOption.Meta.INSTANCE;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets whether the option is long or short.
   * <p>
   * Long indicates that the owner has the right to exercise the option at expiry.
   * @return the value of the property, not null
   */
  public LongShort getLongShort() {
    return longShort;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the bond underlying the option.
   * @return the value of the property, not null
   */
  public ResolvedFixedCouponBond getUnderlying() {
    return underlying;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the expiry date-time of the option.
   * <p>
   * The option is European, and can only be exercised on the expiry date.
   * @return the value of the property, not null
   */
  public ZonedDateTime getExpiry() {
    return expiry;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the quantity that was traded.
   * <p>
   * This will be positive if buying (call) and negative if selling (put).
   * @return the value of the property
   */
  public double getQuantity() {
    return quantity;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the bond's settlement details.
   * @return the value of the property, not null
   */
  public ResolvedFixedCouponBondSettlement getSettlement() {
    return settlement;
  }

  //-----------------------------------------------------------------------
  /**
   * Returns a builder that allows this bean to be mutated.
   * @return the mutable builder, not null
   */
  public Builder toBuilder() {
    return new Builder(this);
  }

  @Override
  public boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (obj != null && obj.getClass() == this.getClass()) {
      ResolvedFixedCouponBondOption other = (ResolvedFixedCouponBondOption) obj;
      return JodaBeanUtils.equal(longShort, other.longShort) &&
          JodaBeanUtils.equal(underlying, other.underlying) &&
          JodaBeanUtils.equal(expiry, other.expiry) &&
          JodaBeanUtils.equal(quantity, other.quantity) &&
          JodaBeanUtils.equal(settlement, other.settlement);
    }
    return false;
  }

  @Override
  public int hashCode() {
    int hash = getClass().hashCode();
    hash = hash * 31 + JodaBeanUtils.hashCode(longShort);
    hash = hash * 31 + JodaBeanUtils.hashCode(underlying);
    hash = hash * 31 + JodaBeanUtils.hashCode(expiry);
    hash = hash * 31 + JodaBeanUtils.hashCode(quantity);
    hash = hash * 31 + JodaBeanUtils.hashCode(settlement);
    return hash;
  }

  @Override
  public String toString() {
    StringBuilder buf = new StringBuilder(192);
    buf.append("ResolvedFixedCouponBondOption{");
    buf.append("longShort").append('=').append(JodaBeanUtils.toString(longShort)).append(',').append(' ');
    buf.append("underlying").append('=').append(JodaBeanUtils.toString(underlying)).append(',').append(' ');
    buf.append("expiry").append('=').append(JodaBeanUtils.toString(expiry)).append(',').append(' ');
    buf.append("quantity").append('=').append(JodaBeanUtils.toString(quantity)).append(',').append(' ');
    buf.append("settlement").append('=').append(JodaBeanUtils.toString(settlement));
    buf.append('}');
    return buf.toString();
  }

  //-----------------------------------------------------------------------
  /**
   * The meta-bean for {@code ResolvedFixedCouponBondOption}.
   */
  public static final class Meta extends DirectMetaBean {
    /**
     * The singleton instance of the meta-bean.
     */
    static final Meta INSTANCE = new Meta();

    /**
     * The meta-property for the {@code longShort} property.
     */
    private final MetaProperty<LongShort> longShort = DirectMetaProperty.ofImmutable(
        this, "longShort", ResolvedFixedCouponBondOption.class, LongShort.class);
    /**
     * The meta-property for the {@code underlying} property.
     */
    private final MetaProperty<ResolvedFixedCouponBond> underlying = DirectMetaProperty.ofImmutable(
        this, "underlying", ResolvedFixedCouponBondOption.class, ResolvedFixedCouponBond.class);
    /**
     * The meta-property for the {@code expiry} property.
     */
    private final MetaProperty<ZonedDateTime> expiry = DirectMetaProperty.ofImmutable(
        this, "expiry", ResolvedFixedCouponBondOption.class, ZonedDateTime.class);
    /**
     * The meta-property for the {@code quantity} property.
     */
    private final MetaProperty<Double> quantity = DirectMetaProperty.ofImmutable(
        this, "quantity", ResolvedFixedCouponBondOption.class, Double.TYPE);
    /**
     * The meta-property for the {@code settlement} property.
     */
    private final MetaProperty<ResolvedFixedCouponBondSettlement> settlement = DirectMetaProperty.ofImmutable(
        this, "settlement", ResolvedFixedCouponBondOption.class, ResolvedFixedCouponBondSettlement.class);
    /**
     * The meta-properties.
     */
    private final Map<String, MetaProperty<?>> metaPropertyMap$ = new DirectMetaPropertyMap(
        this, null,
        "longShort",
        "underlying",
        "expiry",
        "quantity",
        "settlement");

    /**
     * Restricted constructor.
     */
    private Meta() {
    }

    @Override
    protected MetaProperty<?> metaPropertyGet(String propertyName) {
      switch (propertyName.hashCode()) {
        case 116685664:  // longShort
          return longShort;
        case -1770633379:  // underlying
          return underlying;
        case -1289159373:  // expiry
          return expiry;
        case -1285004149:  // quantity
          return quantity;
        case 73828649:  // settlement
          return settlement;
      }
      return super.metaPropertyGet(propertyName);
    }

    @Override
    public ResolvedFixedCouponBondOption.Builder builder() {
      return new ResolvedFixedCouponBondOption.Builder();
    }

    @Override
    public Class<? extends ResolvedFixedCouponBondOption> beanType() {
      return ResolvedFixedCouponBondOption.class;
    }

    @Override
    public Map<String, MetaProperty<?>> metaPropertyMap() {
      return metaPropertyMap$;
    }

    //-----------------------------------------------------------------------
    /**
     * The meta-property for the {@code longShort} property.
     * @return the meta-property, not null
     */
    public MetaProperty<LongShort> longShort() {
      return longShort;
    }

    /**
     * The meta-property for the {@code underlying} property.
     * @return the meta-property, not null
     */
    public MetaProperty<ResolvedFixedCouponBond> underlying() {
      return underlying;
    }

    /**
     * The meta-property for the {@code expiry} property.
     * @return the meta-property, not null
     */
    public MetaProperty<ZonedDateTime> expiry() {
      return expiry;
    }

    /**
     * The meta-property for the {@code quantity} property.
     * @return the meta-property, not null
     */
    public MetaProperty<Double> quantity() {
      return quantity;
    }

    /**
     * The meta-property for the {@code settlement} property.
     * @return the meta-property, not null
     */
    public MetaProperty<ResolvedFixedCouponBondSettlement> settlement() {
      return settlement;
    }

    //-----------------------------------------------------------------------
    @Override
    protected Object propertyGet(Bean bean, String propertyName, boolean quiet) {
      switch (propertyName.hashCode()) {
        case 116685664:  // longShort
          return ((ResolvedFixedCouponBondOption) bean).getLongShort();
        case -1770633379:  // underlying
          return ((ResolvedFixedCouponBondOption) bean).getUnderlying();
        case -1289159373:  // expiry
          return ((ResolvedFixedCouponBondOption) bean).getExpiry();
        case -1285004149:  // quantity
          return ((ResolvedFixedCouponBondOption) bean).getQuantity();
        case 73828649:  // settlement
          return ((ResolvedFixedCouponBondOption) bean).getSettlement();
      }
      return super.propertyGet(bean, propertyName, quiet);
    }

    @Override
    protected void propertySet(Bean bean, String propertyName, Object newValue, boolean quiet) {
      metaProperty(propertyName);
      if (quiet) {
        return;
      }
      throw new UnsupportedOperationException("Property cannot be written: " + propertyName);
    }

  }

  //-----------------------------------------------------------------------
  /**
   * The bean-builder for {@code ResolvedFixedCouponBondOption}.
   */
  public static final class Builder extends DirectFieldsBeanBuilder<ResolvedFixedCouponBondOption> {

    private LongShort longShort;
    private ResolvedFixedCouponBond underlying;
    private ZonedDateTime expiry;
    private double quantity;
    private ResolvedFixedCouponBondSettlement settlement;

    /**
     * Restricted constructor.
     */
    private Builder() {
    }

    /**
     * Restricted copy constructor.
     * @param beanToCopy  the bean to copy from, not null
     */
    private Builder(ResolvedFixedCouponBondOption beanToCopy) {
      this.longShort = beanToCopy.getLongShort();
      this.underlying = beanToCopy.getUnderlying();
      this.expiry = beanToCopy.getExpiry();
      this.quantity = beanToCopy.getQuantity();
      this.settlement = beanToCopy.getSettlement();
    }

    //-----------------------------------------------------------------------
    @Override
    public Object get(String propertyName) {
      switch (propertyName.hashCode()) {
        case 116685664:  // longShort
          return longShort;
        case -1770633379:  // underlying
          return underlying;
        case -1289159373:  // expiry
          return expiry;
        case -1285004149:  // quantity
          return quantity;
        case 73828649:  // settlement
          return settlement;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
    }

    @Override
    public Builder set(String propertyName, Object newValue) {
      switch (propertyName.hashCode()) {
        case 116685664:  // longShort
          this.longShort = (LongShort) newValue;
          break;
        case -1770633379:  // underlying
          this.underlying = (ResolvedFixedCouponBond) newValue;
          break;
        case -1289159373:  // expiry
          this.expiry = (ZonedDateTime) newValue;
          break;
        case -1285004149:  // quantity
          this.quantity = (Double) newValue;
          break;
        case 73828649:  // settlement
          this.settlement = (ResolvedFixedCouponBondSettlement) newValue;
          break;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
      return this;
    }

    @Override
    public Builder set(MetaProperty<?> property, Object value) {
      super.set(property, value);
      return this;
    }

    @Override
    public ResolvedFixedCouponBondOption build() {
      return new ResolvedFixedCouponBondOption(
          longShort,
          underlying,
          expiry,
          quantity,
          settlement);
    }

    //-----------------------------------------------------------------------
    /**
     * Sets whether the option is long or short.
     * <p>
     * Long indicates that the owner has the right to exercise the option at expiry.
     * @param longShort  the new value, not null
     * @return this, for chaining, not null
     */
    public Builder longShort(LongShort longShort) {
      JodaBeanUtils.notNull(longShort, "longShort");
      this.longShort = longShort;
      return this;
    }

    /**
     * Sets the bond underlying the option.
     * @param underlying  the new value, not null
     * @return this, for chaining, not null
     */
    public Builder underlying(ResolvedFixedCouponBond underlying) {
      JodaBeanUtils.notNull(underlying, "underlying");
      this.underlying = underlying;
      return this;
    }

    /**
     * Sets the expiry date-time of the option.
     * <p>
     * The option is European, and can only be exercised on the expiry date.
     * @param expiry  the new value, not null
     * @return this, for chaining, not null
     */
    public Builder expiry(ZonedDateTime expiry) {
      JodaBeanUtils.notNull(expiry, "expiry");
      this.expiry = expiry;
      return this;
    }

    /**
     * Sets the quantity that was traded.
     * <p>
     * This will be positive if buying (call) and negative if selling (put).
     * @param quantity  the new value
     * @return this, for chaining, not null
     */
    public Builder quantity(double quantity) {
      this.quantity = quantity;
      return this;
    }

    /**
     * Sets the bond's settlement details.
     * @param settlement  the new value, not null
     * @return this, for chaining, not null
     */
    public Builder settlement(ResolvedFixedCouponBondSettlement settlement) {
      JodaBeanUtils.notNull(settlement, "settlement");
      this.settlement = settlement;
      return this;
    }

    //-----------------------------------------------------------------------
    @Override
    public String toString() {
      StringBuilder buf = new StringBuilder(192);
      buf.append("ResolvedFixedCouponBondOption.Builder{");
      buf.append("longShort").append('=').append(JodaBeanUtils.toString(longShort)).append(',').append(' ');
      buf.append("underlying").append('=').append(JodaBeanUtils.toString(underlying)).append(',').append(' ');
      buf.append("expiry").append('=').append(JodaBeanUtils.toString(expiry)).append(',').append(' ');
      buf.append("quantity").append('=').append(JodaBeanUtils.toString(quantity)).append(',').append(' ');
      buf.append("settlement").append('=').append(JodaBeanUtils.toString(settlement));
      buf.append('}');
      return buf.toString();
    }

  }

  //-------------------------- AUTOGENERATED END --------------------------
}
