/*
 * Copyright (C) 2020 - present by OpenGamma Inc. and the OpenGamma group of companies
 *
 * Please see distribution for license.
 */
package com.opengamma.strata.product.index.type;

import java.io.Serializable;
import java.lang.invoke.MethodHandles;
import java.time.LocalDate;

import org.joda.beans.ImmutableBean;
import org.joda.beans.JodaBeanUtils;
import org.joda.beans.MetaBean;
import org.joda.beans.TypedMetaBean;
import org.joda.beans.gen.BeanDefinition;
import org.joda.beans.gen.PropertyDefinition;
import org.joda.beans.impl.light.LightMetaBean;

import com.opengamma.strata.basics.ReferenceData;
import com.opengamma.strata.basics.ReferenceDataNotFoundException;
import com.opengamma.strata.basics.date.SequenceDate;
import com.opengamma.strata.basics.index.OvernightIndex;
import com.opengamma.strata.product.SecurityId;
import com.opengamma.strata.product.TradeTemplate;
import com.opengamma.strata.product.index.OvernightFutureTrade;

/**
 * A template for creating an Overnight Future trade.
 */
@BeanDefinition(style = "light")
public final class OvernightFutureTemplate
    implements TradeTemplate, ImmutableBean, Serializable {

  /**
   * The instructions that define which future is desired.
   */
  @PropertyDefinition(validate = "notNull")
  private final SequenceDate sequenceDate;
  /**
   * The underlying contract specification.
   * <p>
   * This specifies the contract of the Overnight Futures to be created.
   */
  @PropertyDefinition(validate = "notNull")
  private final OvernightFutureContractSpec contractSpec;

  //-------------------------------------------------------------------------
  /**
   * Obtains a template based on the specified contract specification and sequence date.
   * <p>
   * The specific future is defined by two date-related inputs -
   * the sequence date and the date sequence embedded in the contract specification.
   * 
   * @param sequenceDate  the instructions that define which future is desired
   * @param contractSpec  the contract specification
   * @return the template
   */
  public static OvernightFutureTemplate of(SequenceDate sequenceDate, OvernightFutureContractSpec contractSpec) {
    return new OvernightFutureTemplate(sequenceDate, contractSpec);
  }

  //-------------------------------------------------------------------------
  /**
   * Gets the underlying index.
   * 
   * @return the index
   */
  public OvernightIndex getIndex() {
    return contractSpec.getIndex();
  }

  /**
   * Creates a trade based on this template.
   * <p>
   * This returns a trade based on the specified date.
   * 
   * @param tradeDate  the date of the trade
   * @param securityId  the identifier of the security
   * @param quantity  the number of contracts traded, positive if buying, negative if selling
   * @param price  the trade price
   * @param refData  the reference data, used to resolve the trade dates
   * @return the trade
   * @throws ReferenceDataNotFoundException if an identifier cannot be resolved in the reference data
   */
  public OvernightFutureTrade createTrade(
      LocalDate tradeDate,
      SecurityId securityId,
      double quantity,
      double price,
      ReferenceData refData) {

    return contractSpec.createTrade(tradeDate, securityId, sequenceDate, quantity, price, refData);
  }

  /**
   * Calculates the reference date of the trade.
   * 
   * @param tradeDate  the date of the trade
   * @param refData  the reference data, used to resolve the date
   * @return the future reference date
   */
  public LocalDate calculateReferenceDateFromTradeDate(LocalDate tradeDate, ReferenceData refData) {
    return contractSpec.calculateReferenceDate(tradeDate, sequenceDate, refData);
  }

  /**
   * Calculates the last fixing date of the trade.
   * 
   * @param tradeDate  the date of the trade
   * @param refData  the reference data, used to resolve the date
   * @return the future reference date
   */
  public LocalDate calculateLastFixingDateFromTradeDate(LocalDate tradeDate, ReferenceData refData) {
    LocalDate referenceDate = calculateReferenceDateFromTradeDate(tradeDate, refData);
    return contractSpec.calculateLastFixingDate(referenceDate, refData);
  }

  //------------------------- AUTOGENERATED START -------------------------
  /**
   * The meta-bean for {@code OvernightFutureTemplate}.
   */
  private static final TypedMetaBean<OvernightFutureTemplate> META_BEAN =
      LightMetaBean.of(
          OvernightFutureTemplate.class,
          MethodHandles.lookup(),
          new String[] {
              "sequenceDate",
              "contractSpec"},
          new Object[0]);

  /**
   * The meta-bean for {@code OvernightFutureTemplate}.
   * @return the meta-bean, not null
   */
  public static TypedMetaBean<OvernightFutureTemplate> meta() {
    return META_BEAN;
  }

  static {
    MetaBean.register(META_BEAN);
  }

  /**
   * The serialization version id.
   */
  private static final long serialVersionUID = 1L;

  private OvernightFutureTemplate(
      SequenceDate sequenceDate,
      OvernightFutureContractSpec contractSpec) {
    JodaBeanUtils.notNull(sequenceDate, "sequenceDate");
    JodaBeanUtils.notNull(contractSpec, "contractSpec");
    this.sequenceDate = sequenceDate;
    this.contractSpec = contractSpec;
  }

  @Override
  public TypedMetaBean<OvernightFutureTemplate> metaBean() {
    return META_BEAN;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the instructions that define which future is desired.
   * @return the value of the property, not null
   */
  public SequenceDate getSequenceDate() {
    return sequenceDate;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the underlying contract specification.
   * <p>
   * This specifies the contract of the Overnight Futures to be created.
   * @return the value of the property, not null
   */
  public OvernightFutureContractSpec getContractSpec() {
    return contractSpec;
  }

  //-----------------------------------------------------------------------
  @Override
  public boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (obj != null && obj.getClass() == this.getClass()) {
      OvernightFutureTemplate other = (OvernightFutureTemplate) obj;
      return JodaBeanUtils.equal(sequenceDate, other.sequenceDate) &&
          JodaBeanUtils.equal(contractSpec, other.contractSpec);
    }
    return false;
  }

  @Override
  public int hashCode() {
    int hash = getClass().hashCode();
    hash = hash * 31 + JodaBeanUtils.hashCode(sequenceDate);
    hash = hash * 31 + JodaBeanUtils.hashCode(contractSpec);
    return hash;
  }

  @Override
  public String toString() {
    StringBuilder buf = new StringBuilder(96);
    buf.append("OvernightFutureTemplate{");
    buf.append("sequenceDate").append('=').append(JodaBeanUtils.toString(sequenceDate)).append(',').append(' ');
    buf.append("contractSpec").append('=').append(JodaBeanUtils.toString(contractSpec));
    buf.append('}');
    return buf.toString();
  }

  //-------------------------- AUTOGENERATED END --------------------------
}
