///////////////////////////////////////////////////////////////////////////////////////////////
// checkstyle: Checks Java source code and other text files for adherence to a set of rules.
// Copyright (C) 2001-2025 the original author or authors.
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///////////////////////////////////////////////////////////////////////////////////////////////

package com.puppycrawl.tools.checkstyle.checks.coding;

import com.puppycrawl.tools.checkstyle.StatelessCheck;
import com.puppycrawl.tools.checkstyle.api.AbstractCheck;
import com.puppycrawl.tools.checkstyle.api.DetailAST;
import com.puppycrawl.tools.checkstyle.api.TokenTypes;

/**
 * <div>
 * Check that the {@code default} is after all the cases in a {@code switch} statement.
 * </div>
 *
 * <p>
 * Rationale: Java allows {@code default} anywhere within the
 * {@code switch} statement. But it is more readable if it comes after the last {@code case}.
 * </p>
 *
 * @since 3.4
 */
@StatelessCheck
public class DefaultComesLastCheck extends AbstractCheck {

    /**
     * A key is pointing to the warning message text in "messages.properties"
     * file.
     */
    public static final String MSG_KEY = "default.comes.last";

    /**
     * A key is pointing to the warning message text in "messages.properties"
     * file.
     */
    public static final String MSG_KEY_SKIP_IF_LAST_AND_SHARED_WITH_CASE =
            "default.comes.last.in.casegroup";

    /** Control whether to allow {@code default} along with {@code case} if they are not last. */
    private boolean skipIfLastAndSharedWithCase;

    @Override
    public int[] getAcceptableTokens() {
        return getRequiredTokens();
    }

    @Override
    public int[] getDefaultTokens() {
        return getRequiredTokens();
    }

    @Override
    public int[] getRequiredTokens() {
        return new int[] {
            TokenTypes.LITERAL_DEFAULT,
        };
    }

    /**
     * Setter to control whether to allow {@code default} along with
     * {@code case} if they are not last.
     *
     * @param newValue whether to ignore checking.
     * @since 7.7
     */
    public void setSkipIfLastAndSharedWithCase(boolean newValue) {
        skipIfLastAndSharedWithCase = newValue;
    }

    @Override
    public void visitToken(DetailAST ast) {
        final DetailAST defaultGroupAST = ast.getParent();

        // Switch rules are not subject to fall through.
        final boolean isSwitchRule = defaultGroupAST.getType() == TokenTypes.SWITCH_RULE;

        if (skipIfLastAndSharedWithCase && !isSwitchRule) {
            if (isNextSiblingOf(ast, TokenTypes.LITERAL_CASE)) {
                log(ast, MSG_KEY_SKIP_IF_LAST_AND_SHARED_WITH_CASE);
            }
            else if (ast.getPreviousSibling() == null
                && isNextSiblingOf(defaultGroupAST,
                                                   TokenTypes.CASE_GROUP)) {
                log(ast, MSG_KEY);
            }
        }
        else if (isNextSiblingOf(defaultGroupAST,
                                            TokenTypes.CASE_GROUP)
                    || isNextSiblingOf(defaultGroupAST,
                                            TokenTypes.SWITCH_RULE)) {
            log(ast, MSG_KEY);
        }
    }

    /**
     * Return true only if passed tokenType in argument is found or returns false.
     *
     * @param ast root node.
     * @param tokenType tokentype to be processed.
     * @return true if desired token is found or else false.
     */
    private static boolean isNextSiblingOf(DetailAST ast, int tokenType) {
        return ast.getNextSibling() != null && ast.getNextSibling().getType() == tokenType;
    }

}
