//  The contents of this file are subject to the Mozilla Public License
//  Version 1.1 (the "License"); you may not use this file except in
//  compliance with the License. You may obtain a copy of the License
//  at http://www.mozilla.org/MPL/
//
//  Software distributed under the License is distributed on an "AS IS"
//  basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
//  the License for the specific language governing rights and
//  limitations under the License.
//
//  The Original Code is RabbitMQ.
//
//  The Initial Developer of the Original Code is GoPivotal, Inc.
//  Copyright (c) 2007-2013 GoPivotal, Inc.  All rights reserved.
//


package com.rabbitmq.client.impl;

import java.io.DataOutputStream;
import java.io.IOException;
import java.util.Date;
import java.util.Map;

import com.rabbitmq.client.ContentHeader;
import com.rabbitmq.client.LongString;

/**
 * Generates an AMQP wire-protocol packet from a {@link ContentHeader}.
 * Methods on this object are usually called from autogenerated code.
 */
public class ContentHeaderPropertyWriter {
    /** Accumulates our output */
    private final ValueWriter out;

    /** Current flags word being accumulated */
    public int flagWord;

    /** Position within current flags word */
    public int bitCount;

    /**
     * Constructs a fresh ContentHeaderPropertyWriter.
     */
    public ContentHeaderPropertyWriter(DataOutputStream out) {
        this.out = new ValueWriter(out);
        this.flagWord = 0;
        this.bitCount = 0;
    }

    private void emitFlagWord(boolean continuationBit) throws IOException {
        out.writeShort(continuationBit ? (flagWord | 1) : flagWord);
        flagWord = 0;
        bitCount = 0;
    }

    public void writePresence(boolean present) throws IOException {
        if (bitCount == 15) {
            emitFlagWord(true);
        }

        if (present) {
            int bit = 15 - bitCount;
            flagWord = flagWord | (1 << bit);
        }
        bitCount++;
    }

    public void finishPresence() throws IOException {
        emitFlagWord(false);
    }

    public void writeShortstr(String str) throws IOException {
        out.writeShortstr(str);
    }

    public void writeLongstr(String str) throws IOException {
        out.writeLongstr(str);
    }

    public void writeLongstr(LongString str) throws IOException {
        out.writeLongstr(str);
    }

    public void writeShort(Integer s) throws IOException {
        out.writeShort(s);
    }

    public void writeLong(Integer l) throws IOException {
        out.writeLong(l);
    }

    public void writeLonglong(Long ll) throws IOException {
        out.writeLonglong(ll);
    }

    public void writeTable(Map<String, Object> table) throws IOException {
        out.writeTable(table);
    }

    public void writeOctet(Integer octet) throws IOException {
        out.writeOctet(octet);
    }

    public void writeOctet(int octet) throws IOException {
        out.writeOctet(octet);
    }

    public void writeTimestamp(Date timestamp) throws IOException {
        out.writeTimestamp(timestamp);
    }
}
