/*
 * Copyright (c) 1997, 2022 Oracle and/or its affiliates. All rights reserved.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Distribution License v. 1.0, which is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

package com.sun.xml.ws.api.pipe.helper;

import com.sun.istack.NotNull;
import com.sun.xml.ws.api.message.Packet;
import com.sun.xml.ws.api.pipe.Fiber;
import com.sun.xml.ws.api.pipe.NextAction;
import com.sun.xml.ws.api.pipe.Pipe;
import com.sun.xml.ws.api.pipe.PipeCloner;
import com.sun.xml.ws.api.pipe.Tube;
import com.sun.xml.ws.api.pipe.TubeCloner;

/**
 * {@link Tube} that invokes {@link Pipe}.
 *
 * <p>
 * This can be used to make a {@link Pipe} look like a {@link Tube}.
 *
 * @author Kohsuke Kawaguchi
 * @author Jitendra Kotamraju
 */
public class PipeAdapter extends AbstractTubeImpl {
    private final Pipe next;

    public static Tube adapt(Pipe p) {
        if (p instanceof Tube) {
            return (Tube) p;
        } else {
            return new PipeAdapter(p);
        }
    }

    public static Pipe adapt(Tube p) {
        if (p instanceof Pipe) {
            return (Pipe) p;
        } else {
            class TubeAdapter extends AbstractPipeImpl {
                private final Tube t;

                public TubeAdapter(Tube t) {
                    this.t = t;
                }

                private TubeAdapter(TubeAdapter that, PipeCloner cloner) {
                    super(that, cloner);
                    this.t = cloner.copy(that.t);
                }

                @Override
                public Packet process(Packet request) {
                    return Fiber.current().runSync(t,request);
                }

                @Override
                public Pipe copy(PipeCloner cloner) {
                    return new TubeAdapter(this,cloner);
                }
            }

            return new TubeAdapter(p);
        }
    }


    private PipeAdapter(Pipe next) {
        this.next = next;
    }

    /**
     * Copy constructor
     */
    private PipeAdapter(PipeAdapter that, TubeCloner cloner) {
        super(that,cloner);
        this.next = ((PipeCloner)cloner).copy(that.next);
    }

    /**
     * Uses the current fiber and runs the whole pipe to the completion
     * (meaning everything from now on will run synchronously.)
     */
    @Override
    public @NotNull NextAction processRequest(@NotNull Packet p) {
        return doReturnWith(next.process(p));
    }

    @Override
    public @NotNull NextAction processResponse(@NotNull Packet p) {
        throw new IllegalStateException();
    }

    @Override
    @NotNull
    public NextAction processException(@NotNull Throwable t) {
        throw new IllegalStateException();
    }

    @Override
    public void preDestroy() {
        next.preDestroy();
    }

    @Override
    public PipeAdapter copy(TubeCloner cloner) {
        return new PipeAdapter(this,cloner);
    }

    public String toString() {
        return super.toString()+"["+next.toString()+"]";
    }
}
