package assertk.assertions

import kotlin.jvm.JvmName
import assertk.Assert
import assertk.all
import assertk.assertions.support.expected
import assertk.assertions.support.show
import assertk.assertions.support.fail
import assertk.assertions.support.appendName

/**
 * Asserts the FloatArray contains the expected element, using `in`.
 * @see [doesNotContain]
 */
@JvmName("floatArrayContains")
fun Assert<FloatArray>.contains(element: Float) = given { actual ->
    if (element in actual.asList()) return
    expected("to contain:${show(element)} but was:${show(actual)}")
}

/**
 * Asserts the FloatArray does not contain the expected element, using `!in`.
 * @see [contains]
 */
@JvmName("floatArrayDoesNotContain")
fun Assert<FloatArray>.doesNotContain(element: Float) = given { actual ->
    if (element !in actual.asList()) return
    expected("to not contain:${show(element)} but was:${show(actual)}")
}

/**
 * Asserts the FloatArray does not contain any of the expected elements.
 * @see [containsAll]
 */
fun Assert<FloatArray>.containsNone(vararg elements: Float) = given { actual ->
    val actualList = actual.asList()
    if (elements.none { it in actualList }) {
        return
    }

    val notExpected = elements.filter { it in actualList }
    expected("to contain none of:${show(elements)} but was:${show(actual)}\n elements not expected:${show(notExpected)}")
}

/**
 * Asserts the FloatArray contains all the expected elements, in any order. The array may also contain
 * additional elements.
 * @see [containsExactly]
 */
@JvmName("floatArrayContainsAll")
fun Assert<FloatArray>.containsAll(vararg elements: Float) = given { actual ->
    val actualList =  actual.asList()
    val elementsList = elements.asList()
    if (elementsList.all { actualList.contains(it) }) return
    val notFound = elementsList.filterNot { it in actualList }
    expected("to contain all:${show(elements)} but was:${show(actual)}\n elements not found:${show(notFound)}")
}

/**
 * Asserts the FloatArray contains only the expected elements, in any order. Duplicate values
 * in the expected and actual are ignored.
 *
 * [1, 2] containsOnly [2, 1] passes
 * [1, 2, 2] containsOnly [2, 1] passes
 * [1, 2] containsOnly [2, 2, 1] passes
 *
 * @see [containsNone]
 * @see [containsExactly]
 * @see [containsAll]
 * @see [containsExactlyInAnyOrder]
 */
fun Assert<FloatArray>.containsOnly(vararg elements: Float) = given { actual ->
    val actualList = actual.asList()
    val elementsList = elements.asList()
    val notInActual = elementsList.filterNot { it in actualList }
    val notInExpected = actualList.filterNot { it in elementsList }
    if (notInExpected.isEmpty() && notInActual.isEmpty()) {
        return
    }
    expected(StringBuilder("to contain only:${show(elements)} but was:${show(actual)}").apply {
        if (notInActual.isNotEmpty()) {
            append("\n elements not found:${show(notInActual)}")
        }
        if (notInExpected.isNotEmpty()) {
            append("\n extra elements found:${show(notInExpected)}")
        }
    }.toString())
}

/**
 * Asserts the FloatArray contains exactly the expected elements. They must be in the same order and
 * there must not be any extra elements.
 *
 * [1, 2] containsOnly [2, 1] fails
 * [1, 2, 2] containsOnly [2, 1] fails
 * [1, 2] containsOnly [2, 2, 1] fails
 *
 * @see [containsAll]
 */
@JvmName("floatArrayContainsExactly")
fun Assert<FloatArray>.containsExactly(vararg elements: Float) = given { actual ->
    val actualList = actual.asList()
    val elementsList = elements.asList()
    if (actualList == elementsList) return

    expected(listDifferExpected(elementsList, actualList))
}

/**
 * Asserts the FloatArray contains exactly the expected elements, in any order. Each value in expected
 * must correspond to a matching value in actual, and visa-versa.
 *
 * [1, 2] containsExactlyInAnyOrder [2, 1] passes
 * [1, 2, 2] containsExactlyInAnyOrder [2, 1] fails
 * [1, 2] containsExactlyInAnyOrder [2, 2, 1] fails
 *
 * @see [containsNone]
 * @see [containsExactly]
 * @see [containsAll]
 * @see [containsOnly]
 */
@JvmName("floatArrayContainsExactlyInAnyOrder")
fun Assert<FloatArray>.containsExactlyInAnyOrder(vararg elements: Float) = given { actual ->
    val actualList = actual.asList()
    val elementsList = elements.asList()
    val notInActual = elementsList.toMutableList()
    val notInExpected = actualList.toMutableList()
    elements.forEach {
        if (notInExpected.contains(it)) {
            notInExpected.removeFirst(it)
            notInActual.removeFirst(it)
        }
    }
    if (notInExpected.isEmpty() && notInActual.isEmpty()) {
        return
    }
    expected(StringBuilder("to contain exactly in any order:${show(elements)} but was:${show(actual)}").apply {
        if (notInActual.isNotEmpty()) {
            append("\n elements not found:${show(notInActual)}")
        }
        if (notInExpected.isNotEmpty()) {
            append("\n extra elements found:${show(notInExpected)}")
        }
    }.toString())
}

/**
 * Asserts the DoubleArray contains the expected element, using `in`.
 * @see [doesNotContain]
 */
@JvmName("doubleArrayContains")
fun Assert<DoubleArray>.contains(element: Double) = given { actual ->
    if (element in actual.asList()) return
    expected("to contain:${show(element)} but was:${show(actual)}")
}

/**
 * Asserts the DoubleArray does not contain the expected element, using `!in`.
 * @see [contains]
 */
@JvmName("doubleArrayDoesNotContain")
fun Assert<DoubleArray>.doesNotContain(element: Double) = given { actual ->
    if (element !in actual.asList()) return
    expected("to not contain:${show(element)} but was:${show(actual)}")
}

/**
 * Asserts the DoubleArray does not contain any of the expected elements.
 * @see [containsAll]
 */
fun Assert<DoubleArray>.containsNone(vararg elements: Double) = given { actual ->
    val actualList = actual.asList()
    if (elements.none { it in actualList }) {
        return
    }

    val notExpected = elements.filter { it in actualList }
    expected("to contain none of:${show(elements)} but was:${show(actual)}\n elements not expected:${show(notExpected)}")
}

/**
 * Asserts the DoubleArray contains all the expected elements, in any order. The array may also contain
 * additional elements.
 * @see [containsExactly]
 */
@JvmName("doubleArrayContainsAll")
fun Assert<DoubleArray>.containsAll(vararg elements: Double) = given { actual ->
    val actualList =  actual.asList()
    val elementsList = elements.asList()
    if (elementsList.all { actualList.contains(it) }) return
    val notFound = elementsList.filterNot { it in actualList }
    expected("to contain all:${show(elements)} but was:${show(actual)}\n elements not found:${show(notFound)}")
}

/**
 * Asserts the DoubleArray contains only the expected elements, in any order. Duplicate values
 * in the expected and actual are ignored.
 *
 * [1, 2] containsOnly [2, 1] passes
 * [1, 2, 2] containsOnly [2, 1] passes
 * [1, 2] containsOnly [2, 2, 1] passes
 *
 * @see [containsNone]
 * @see [containsExactly]
 * @see [containsAll]
 * @see [containsExactlyInAnyOrder]
 */
fun Assert<DoubleArray>.containsOnly(vararg elements: Double) = given { actual ->
    val actualList = actual.asList()
    val elementsList = elements.asList()
    val notInActual = elementsList.filterNot { it in actualList }
    val notInExpected = actualList.filterNot { it in elementsList }
    if (notInExpected.isEmpty() && notInActual.isEmpty()) {
        return
    }
    expected(StringBuilder("to contain only:${show(elements)} but was:${show(actual)}").apply {
        if (notInActual.isNotEmpty()) {
            append("\n elements not found:${show(notInActual)}")
        }
        if (notInExpected.isNotEmpty()) {
            append("\n extra elements found:${show(notInExpected)}")
        }
    }.toString())
}

/**
 * Asserts the DoubleArray contains exactly the expected elements. They must be in the same order and
 * there must not be any extra elements.
 *
 * [1, 2] containsOnly [2, 1] fails
 * [1, 2, 2] containsOnly [2, 1] fails
 * [1, 2] containsOnly [2, 2, 1] fails
 *
 * @see [containsAll]
 */
@JvmName("doubleArrayContainsExactly")
fun Assert<DoubleArray>.containsExactly(vararg elements: Double) = given { actual ->
    val actualList = actual.asList()
    val elementsList = elements.asList()
    if (actualList == elementsList) return

    expected(listDifferExpected(elementsList, actualList))
}

/**
 * Asserts the DoubleArray contains exactly the expected elements, in any order. Each value in expected
 * must correspond to a matching value in actual, and visa-versa.
 *
 * [1, 2] containsExactlyInAnyOrder [2, 1] passes
 * [1, 2, 2] containsExactlyInAnyOrder [2, 1] fails
 * [1, 2] containsExactlyInAnyOrder [2, 2, 1] fails
 *
 * @see [containsNone]
 * @see [containsExactly]
 * @see [containsAll]
 * @see [containsOnly]
 */
@JvmName("doubleArrayContainsExactlyInAnyOrder")
fun Assert<DoubleArray>.containsExactlyInAnyOrder(vararg elements: Double) = given { actual ->
    val actualList = actual.asList()
    val elementsList = elements.asList()
    val notInActual = elementsList.toMutableList()
    val notInExpected = actualList.toMutableList()
    elements.forEach {
        if (notInExpected.contains(it)) {
            notInExpected.removeFirst(it)
            notInActual.removeFirst(it)
        }
    }
    if (notInExpected.isEmpty() && notInActual.isEmpty()) {
        return
    }
    expected(StringBuilder("to contain exactly in any order:${show(elements)} but was:${show(actual)}").apply {
        if (notInActual.isNotEmpty()) {
            append("\n elements not found:${show(notInActual)}")
        }
        if (notInExpected.isNotEmpty()) {
            append("\n extra elements found:${show(notInExpected)}")
        }
    }.toString())
}

