package assertk.assertions

import kotlin.jvm.JvmName
import assertk.Assert
import assertk.all
import assertk.assertions.support.expected
import assertk.assertions.support.show
import assertk.assertions.support.fail
import assertk.assertions.support.appendName

/**
 * Returns an assert on the ByteArray's size.
 */
@JvmName("byteArraySize")
fun Assert<ByteArray>.size() = prop("size") { it.size }

/**
 * Asserts the ByteArray contents are equal to the expected one, using [contentDeepEquals].
 * @see isNotEqualTo
 */
fun Assert<ByteArray>.isEqualTo(expected: ByteArray) = given { actual ->
    if (actual.contentEquals(expected)) return
    fail(expected, actual)
}

/**
 * Asserts the ByteArray contents are not equal to the expected one, using [contentDeepEquals].
 * @see isEqualTo
 */
fun Assert<ByteArray>.isNotEqualTo(expected: ByteArray) = given { actual ->
    if (!(actual.contentEquals(expected))) return
    val showExpected = show(expected)
    val showActual = show(actual)
    // if they display the same, only show one.
    if (showExpected == showActual) {
        expected("to not be equal to:$showActual")
    } else {
        expected(":$showExpected not to be equal to:$showActual")
    }
}

/**
 * Asserts the ByteArray is empty.
 * @see [isNotEmpty]
 * @see [isNullOrEmpty]
 */
@JvmName("byteArrayIsEmpty")
fun Assert<ByteArray>.isEmpty() = given { actual ->
    if (actual.isEmpty()) return
    expected("to be empty but was:${show(actual)}")
}

/**
 * Asserts the ByteArray is not empty.
 * @see [isEmpty]
 */
@JvmName("byteArrayIsNotEmpty")
fun Assert<ByteArray>.isNotEmpty() = given { actual ->
    if (actual.isNotEmpty()) return
    expected("to not be empty")
}

/**
 * Asserts the ByteArray is null or empty.
 * @see [isEmpty]
 */
@JvmName("byteArrayIsNullOrEmpty")
fun Assert<ByteArray?>.isNullOrEmpty() = given { actual ->
    if (actual == null || actual.isEmpty()) return
    expected("to be null or empty but was:${show(actual)}")
}

/**
 * Asserts the ByteArray has the expected size.
 */
@JvmName("byteArrayHasSize")
fun Assert<ByteArray>.hasSize(size: Int) {
    size().isEqualTo(size)
}

/**
 * Asserts the ByteArray has the same size as the expected array.
 */
@JvmName("byteArrayHasSameSizeAs")
fun Assert<ByteArray>.hasSameSizeAs(other: ByteArray) = given { actual ->
    val actualSize = actual.size
    val otherSize = other.size
    if (actualSize == otherSize) return
    expected("to have same size as:${show(other)} ($otherSize) but was size:($actualSize)")
}

/**
 * Returns an assert that assertion on the value at the given index in the array.
 *
 * ```
 * assertThat(byteArrayOf(0, 1, 2)).index(1).isPositive()
 * ```
 */
@JvmName("byteArrayIndex")
fun Assert<ByteArray>.index(index: Int): Assert<Byte> =
    transform(appendName(show(index, "[]"))) { actual ->
        if (index in 0 until actual.size) {
            actual[index]
        } else {
            expected("index to be in range:[0-${actual.size}) but was:${show(index)}")
        }
    }

/**
 * Asserts on each item in the ByteArray. The given lambda will be run for each item.
 *
 * ```
 * assertThat(byteArrayOf("one", "two")).each {
 *   it.hasLength(3)
 * }
 * ```
 */
@JvmName("byteArrayEach")
fun Assert<ByteArray>.each(f: (Assert<Byte>) -> Unit) = given { actual ->
    all {
        actual.forEachIndexed { index, item ->
            f(assertThat(item, name = appendName(show(index, "[]"))))
        }
    }
}

/**
 * Returns an assert on the IntArray's size.
 */
@JvmName("intArraySize")
fun Assert<IntArray>.size() = prop("size") { it.size }

/**
 * Asserts the IntArray contents are equal to the expected one, using [contentDeepEquals].
 * @see isNotEqualTo
 */
fun Assert<IntArray>.isEqualTo(expected: IntArray) = given { actual ->
    if (actual.contentEquals(expected)) return
    fail(expected, actual)
}

/**
 * Asserts the IntArray contents are not equal to the expected one, using [contentDeepEquals].
 * @see isEqualTo
 */
fun Assert<IntArray>.isNotEqualTo(expected: IntArray) = given { actual ->
    if (!(actual.contentEquals(expected))) return
    val showExpected = show(expected)
    val showActual = show(actual)
    // if they display the same, only show one.
    if (showExpected == showActual) {
        expected("to not be equal to:$showActual")
    } else {
        expected(":$showExpected not to be equal to:$showActual")
    }
}

/**
 * Asserts the IntArray is empty.
 * @see [isNotEmpty]
 * @see [isNullOrEmpty]
 */
@JvmName("intArrayIsEmpty")
fun Assert<IntArray>.isEmpty() = given { actual ->
    if (actual.isEmpty()) return
    expected("to be empty but was:${show(actual)}")
}

/**
 * Asserts the IntArray is not empty.
 * @see [isEmpty]
 */
@JvmName("intArrayIsNotEmpty")
fun Assert<IntArray>.isNotEmpty() = given { actual ->
    if (actual.isNotEmpty()) return
    expected("to not be empty")
}

/**
 * Asserts the IntArray is null or empty.
 * @see [isEmpty]
 */
@JvmName("intArrayIsNullOrEmpty")
fun Assert<IntArray?>.isNullOrEmpty() = given { actual ->
    if (actual == null || actual.isEmpty()) return
    expected("to be null or empty but was:${show(actual)}")
}

/**
 * Asserts the IntArray has the expected size.
 */
@JvmName("intArrayHasSize")
fun Assert<IntArray>.hasSize(size: Int) {
    size().isEqualTo(size)
}

/**
 * Asserts the IntArray has the same size as the expected array.
 */
@JvmName("intArrayHasSameSizeAs")
fun Assert<IntArray>.hasSameSizeAs(other: IntArray) = given { actual ->
    val actualSize = actual.size
    val otherSize = other.size
    if (actualSize == otherSize) return
    expected("to have same size as:${show(other)} ($otherSize) but was size:($actualSize)")
}

/**
 * Returns an assert that assertion on the value at the given index in the array.
 *
 * ```
 * assertThat(intArrayOf(0, 1, 2)).index(1).isPositive()
 * ```
 */
@JvmName("intArrayIndex")
fun Assert<IntArray>.index(index: Int): Assert<Int> =
    transform(appendName(show(index, "[]"))) { actual ->
        if (index in 0 until actual.size) {
            actual[index]
        } else {
            expected("index to be in range:[0-${actual.size}) but was:${show(index)}")
        }
    }

/**
 * Asserts on each item in the IntArray. The given lambda will be run for each item.
 *
 * ```
 * assertThat(intArrayOf("one", "two")).each {
 *   it.hasLength(3)
 * }
 * ```
 */
@JvmName("intArrayEach")
fun Assert<IntArray>.each(f: (Assert<Int>) -> Unit) = given { actual ->
    all {
        actual.forEachIndexed { index, item ->
            f(assertThat(item, name = appendName(show(index, "[]"))))
        }
    }
}

/**
 * Returns an assert on the ShortArray's size.
 */
@JvmName("shortArraySize")
fun Assert<ShortArray>.size() = prop("size") { it.size }

/**
 * Asserts the ShortArray contents are equal to the expected one, using [contentDeepEquals].
 * @see isNotEqualTo
 */
fun Assert<ShortArray>.isEqualTo(expected: ShortArray) = given { actual ->
    if (actual.contentEquals(expected)) return
    fail(expected, actual)
}

/**
 * Asserts the ShortArray contents are not equal to the expected one, using [contentDeepEquals].
 * @see isEqualTo
 */
fun Assert<ShortArray>.isNotEqualTo(expected: ShortArray) = given { actual ->
    if (!(actual.contentEquals(expected))) return
    val showExpected = show(expected)
    val showActual = show(actual)
    // if they display the same, only show one.
    if (showExpected == showActual) {
        expected("to not be equal to:$showActual")
    } else {
        expected(":$showExpected not to be equal to:$showActual")
    }
}

/**
 * Asserts the ShortArray is empty.
 * @see [isNotEmpty]
 * @see [isNullOrEmpty]
 */
@JvmName("shortArrayIsEmpty")
fun Assert<ShortArray>.isEmpty() = given { actual ->
    if (actual.isEmpty()) return
    expected("to be empty but was:${show(actual)}")
}

/**
 * Asserts the ShortArray is not empty.
 * @see [isEmpty]
 */
@JvmName("shortArrayIsNotEmpty")
fun Assert<ShortArray>.isNotEmpty() = given { actual ->
    if (actual.isNotEmpty()) return
    expected("to not be empty")
}

/**
 * Asserts the ShortArray is null or empty.
 * @see [isEmpty]
 */
@JvmName("shortArrayIsNullOrEmpty")
fun Assert<ShortArray?>.isNullOrEmpty() = given { actual ->
    if (actual == null || actual.isEmpty()) return
    expected("to be null or empty but was:${show(actual)}")
}

/**
 * Asserts the ShortArray has the expected size.
 */
@JvmName("shortArrayHasSize")
fun Assert<ShortArray>.hasSize(size: Int) {
    size().isEqualTo(size)
}

/**
 * Asserts the ShortArray has the same size as the expected array.
 */
@JvmName("shortArrayHasSameSizeAs")
fun Assert<ShortArray>.hasSameSizeAs(other: ShortArray) = given { actual ->
    val actualSize = actual.size
    val otherSize = other.size
    if (actualSize == otherSize) return
    expected("to have same size as:${show(other)} ($otherSize) but was size:($actualSize)")
}

/**
 * Returns an assert that assertion on the value at the given index in the array.
 *
 * ```
 * assertThat(shortArrayOf(0, 1, 2)).index(1).isPositive()
 * ```
 */
@JvmName("shortArrayIndex")
fun Assert<ShortArray>.index(index: Int): Assert<Short> =
    transform(appendName(show(index, "[]"))) { actual ->
        if (index in 0 until actual.size) {
            actual[index]
        } else {
            expected("index to be in range:[0-${actual.size}) but was:${show(index)}")
        }
    }

/**
 * Asserts on each item in the ShortArray. The given lambda will be run for each item.
 *
 * ```
 * assertThat(shortArrayOf("one", "two")).each {
 *   it.hasLength(3)
 * }
 * ```
 */
@JvmName("shortArrayEach")
fun Assert<ShortArray>.each(f: (Assert<Short>) -> Unit) = given { actual ->
    all {
        actual.forEachIndexed { index, item ->
            f(assertThat(item, name = appendName(show(index, "[]"))))
        }
    }
}

/**
 * Returns an assert on the LongArray's size.
 */
@JvmName("longArraySize")
fun Assert<LongArray>.size() = prop("size") { it.size }

/**
 * Asserts the LongArray contents are equal to the expected one, using [contentDeepEquals].
 * @see isNotEqualTo
 */
fun Assert<LongArray>.isEqualTo(expected: LongArray) = given { actual ->
    if (actual.contentEquals(expected)) return
    fail(expected, actual)
}

/**
 * Asserts the LongArray contents are not equal to the expected one, using [contentDeepEquals].
 * @see isEqualTo
 */
fun Assert<LongArray>.isNotEqualTo(expected: LongArray) = given { actual ->
    if (!(actual.contentEquals(expected))) return
    val showExpected = show(expected)
    val showActual = show(actual)
    // if they display the same, only show one.
    if (showExpected == showActual) {
        expected("to not be equal to:$showActual")
    } else {
        expected(":$showExpected not to be equal to:$showActual")
    }
}

/**
 * Asserts the LongArray is empty.
 * @see [isNotEmpty]
 * @see [isNullOrEmpty]
 */
@JvmName("longArrayIsEmpty")
fun Assert<LongArray>.isEmpty() = given { actual ->
    if (actual.isEmpty()) return
    expected("to be empty but was:${show(actual)}")
}

/**
 * Asserts the LongArray is not empty.
 * @see [isEmpty]
 */
@JvmName("longArrayIsNotEmpty")
fun Assert<LongArray>.isNotEmpty() = given { actual ->
    if (actual.isNotEmpty()) return
    expected("to not be empty")
}

/**
 * Asserts the LongArray is null or empty.
 * @see [isEmpty]
 */
@JvmName("longArrayIsNullOrEmpty")
fun Assert<LongArray?>.isNullOrEmpty() = given { actual ->
    if (actual == null || actual.isEmpty()) return
    expected("to be null or empty but was:${show(actual)}")
}

/**
 * Asserts the LongArray has the expected size.
 */
@JvmName("longArrayHasSize")
fun Assert<LongArray>.hasSize(size: Int) {
    size().isEqualTo(size)
}

/**
 * Asserts the LongArray has the same size as the expected array.
 */
@JvmName("longArrayHasSameSizeAs")
fun Assert<LongArray>.hasSameSizeAs(other: LongArray) = given { actual ->
    val actualSize = actual.size
    val otherSize = other.size
    if (actualSize == otherSize) return
    expected("to have same size as:${show(other)} ($otherSize) but was size:($actualSize)")
}

/**
 * Returns an assert that assertion on the value at the given index in the array.
 *
 * ```
 * assertThat(longArrayOf(0, 1, 2)).index(1).isPositive()
 * ```
 */
@JvmName("longArrayIndex")
fun Assert<LongArray>.index(index: Int): Assert<Long> =
    transform(appendName(show(index, "[]"))) { actual ->
        if (index in 0 until actual.size) {
            actual[index]
        } else {
            expected("index to be in range:[0-${actual.size}) but was:${show(index)}")
        }
    }

/**
 * Asserts on each item in the LongArray. The given lambda will be run for each item.
 *
 * ```
 * assertThat(longArrayOf("one", "two")).each {
 *   it.hasLength(3)
 * }
 * ```
 */
@JvmName("longArrayEach")
fun Assert<LongArray>.each(f: (Assert<Long>) -> Unit) = given { actual ->
    all {
        actual.forEachIndexed { index, item ->
            f(assertThat(item, name = appendName(show(index, "[]"))))
        }
    }
}

/**
 * Returns an assert on the FloatArray's size.
 */
@JvmName("floatArraySize")
fun Assert<FloatArray>.size() = prop("size") { it.size }

/**
 * Asserts the FloatArray contents are equal to the expected one, using [contentDeepEquals].
 * @see isNotEqualTo
 */
fun Assert<FloatArray>.isEqualTo(expected: FloatArray) = given { actual ->
    if (actual.contentEquals(expected)) return
    fail(expected, actual)
}

/**
 * Asserts the FloatArray contents are not equal to the expected one, using [contentDeepEquals].
 * @see isEqualTo
 */
fun Assert<FloatArray>.isNotEqualTo(expected: FloatArray) = given { actual ->
    if (!(actual.contentEquals(expected))) return
    val showExpected = show(expected)
    val showActual = show(actual)
    // if they display the same, only show one.
    if (showExpected == showActual) {
        expected("to not be equal to:$showActual")
    } else {
        expected(":$showExpected not to be equal to:$showActual")
    }
}

/**
 * Asserts the FloatArray is empty.
 * @see [isNotEmpty]
 * @see [isNullOrEmpty]
 */
@JvmName("floatArrayIsEmpty")
fun Assert<FloatArray>.isEmpty() = given { actual ->
    if (actual.isEmpty()) return
    expected("to be empty but was:${show(actual)}")
}

/**
 * Asserts the FloatArray is not empty.
 * @see [isEmpty]
 */
@JvmName("floatArrayIsNotEmpty")
fun Assert<FloatArray>.isNotEmpty() = given { actual ->
    if (actual.isNotEmpty()) return
    expected("to not be empty")
}

/**
 * Asserts the FloatArray is null or empty.
 * @see [isEmpty]
 */
@JvmName("floatArrayIsNullOrEmpty")
fun Assert<FloatArray?>.isNullOrEmpty() = given { actual ->
    if (actual == null || actual.isEmpty()) return
    expected("to be null or empty but was:${show(actual)}")
}

/**
 * Asserts the FloatArray has the expected size.
 */
@JvmName("floatArrayHasSize")
fun Assert<FloatArray>.hasSize(size: Int) {
    size().isEqualTo(size)
}

/**
 * Asserts the FloatArray has the same size as the expected array.
 */
@JvmName("floatArrayHasSameSizeAs")
fun Assert<FloatArray>.hasSameSizeAs(other: FloatArray) = given { actual ->
    val actualSize = actual.size
    val otherSize = other.size
    if (actualSize == otherSize) return
    expected("to have same size as:${show(other)} ($otherSize) but was size:($actualSize)")
}

/**
 * Returns an assert that assertion on the value at the given index in the array.
 *
 * ```
 * assertThat(floatArrayOf(0, 1, 2)).index(1).isPositive()
 * ```
 */
@JvmName("floatArrayIndex")
fun Assert<FloatArray>.index(index: Int): Assert<Float> =
    transform(appendName(show(index, "[]"))) { actual ->
        if (index in 0 until actual.size) {
            actual[index]
        } else {
            expected("index to be in range:[0-${actual.size}) but was:${show(index)}")
        }
    }

/**
 * Asserts on each item in the FloatArray. The given lambda will be run for each item.
 *
 * ```
 * assertThat(floatArrayOf("one", "two")).each {
 *   it.hasLength(3)
 * }
 * ```
 */
@JvmName("floatArrayEach")
fun Assert<FloatArray>.each(f: (Assert<Float>) -> Unit) = given { actual ->
    all {
        actual.forEachIndexed { index, item ->
            f(assertThat(item, name = appendName(show(index, "[]"))))
        }
    }
}

/**
 * Returns an assert on the DoubleArray's size.
 */
@JvmName("doubleArraySize")
fun Assert<DoubleArray>.size() = prop("size") { it.size }

/**
 * Asserts the DoubleArray contents are equal to the expected one, using [contentDeepEquals].
 * @see isNotEqualTo
 */
fun Assert<DoubleArray>.isEqualTo(expected: DoubleArray) = given { actual ->
    if (actual.contentEquals(expected)) return
    fail(expected, actual)
}

/**
 * Asserts the DoubleArray contents are not equal to the expected one, using [contentDeepEquals].
 * @see isEqualTo
 */
fun Assert<DoubleArray>.isNotEqualTo(expected: DoubleArray) = given { actual ->
    if (!(actual.contentEquals(expected))) return
    val showExpected = show(expected)
    val showActual = show(actual)
    // if they display the same, only show one.
    if (showExpected == showActual) {
        expected("to not be equal to:$showActual")
    } else {
        expected(":$showExpected not to be equal to:$showActual")
    }
}

/**
 * Asserts the DoubleArray is empty.
 * @see [isNotEmpty]
 * @see [isNullOrEmpty]
 */
@JvmName("doubleArrayIsEmpty")
fun Assert<DoubleArray>.isEmpty() = given { actual ->
    if (actual.isEmpty()) return
    expected("to be empty but was:${show(actual)}")
}

/**
 * Asserts the DoubleArray is not empty.
 * @see [isEmpty]
 */
@JvmName("doubleArrayIsNotEmpty")
fun Assert<DoubleArray>.isNotEmpty() = given { actual ->
    if (actual.isNotEmpty()) return
    expected("to not be empty")
}

/**
 * Asserts the DoubleArray is null or empty.
 * @see [isEmpty]
 */
@JvmName("doubleArrayIsNullOrEmpty")
fun Assert<DoubleArray?>.isNullOrEmpty() = given { actual ->
    if (actual == null || actual.isEmpty()) return
    expected("to be null or empty but was:${show(actual)}")
}

/**
 * Asserts the DoubleArray has the expected size.
 */
@JvmName("doubleArrayHasSize")
fun Assert<DoubleArray>.hasSize(size: Int) {
    size().isEqualTo(size)
}

/**
 * Asserts the DoubleArray has the same size as the expected array.
 */
@JvmName("doubleArrayHasSameSizeAs")
fun Assert<DoubleArray>.hasSameSizeAs(other: DoubleArray) = given { actual ->
    val actualSize = actual.size
    val otherSize = other.size
    if (actualSize == otherSize) return
    expected("to have same size as:${show(other)} ($otherSize) but was size:($actualSize)")
}

/**
 * Returns an assert that assertion on the value at the given index in the array.
 *
 * ```
 * assertThat(doubleArrayOf(0, 1, 2)).index(1).isPositive()
 * ```
 */
@JvmName("doubleArrayIndex")
fun Assert<DoubleArray>.index(index: Int): Assert<Double> =
    transform(appendName(show(index, "[]"))) { actual ->
        if (index in 0 until actual.size) {
            actual[index]
        } else {
            expected("index to be in range:[0-${actual.size}) but was:${show(index)}")
        }
    }

/**
 * Asserts on each item in the DoubleArray. The given lambda will be run for each item.
 *
 * ```
 * assertThat(doubleArrayOf("one", "two")).each {
 *   it.hasLength(3)
 * }
 * ```
 */
@JvmName("doubleArrayEach")
fun Assert<DoubleArray>.each(f: (Assert<Double>) -> Unit) = given { actual ->
    all {
        actual.forEachIndexed { index, item ->
            f(assertThat(item, name = appendName(show(index, "[]"))))
        }
    }
}

/**
 * Returns an assert on the CharArray's size.
 */
@JvmName("charArraySize")
fun Assert<CharArray>.size() = prop("size") { it.size }

/**
 * Asserts the CharArray contents are equal to the expected one, using [contentDeepEquals].
 * @see isNotEqualTo
 */
fun Assert<CharArray>.isEqualTo(expected: CharArray) = given { actual ->
    if (actual.contentEquals(expected)) return
    fail(expected, actual)
}

/**
 * Asserts the CharArray contents are not equal to the expected one, using [contentDeepEquals].
 * @see isEqualTo
 */
fun Assert<CharArray>.isNotEqualTo(expected: CharArray) = given { actual ->
    if (!(actual.contentEquals(expected))) return
    val showExpected = show(expected)
    val showActual = show(actual)
    // if they display the same, only show one.
    if (showExpected == showActual) {
        expected("to not be equal to:$showActual")
    } else {
        expected(":$showExpected not to be equal to:$showActual")
    }
}

/**
 * Asserts the CharArray is empty.
 * @see [isNotEmpty]
 * @see [isNullOrEmpty]
 */
@JvmName("charArrayIsEmpty")
fun Assert<CharArray>.isEmpty() = given { actual ->
    if (actual.isEmpty()) return
    expected("to be empty but was:${show(actual)}")
}

/**
 * Asserts the CharArray is not empty.
 * @see [isEmpty]
 */
@JvmName("charArrayIsNotEmpty")
fun Assert<CharArray>.isNotEmpty() = given { actual ->
    if (actual.isNotEmpty()) return
    expected("to not be empty")
}

/**
 * Asserts the CharArray is null or empty.
 * @see [isEmpty]
 */
@JvmName("charArrayIsNullOrEmpty")
fun Assert<CharArray?>.isNullOrEmpty() = given { actual ->
    if (actual == null || actual.isEmpty()) return
    expected("to be null or empty but was:${show(actual)}")
}

/**
 * Asserts the CharArray has the expected size.
 */
@JvmName("charArrayHasSize")
fun Assert<CharArray>.hasSize(size: Int) {
    size().isEqualTo(size)
}

/**
 * Asserts the CharArray has the same size as the expected array.
 */
@JvmName("charArrayHasSameSizeAs")
fun Assert<CharArray>.hasSameSizeAs(other: CharArray) = given { actual ->
    val actualSize = actual.size
    val otherSize = other.size
    if (actualSize == otherSize) return
    expected("to have same size as:${show(other)} ($otherSize) but was size:($actualSize)")
}

/**
 * Returns an assert that assertion on the value at the given index in the array.
 *
 * ```
 * assertThat(charArrayOf(0, 1, 2)).index(1).isPositive()
 * ```
 */
@JvmName("charArrayIndex")
fun Assert<CharArray>.index(index: Int): Assert<Char> =
    transform(appendName(show(index, "[]"))) { actual ->
        if (index in 0 until actual.size) {
            actual[index]
        } else {
            expected("index to be in range:[0-${actual.size}) but was:${show(index)}")
        }
    }

/**
 * Asserts on each item in the CharArray. The given lambda will be run for each item.
 *
 * ```
 * assertThat(charArrayOf("one", "two")).each {
 *   it.hasLength(3)
 * }
 * ```
 */
@JvmName("charArrayEach")
fun Assert<CharArray>.each(f: (Assert<Char>) -> Unit) = given { actual ->
    all {
        actual.forEachIndexed { index, item ->
            f(assertThat(item, name = appendName(show(index, "[]"))))
        }
    }
}

