package com.xpn.xwiki.wysiwyg.client.plugin.image.ui;

import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.ui.Button;
import com.google.gwt.user.client.ui.ClickListener;
import com.google.gwt.user.client.ui.FlowPanel;
import com.google.gwt.user.client.ui.FormHandler;
import com.google.gwt.user.client.ui.FormSubmitCompleteEvent;
import com.google.gwt.user.client.ui.FormSubmitEvent;
import com.google.gwt.user.client.ui.SourcesTabEvents;
import com.google.gwt.user.client.ui.TabListener;
import com.google.gwt.user.client.ui.TabPanel;
import com.google.gwt.user.client.ui.Widget;
import com.xpn.xwiki.wysiwyg.client.editor.Strings;
import com.xpn.xwiki.wysiwyg.client.plugin.image.ImageConfig;
import com.xpn.xwiki.wysiwyg.client.plugin.image.ImageHTMLGenerator;
import com.xpn.xwiki.wysiwyg.client.widget.CompositeDialogBox;

/**
 * Dialog for the image insert plugin, to handle choosing an image to insert from the existing images or uploading a new
 * one.
 * 
 * @version $Id$
 */
public class ImageDialog extends CompositeDialogBox implements ClickListener, FormHandler, TabListener
{
    /**
     * The HTML block for the image to be inserted. Will be generated by the {@link ImageHTMLGenerator} and returned to
     * the image plugin on dialog close.
     */
    private String imageHTMLBlock;

    /**
     * Image selection panel to pick the image to use.
     */
    private ImageSelector imageSelector;

    /**
     * Image parameters panel to configure the inserted image.
     */
    private ImageParametersPanel imageParameters;

    /**
     * Form to upload an image to the current document.
     */
    private FileUploadForm fileUploadForm;

    /**
     * Button to insert the selected image, according to its parameters.
     */
    private Button insertImageButton;

    /**
     * Tab panel to add the image selection and the image settings in different tabs.
     */
    private TabPanel tabsPanel;

    /**
     * The wiki of the document we are currently editing.
     */
    private String currentWiki;

    /**
     * The space of the document we are currently editing.
     */
    private String currentSpace;

    /**
     * The page name of the document we are currently editing.
     */
    private String currentPage;
    
    /**
     * Flag to store of the dialog has been loaded or not.
     */
    private boolean isLoaded;

    /**
     * Creates an image insertion dialog, for the referred page, and the default file upload url.
     * 
     * @param currentWiki the wiki of the document we are currently editing.
     * @param currentSpace the space of the document we are currently editing.
     * @param currentPage the page of the document we are currently editing.
     */
    public ImageDialog(String currentWiki, String currentSpace, String currentPage)
    {
        super(false, true);
        this.currentWiki = currentWiki;
        this.currentSpace = currentSpace;
        this.currentPage = currentPage;

        getDialog().setText(Strings.INSTANCE.image());

        FlowPanel mainPanel = new FlowPanel();
        tabsPanel = new TabPanel();
        imageSelector = new ImageSelector(currentWiki, currentSpace, currentPage);
        String uploadUrl = "../../upload/" + currentSpace + "/" + currentPage;
        fileUploadForm = new FileUploadForm(uploadUrl, "filepath");
        fileUploadForm.addFormHandler(this);
        FlowPanel imageSelectionPanel = new FlowPanel();
        imageSelectionPanel.add(imageSelector);
        imageSelectionPanel.add(fileUploadForm);
        tabsPanel.add(imageSelectionPanel, Strings.INSTANCE.imageSelectTabTitle());

        imageParameters = new ImageParametersPanel();
        tabsPanel.add(imageParameters, Strings.INSTANCE.imageSettingsTabTitle());
        tabsPanel.selectTab(0);
        tabsPanel.addTabListener(this);

        mainPanel.add(tabsPanel);

        insertImageButton = new Button(Strings.INSTANCE.imageInsertButton());
        insertImageButton.addStyleName("xInsertButton");
        insertImageButton.addClickListener(this);
        mainPanel.add(insertImageButton);

        mainPanel.addStyleName("xImageDialogMain");
        initWidget(mainPanel);
    }

    /**
     * {@inheritDoc}
     * 
     * @see ClickListener#onClick(Widget)
     */
    public void onClick(Widget sender)
    {
        if (sender == insertImageButton) {
            if (validateUserInput()) {
                insertImage();
            }
        }
    }

    /**
     * Function to execute the code for an image insertion: get parameters, prepare html block, etc.
     */
    private void insertImage()
    {
        ImageConfig selectedImage = imageSelector.getSelectedImage();
        selectedImage.setAlignment(imageParameters.getSelectedAlignment());
        selectedImage.setAltText(imageParameters.getAltText());
        selectedImage.setWidth(imageParameters.getWidth());
        selectedImage.setHeight(imageParameters.getHeight());
        imageHTMLBlock =
            ImageHTMLGenerator.getInstance()
                .getAttachedImageHTML(selectedImage, currentWiki, currentSpace, currentPage);
        this.hide();
    }

    /**
     * @return the image HTML block built as a result of the user input in this dialog.
     */
    public String getImageHTMLBlock()
    {
        return imageHTMLBlock;
    }

    /**
     * {@inheritDoc}
     * 
     * @see CompositeDialogBox#center()
     */
    public void center()
    {
        super.center();
        // reset the value of the block each time we show this dialog.
        imageHTMLBlock = null;
        if (this.tabsPanel != null) {
            tabsPanel.selectTab(0);
        }
    }

    /**
     * {@inheritDoc}
     * 
     * @see FormHandler#onSubmit(FormSubmitEvent)
     */
    public void onSubmit(FormSubmitEvent event)
    {
        // nothing
    }

    /**
     * {@inheritDoc}
     * 
     * @see FormHandler#onSubmitComplete(FormSubmitCompleteEvent)
     */
    public void onSubmitComplete(FormSubmitCompleteEvent event)
    {
        imageSelector.setSelection(currentWiki, currentSpace, currentPage, null, false);
    }

    /**
     * {@inheritDoc}
     * 
     * @see TabListener#onBeforeTabSelected(SourcesTabEvents, int)
     */
    public boolean onBeforeTabSelected(SourcesTabEvents sender, int tabIndex)
    {
        // nothing here
        return true;
    }

    /**
     * {@inheritDoc}
     * 
     * @see TabListener#onTabSelected(SourcesTabEvents, int)
     */
    public void onTabSelected(SourcesTabEvents sender, int tabIndex)
    {
        // set the focus if it's the second tab the one which is selected
        if (tabIndex == 1) {
            imageParameters.onShow();
        }
    }

    /**
     * Validates the user input for this dialog.
     * 
     * @return true if the user data is valid and dialog can be closed, false otherwise. 
     */
    public boolean validateUserInput()
    {
        // check to have an image selected
        if (imageSelector.getSelectedImage() == null) {
            Window.alert(Strings.INSTANCE.imageNoImageSelectedError());
            return false;
        }
        // all other parameters are optional, return true
        return true;
    }

    /**
     * Sets the image that is edited by this dialog.
     * 
     * @param config the {@link ImageConfig} corresponding to the edited image.
     */
    public void setImageConfig(ImageConfig config)
    {
        if (config.getImageFileName() != null || !isLoaded) {
            // Change the selectors only if we're editing an image
            imageSelector.setSelection(config.getWiki() != null ? config.getWiki() : currentWiki,
                config.getSpace() != null ? config.getSpace() : currentSpace, config.getPage() != null ? config
                    .getPage() : currentPage, config.getImageFileName(), !isLoaded);
            if (!isLoaded) {
                isLoaded = true;
            }
        } 
        imageParameters.setParameters(config);
    }
}
