package de.flapdoodle.embed.mongo.client;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import org.bson.Document;

/**
 * Immutable implementation of {@link MongoClientAction.RunCommand}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableRunCommand.builder()}.
 * Use the static factory method to create immutable instances:
 * {@code ImmutableRunCommand.of()}.
 */
@SuppressWarnings({"all"})
final class ImmutableRunCommand extends MongoClientAction.RunCommand {
  private final String database;
  private final Document command;

  private ImmutableRunCommand(String database, Document command) {
    this.database = Objects.requireNonNull(database, "database");
    this.command = Objects.requireNonNull(command, "command");
  }

  private ImmutableRunCommand(ImmutableRunCommand original, String database, Document command) {
    this.database = database;
    this.command = command;
  }

  /**
   * @return The value of the {@code database} attribute
   */
  @Override
  public String database() {
    return database;
  }

  /**
   * @return The value of the {@code command} attribute
   */
  @Override
  public Document command() {
    return command;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link MongoClientAction.RunCommand#database() database} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for database
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableRunCommand withDatabase(String value) {
    String newValue = Objects.requireNonNull(value, "database");
    if (this.database.equals(newValue)) return this;
    return new ImmutableRunCommand(this, newValue, this.command);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link MongoClientAction.RunCommand#command() command} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for command
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableRunCommand withCommand(Document value) {
    if (this.command == value) return this;
    Document newValue = Objects.requireNonNull(value, "command");
    return new ImmutableRunCommand(this, this.database, newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableRunCommand} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableRunCommand
        && equalTo(0, (ImmutableRunCommand) another);
  }

  private boolean equalTo(int synthetic, ImmutableRunCommand another) {
    return database.equals(another.database)
        && command.equals(another.command);
  }

  /**
   * Computes a hash code from attributes: {@code database}, {@code command}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + database.hashCode();
    h += (h << 5) + command.hashCode();
    return h;
  }

  /**
   * Prints the immutable value {@code RunCommand} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "RunCommand{"
        + "database=" + database
        + ", command=" + command
        + "}";
  }

  /**
   * Construct a new immutable {@code RunCommand} instance.
   * @param database The value for the {@code database} attribute
   * @param command The value for the {@code command} attribute
   * @return An immutable RunCommand instance
   */
  public static ImmutableRunCommand of(String database, Document command) {
    return new ImmutableRunCommand(database, command);
  }

  /**
   * Creates an immutable copy of a {@link MongoClientAction.RunCommand} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable RunCommand instance
   */
  public static ImmutableRunCommand copyOf(MongoClientAction.RunCommand instance) {
    if (instance instanceof ImmutableRunCommand) {
      return (ImmutableRunCommand) instance;
    }
    return ImmutableRunCommand.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableRunCommand ImmutableRunCommand}.
   * <pre>
   * ImmutableRunCommand.builder()
   *    .database(String) // required {@link MongoClientAction.RunCommand#database() database}
   *    .command(org.bson.Document) // required {@link MongoClientAction.RunCommand#command() command}
   *    .build();
   * </pre>
   * @return A new ImmutableRunCommand builder
   */
  public static ImmutableRunCommand.Builder builder() {
    return new ImmutableRunCommand.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableRunCommand ImmutableRunCommand}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  public static final class Builder {
    private static final long INIT_BIT_DATABASE = 0x1L;
    private static final long INIT_BIT_COMMAND = 0x2L;
    private long initBits = 0x3L;

    private String database;
    private Document command;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code de.flapdoodle.embed.mongo.client.MongoClientAction.RunCommand} instance.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(MongoClientAction.RunCommand instance) {
      Objects.requireNonNull(instance, "instance");
      from((short) 0, (Object) instance);
      return this;
    }

    /**
     * Fill a builder with attribute values from the provided {@code de.flapdoodle.embed.mongo.client.MongoClientAction.Action} instance.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(MongoClientAction.Action instance) {
      Objects.requireNonNull(instance, "instance");
      from((short) 0, (Object) instance);
      return this;
    }

    private void from(short _unused, Object object) {
      if (object instanceof MongoClientAction.RunCommand) {
        MongoClientAction.RunCommand instance = (MongoClientAction.RunCommand) object;
        this.command(instance.command());
      }
      if (object instanceof MongoClientAction.Action) {
        MongoClientAction.Action instance = (MongoClientAction.Action) object;
        this.database(instance.database());
      }
    }

    /**
     * Initializes the value for the {@link MongoClientAction.RunCommand#database() database} attribute.
     * @param database The value for database 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder database(String database) {
      this.database = Objects.requireNonNull(database, "database");
      initBits &= ~INIT_BIT_DATABASE;
      return this;
    }

    /**
     * Initializes the value for the {@link MongoClientAction.RunCommand#command() command} attribute.
     * @param command The value for command 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder command(Document command) {
      this.command = Objects.requireNonNull(command, "command");
      initBits &= ~INIT_BIT_COMMAND;
      return this;
    }

    /**
     * Builds a new {@link ImmutableRunCommand ImmutableRunCommand}.
     * @return An immutable instance of RunCommand
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableRunCommand build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutableRunCommand(null, database, command);
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_DATABASE) != 0) attributes.add("database");
      if ((initBits & INIT_BIT_COMMAND) != 0) attributes.add("command");
      return "Cannot build RunCommand, some of required attributes are not set " + attributes;
    }
  }
}
