package de.flapdoodle.embed.process.io.directories;

import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * Immutable implementation of {@link TempDir}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableTempDir.builder()}.
 * Use the static factory method to create immutable instances:
 * {@code ImmutableTempDir.of()}.
 */
@SuppressWarnings({"all"})
public final class ImmutableTempDir extends TempDir {
  private final Path value;

  private ImmutableTempDir(Path value) {
    this.value = Objects.requireNonNull(value, "value");
  }

  private ImmutableTempDir(ImmutableTempDir original, Path value) {
    this.value = value;
  }

  /**
   * @return The value of the {@code value} attribute
   */
  @Override
  public Path value() {
    return value;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link TempDir#value() value} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for value
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableTempDir withValue(Path value) {
    if (this.value == value) return this;
    Path newValue = Objects.requireNonNull(value, "value");
    return new ImmutableTempDir(this, newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableTempDir} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableTempDir
        && equalTo(0, (ImmutableTempDir) another);
  }

  private boolean equalTo(int synthetic, ImmutableTempDir another) {
    return value.equals(another.value);
  }

  /**
   * Computes a hash code from attributes: {@code value}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + value.hashCode();
    return h;
  }

  /**
   * Construct a new immutable {@code TempDir} instance.
   * @param value The value for the {@code value} attribute
   * @return An immutable TempDir instance
   */
  public static ImmutableTempDir of(Path value) {
    return new ImmutableTempDir(value);
  }

  /**
   * Creates an immutable copy of a {@link TempDir} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable TempDir instance
   */
  public static ImmutableTempDir copyOf(TempDir instance) {
    if (instance instanceof ImmutableTempDir) {
      return (ImmutableTempDir) instance;
    }
    return ImmutableTempDir.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableTempDir ImmutableTempDir}.
   * <pre>
   * ImmutableTempDir.builder()
   *    .value(java.nio.file.Path) // required {@link TempDir#value() value}
   *    .build();
   * </pre>
   * @return A new ImmutableTempDir builder
   */
  public static ImmutableTempDir.Builder builder() {
    return new ImmutableTempDir.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableTempDir ImmutableTempDir}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  public static final class Builder {
    private static final long INIT_BIT_VALUE = 0x1L;
    private long initBits = 0x1L;

    private Path value;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code TempDir} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(TempDir instance) {
      Objects.requireNonNull(instance, "instance");
      this.value(instance.value());
      return this;
    }

    /**
     * Initializes the value for the {@link TempDir#value() value} attribute.
     * @param value The value for value 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder value(Path value) {
      this.value = Objects.requireNonNull(value, "value");
      initBits &= ~INIT_BIT_VALUE;
      return this;
    }

    /**
     * Builds a new {@link ImmutableTempDir ImmutableTempDir}.
     * @return An immutable instance of TempDir
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableTempDir build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutableTempDir(null, value);
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_VALUE) != 0) attributes.add("value");
      return "Cannot build TempDir, some of required attributes are not set " + attributes;
    }
  }
}
