/*

 Copyright (c) 2005-2011, Carlos Amengual.

 Licensed under a BSD-style License. You can find the license here:
 http://www.informatica.info/projects/css/LICENSE.txt

 */

package info.informatica.doc.dom4j;

import info.informatica.doc.style.css.StyleDatabase;
import info.informatica.doc.style.css.dom.SelectorMatcher;

import java.util.List;

import org.apache.log4j.Logger;
import org.dom4j.Element;
import org.dom4j.QName;
import org.w3c.dom.DOMException;
import org.w3c.dom.css.CSSStyleDeclaration;
import org.w3c.dom.css.ElementCSSInlineStyle;

/**
 * An element that is stylable with CSS.
 * 
 * @author Carlos Amengual (amengual at informatica.info)
 * 
 */
public class CSSStylableElement extends XHTMLElement implements
		ElementCSSInlineStyle {

	private static final long serialVersionUID = 2L;
	
	private SelectorMatcher selectorMatcher = null;

	private String idAttr = null;

	static Logger log = Logger.getLogger(CSSStylableElement.class.getName());

	public CSSStylableElement(String name) {
		super(name);
	}

	public CSSStylableElement(QName qname) {
		super(qname);
	}

    public CSSStylableElement(QName qname, int attributeCount) {
        super(qname, attributeCount);
    }

	protected String inlineStyle() {
		return attributeValue("style");
	}

	/**
	 * Gets the inline style declaration from the <code>style</code> XHTML
	 * attribute.
	 * 
	 * @return the style declaration, or null if the element has no
	 *         <code>style</code> attribute or the style could not be parsed.
	 */
	public CSSStyleDeclaration getStyle() {
		DOM4JCSSStyleDeclaration styledecl = null;
		String st = inlineStyle();
		if (st != null && st.length() > 0) {
			styledecl = new DOM4JCSSStyleDeclaration();
			try {
				styledecl.setCssText(st);
				styledecl.setPeerNode(this);
				StyleDatabase styleDb = getDocument().getStyleDatabase();
				if(styleDb != null && styledecl.getStyleDatabase() == null){
					styledecl.setStyleDatabase(styleDb);
				}
			} catch (DOMException e) {
				log.error("Could not parse style", e);
				styledecl = null;
			}
		}
		return styledecl;
	}

	/**
	 * Gets the computed style declaration that applies to this element.
	 * 
	 * @return the computed style declaration, or null if none applies.
	 */
	public CSSStyleDeclaration getComputedStyle() {
		DOM4JCSSStyleSheet css = getDocument().getStyleSheet();
		if(css != null){
			// Get the style declaration
			DOM4JCSSStyleDeclaration styledecl = (DOM4JCSSStyleDeclaration) css
				.getComputedStyle(this, null);
			// Set the default style database, if any
			StyleDatabase styleDb = getDocument().getStyleDatabase();
			if(styleDb != null && styledecl.getStyleDatabase() == null){
				styledecl.setStyleDatabase(styleDb);
			}
			return styledecl;
		} else {
			return null;
		}
	}

	/**
	 * DOM4J CSS Selector matcher.
	 * 
	 * @author Carlos Amengual (amengual at informatica.info)
	 * 
	 */
	class DOM4JSelectorMatcher extends SelectorMatcher {

		DOM4JSelectorMatcher() {
			super();
			setTagname(getName().intern());
			org.dom4j.Element parent = getParent();
			if (parent != null) {
				setParentTagname(parent.getName().intern());
			}
			setClassAttribute(attributeValue("class"));
			String id = attributeValue("id");
			idAttr = id != null ? id : "";
		}

		@Override
		public SelectorMatcher getParentSelectorMatcher() {
			Element parent = getParent();
			if(parent instanceof CSSStylableElement) {
				return ((CSSStylableElement)parent).getSelectorMatcher();
			} else {
				return null;
			}
		}

		@Override
		public SelectorMatcher getPreviousSiblingSelectorMatcher() {
			Element parent = getParent();
			if(parent == null) {
				return null;
			}
			List elements = parent.elements();
			// Determine previous sibling
			Object sibling = elements.get(elements.indexOf(
					CSSStylableElement.this) - 1);
			if(sibling instanceof CSSStylableElement) {
				return ((CSSStylableElement)sibling).getSelectorMatcher();
			} else {
				return null;
			}
		}

		@Override
		public boolean isFirstChild() {
			return getParent().indexOf(CSSStylableElement.this) == 0;
		}

		@Override
		public String getAttributeValue(String attrName) {
			String value = attributeValue(attrName);
			return value != null? value : "";
		}

		@Override
		public boolean hasAttribute(String attrName) {
			return attribute(attrName) != null;
		}

		@Override
		public String getId() {
			return idAttr;
		}

		@Override
		public String getLanguage() {
			/*
			 *  FIXME: In (X)HTML, the lang attribute contains the language,
			 *  but that may not be true for other XML.
			 */
			String lang = attributeValue("lang");
			while(lang == null || lang.length() == 0) {
				Element parent = getParent();
				if(parent == null) {
					break;
				} else {
					lang = parent.attributeValue("lang");
				}
			}
			if(lang == null) {
				lang = "";
			}
			return lang;
		}

	}

	/**
	 * Gets the selector matcher for this element.
	 * 
	 * @return the selector matcher.
	 */
	public SelectorMatcher getSelectorMatcher() {
		if(selectorMatcher == null) {
			selectorMatcher = new DOM4JSelectorMatcher();
		}
		return selectorMatcher;
	}

}
