/*

 Copyright (c) 2005-2007, Carlos Amengual.

 Licensed under a BSD-style License. You can find the license here:
 http://www.informatica.info/projects/css/LICENSE.txt

 */

package info.informatica.doc.dom4j;

import info.informatica.doc.agent.UserAgent;
import info.informatica.doc.style.css.CSS2ComputedProperties;
import info.informatica.doc.style.css.CSSStyleException;
import info.informatica.doc.style.css.dom.ComputedCSSStyle;
import info.informatica.doc.style.css.visual.CSSBox;
import info.informatica.doc.style.css.visual.CSSBoxFactory;
import info.informatica.doc.style.css.visual.CSSContainerBox;
import info.informatica.doc.style.css.visual.NonStaticallyPositioned;
import info.informatica.doc.style.css.visual.ReplacedElementBox;
import info.informatica.doc.style.css.visual.box.AbsolutelyPositionedBox;
import info.informatica.doc.style.css.visual.box.AbstractCSSBox;
import info.informatica.doc.style.css.visual.box.BlockBox;
import info.informatica.doc.style.css.visual.box.BlockContainer;
import info.informatica.doc.style.css.visual.box.InlineContainer;
import info.informatica.doc.style.css.visual.box.InlineTable;
import info.informatica.doc.style.css.visual.box.RunInBox;
import info.informatica.doc.style.css.visual.box.Table;
import info.informatica.doc.style.css.visual.box.TableCellBox;
import info.informatica.doc.style.css.visual.box.TableRowBox;
import info.informatica.doc.style.css.visual.box.TableRowContainer;
import info.informatica.doc.style.css.visual.container.CSSBlockBoxContainer;
import info.informatica.doc.style.css.visual.container.CSSBoxContainer;
import info.informatica.doc.style.css.visual.container.CSSInlineBoxContainer;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.dom4j.CDATA;
import org.dom4j.Element;
import org.dom4j.Node;
import org.dom4j.Text;
import org.w3c.dom.css.CSSPrimitiveValue;

/**
 * Builds the formatting structure for DOM4J documents.
 * <p>
 * This is the class to use for rendering CSS-styled DOM4J documents.
 * 
 * @author Carlos Amengual (amengual at informatica.info)
 *
 */
public class DocumentFormatter {

	private DOM4JCSSErrorHandler errorHandler = null;

	private CSSBoxFactory boxFactory;
	
	private UserAgent<?> userAgent = null;

	public DocumentFormatter(UserAgent<?> userAgent) {
		super();
		this.userAgent = userAgent;
	}

	public DOM4JCSSErrorHandler getErrorHandler() {
		return errorHandler;
	}

	public UserAgent<?> getUserAgent() {
		return userAgent;
	}

	public void setUserAgent(UserAgent<?> userAgent) {
		this.userAgent = userAgent;
	}

	/**
	 * Sets the error handler.
	 * 
	 * @param errorHandler the DOM4J-specific CSS error handler.
	 */
	public void setErrorHandler(DOM4JCSSErrorHandler errorHandler) {
		this.errorHandler = errorHandler;
	}

	public CSSBoxFactory getBoxFactory() {
		return boxFactory;
	}

	public void setBoxFactory(CSSBoxFactory boxFactory) {
		this.boxFactory = boxFactory;
	}

	/**
	 * Builds a CSS formatting structure for the given document.
	 * 
	 * @param xdoc the document to be formatted.
	 * @return the root box containing the format structure.
	 * @throws CSSStyleException if the root box could not be established.
	 */
	public CSSContainerBox formatDocument(XHTMLDocument xdoc) 
		throws CSSStyleException {
		// Do we have error handler?
		if(errorHandler == null) {
			errorHandler = new DummyErrorHandler();
		}
		// Is the factory set?
		if(boxFactory == null) {
			boxFactory = new CSSBoxFactory();
		}
		boxFactory.setErrorHandler(errorHandler);
		CSSContainerBox rootBox = null;
		Element html = xdoc.getRootElement();
		// Iterator of child elements
		Iterator it = html.elementIterator();
		// Iterate for finding the topmost element with style.
		while(it.hasNext()){
			Element elem = (Element) it.next();
			if(elem instanceof CSSStylableElement){
				// found the 'body' element (or so)
				CSSBox box = generateBox((DOM4JCSSStyleDeclaration)
						((CSSStylableElement)elem).getComputedStyle());
				if(box instanceof CSSContainerBox){
					iterateChild((CSSStylableElement)elem, (CSSContainerBox) box, 
							(CSSContainerBox) box);
					rootBox = (CSSContainerBox) box;
					break;
				} else {
					throw new CSSStyleException("Root box not defined as block");
				}
			}
		}
		return rootBox;
	}
	
	private void iterateChild(CSSStylableElement parent, CSSContainerBox container, 
			CSSContainerBox rootBox) {
		List<DOM4JCSSStyleDeclaration> childStyles = new ArrayList<DOM4JCSSStyleDeclaration>();
		Iterator it = parent.nodeIterator();
		// Check if has at least one child block box, and store the styles meanwhile
		boolean hasBlockChild = false;
		while(it.hasNext()) {
			Node node = (Node) it.next();
			// It could be an element, or an anonymous box
			if(node instanceof CSSStylableElement){
				DOM4JCSSStyleDeclaration style = (DOM4JCSSStyleDeclaration)
					((CSSStylableElement)node).getComputedStyle();
				String display = ((CSSPrimitiveValue)style.getCSSValue("display"))
					.getStringValue();
				if(display.equals("none")) {
					continue;
				}
				String position = ((CSSPrimitiveValue)style.getCSSValue("position"))
					.getStringValue();
				if(position.equals("absolute")) {
					AbsolutelyPositionedBox box = new AbsolutelyPositionedBox(style);
					// find container
					while(!(container instanceof AbsolutelyPositionedBox)) {
						container = container.getContainingBlock();
						if(container == null) {
							container = rootBox;
							break;
						}
					}
					box.setContainingBlock(container);
					((BlockContainer)container.asContainerBox()).addAbsolutelyPositioned(box);
					continue;
				} else if(position.equals("fixed")) {
					AbsolutelyPositionedBox box = new AbsolutelyPositionedBox(style);
					box.setContainingBlock(rootBox);
					((BlockContainer)rootBox.asContainerBox()).addAbsolutelyPositioned(box);
					continue;
				} else {
					// Normal flow
					childStyles.add(style);
					if(!display.equals("inline") && 
							!display.equals("inline-table")) {
						// Got a block
						hasBlockChild = true;
					}
				}
			} else if(node instanceof CDATA) {
				String content = node.getText().trim();
				if(content.length() > 0) {
					// Create an empty style for anonymous inline box, 
					// and add to list of styles
					childStyles.add(createAnonymousStyleDeclaration(node, parent));
				}
			} else if(node instanceof Text) {
				String content = node.getText().trim();
				if(content.length() > 0) {
					// Create an empty style for anonymous inline box, 
					// and add to list of styles
					childStyles.add(createAnonymousStyleDeclaration(node, parent));
				}
			}
		}
		// Is the container a run-in ?
		if(container instanceof RunInBox && hasBlockChild) {
			// Part 9.2.3.1 of CSS spec.
			((RunInBox)container).setBlock();
		}
		// Generate the appropriate boxes,
		// and store in a List
		List<CSSBox> childBoxes = new ArrayList<CSSBox>();
		for(int i=0; i<childStyles.size(); i++){
			DOM4JCSSStyleDeclaration style = childStyles.get(i);
			CSSBox box = null;
			String display = ((CSSPrimitiveValue)style
					.getCSSValue("display")).getStringValue();
			if(hasBlockChild) {
				if(display.equals("inline")) {
					// If one child is block, all child must be blocks
					box = boxFactory.createBlockBox(style);
				} else if(display.equals("inline-table")) {
					// inline-table is now block-table
					box = boxFactory.createTableBox(style);
				}
			}
			if(box == null) {
				try {
					box = generateBox(style);
					if(box == null) {
						// This element won't display
						continue;
					}
				} catch (CSSStyleException e) {
					errorHandler.error("Error generating box for " 
							+ style.getPeerXPath(), e,
							(Node) style.getPeerNode());
					continue;
				}
			}
			// Set the containing block
			((AbstractCSSBox)box).setContainingBlock(container);
			childBoxes.add(box);
		}
		if(!childBoxes.isEmpty()) {
			// Set the descendants
			if(hasBlockChild) {
				packBlocks(container, rootBox, childBoxes);
				List<CSSContainerBox> childBlocks = ((BlockContainer)container.asContainerBox()).getStaticallyPositionedList();
				postIterateBlockChild(childBlocks);
			} else {
				// Check whether we have run-in (block!) boxes here
				for(int i=0; i<childBoxes.size(); i++) {
					CSSBox box = childBoxes.get(i);
					if(box instanceof RunInBox) {
						((RunInBox)box).setBlock();
						childBoxes.set(i, ((RunInBox)box).finalBox());
						hasBlockChild = true;
					}
				}
				if(hasBlockChild) {
					// Handle all child boxes as block
					packBlocks(container, rootBox, childBoxes);
					// No need to do post-iteration to look for runners.
				} else {
					// We have only inline-level child boxes
					InlineContainer inCont = new InlineContainer();
					container.setBoxContainer(inCont);
					List<CSSBox> inList = inCont.getInlineBoxes();
					inList.addAll(childBoxes);
				}
			}
		}
	}

	private void packBlocks(CSSContainerBox container, 
			CSSContainerBox rootBox, 
			List<CSSBox> childBoxes) {
		// Handle the block-level child elements
		BlockContainer blockCont = new BlockContainer(childBoxes.size() + 1);
		container.setBoxContainer(blockCont);
		List<CSSContainerBox> childBlocks = blockCont.getStaticallyPositionedList();
		Iterator<CSSBox> childIt = childBoxes.iterator();
		while(childIt.hasNext()) {
			childBlocks.add((CSSContainerBox) childIt.next());
		}
		// Recurse contained boxes
		for(int i=0; i<childBlocks.size(); i++){
			CSSContainerBox descendant = childBlocks.get(i);
			CSSStylableElement elem = (CSSStylableElement) 
				((DOM4JCSSStyleDeclaration)descendant.getComputedStyle())
				.getPeerNode();
			// Check whether it is an anonymous box
			if((descendant instanceof Table) ||
					(descendant instanceof InlineTable)) {
				buildTable(elem, descendant, rootBox);
			} else {
				iterateChild(elem, descendant, rootBox);
				if(descendant instanceof RunInBox) {
					CSSBox finalBox = ((RunInBox)descendant).finalBox();
					if(finalBox != null) {
						// Must be a block
						childBlocks.set(i, (CSSContainerBox) finalBox);
					}
				}
			}
		}
	}

	private void postIterateBlockChild(List<CSSContainerBox> childBlocks) {
		// Handle the possible run-in boxes
		for(int i=0; i<childBlocks.size(); i++){
			CSSBox box = childBlocks.get(i);
			if(box instanceof RunInBox && ((RunInBox)box).finalBox() == null) {
				int nextSiblingIdx = i + 1;
				if(nextSiblingIdx == childBlocks.size()) {
					// It is last sibling, so it becomes block
					((RunInBox)box).setBlock();
					childBlocks.set(i, (CSSContainerBox) ((RunInBox)box).finalBox());
				} else {
					CSSBox nextSiblingBox = childBlocks.get(nextSiblingIdx);
					if(nextSiblingBox instanceof BlockBox && 
							!(nextSiblingBox instanceof NonStaticallyPositioned)) {
						// Part 9.2.3.2 of CSS spec.
						childBlocks.remove(i);
						CSSBoxContainer boxCont = ((BlockBox)nextSiblingBox).asContainerBox();
						if(boxCont instanceof CSSInlineBoxContainer) {
							List<CSSBox> inList = ((InlineContainer)boxCont).getInlineBoxes();
							((RunInBox)box).setInline();
							inList.add(0, ((RunInBox)box).finalBox());
						} else if(boxCont instanceof CSSBlockBoxContainer) {
							List<CSSContainerBox> blockList = ((BlockContainer)boxCont)
								.getStaticallyPositionedList();
							((RunInBox)box).setBlock();
							blockList.add(0, (CSSContainerBox) ((RunInBox)box).finalBox());
						}
					} else {
						// Part 9.2.3.3 of CSS spec.
						((RunInBox)box).setBlock();
						childBlocks.set(i, (CSSContainerBox) ((RunInBox)box).finalBox());
					}
				}
			}
		}
	}

	private void buildTable(CSSStylableElement parent, CSSContainerBox table, 
			CSSContainerBox rootBox) {
		TableRowContainer boxContainer = new TableRowContainer();
		Iterator it = parent.nodeIterator();
		while(it.hasNext()) {
			Node node = (Node) it.next();
			// It could be an element, or an anonymous box
			if(node instanceof CSSStylableElement){
				ComputedCSSStyle style = (ComputedCSSStyle)((CSSStylableElement)node)
					.getComputedStyle();
				String display = ((CSSPrimitiveValue)style.getCSSValue("display"))
					.getStringValue();
				if(display.equals("none")) {
					continue;
				}
				if(display.equals("table-row")) {
					TableRowBox row = (TableRowBox) boxFactory
										.createTableRowBox(style);
					boxContainer.getRows().add(row);
					buildRow((CSSStylableElement)node, row, rootBox);
				}
			} else {
				errorHandler.error("Unknown table-level element: " + 
						node.getName());
			}
		}
		table.setBoxContainer(boxContainer);
	}

	private void buildRow(CSSStylableElement parent, TableRowBox row, 
			CSSContainerBox rootBox) {
		Iterator it = parent.nodeIterator();
		while(it.hasNext()) {
			Node node = (Node) it.next();
			// It could be an element, or an anonymous box
			if(node instanceof CSSStylableElement){
				ComputedCSSStyle style = (ComputedCSSStyle)((CSSStylableElement)node)
					.getComputedStyle();
				String display = ((CSSPrimitiveValue)style.getCSSValue("display"))
					.getStringValue();
				if(display.equals("none")) {
					continue;
				}
				if(display.equals("table-cell")) {
					TableCellBox cell = (TableCellBox) boxFactory
										.createTableCellBox(style);
					row.add(cell);
					// Check for background images
					checkBackgroundImage(style);
					// Children
					iterateChild((CSSStylableElement)node, cell, rootBox);
				}
			} else {
				errorHandler.error("Unknown table-level element: " + 
						node.getName());
			}
		}
	}

	private DOM4JCSSStyleDeclaration createAnonymousStyleDeclaration(
			Node node, CSSStylableElement parent) {
		DOM4JCSSStyleDeclaration style = new DOM4JCSSStyleDeclaration();
		style.setPeerNode((org.w3c.dom.Node) node);
		// Set the style database
		style.setStyleDatabase(
			parent.getDocument().getStyleDatabase());
		return style;
	}

	/**
	 * Creates a box of the appropriate type according to CSS box model.
	 * <p>
	 * Note that table sub-elements are not generated here.
	 * 
	 * @param style the computed style that applies for the box.
	 * @return the generated box, or null if the element generates no box 
	 * (<code>display: none</code>).
	 * @throws CSSStyleException if an error occurs generating the box.
	 */
	protected CSSBox generateBox(DOM4JCSSStyleDeclaration style) 
	throws CSSStyleException {
		org.w3c.dom.Node node = style.getPeerNode();
		CSSBox generated = null;
		if(node instanceof CSSStylableElement) {
			// Check for replaced elements
			CSSStylableElement elem = (CSSStylableElement) node;
			generated = userAgent.getElementReplacer(
					elem.getNamespaceURI()).createReplacedElementBox(elem);
			if(generated == null) {
				// The element is non-replaced: generate the non-replaced box
				generated = boxFactory.create(style);
				if(generated != null) {
					// Check for background images
					checkBackgroundImage(style);
				}
			} else {
				// Initialize replaced elements
				generated.setErrorHandler(errorHandler);
				((ReplacedElementBox)generated).init(userAgent);
			}
		} else {
			// Text nodes
			generated = boxFactory.createInlineBox(style);
		}
		return generated;
	}

	protected void checkBackgroundImage(CSS2ComputedProperties style) {
		String imgUrl = style.getBackgroundImage();
		if(imgUrl != null) {
			try {
				userAgent.download(new URL(imgUrl));
			} catch (MalformedURLException e) {
				errorHandler.error("Bad URL for background image at "
						+ style.getPeerXPath(), e);
			}
		}
	}

}
