/*

 Copyright (c) 2005-2011, Carlos Amengual.

 Licensed under a BSD-style License. You can find the license here:
 http://www.informatica.info/projects/css/LICENSE.txt

 */

package info.informatica.doc.pdf.itext;

import info.informatica.doc.agent.ResourceDownloader;
import info.informatica.doc.agent.DownloadListener;
import info.informatica.doc.agent.UserAgent;
import info.informatica.doc.style.css.CSS2ComputedProperties;
import info.informatica.doc.style.css.StyleDatabase;
import info.informatica.doc.style.css.dom.ComputedCSSStyle;
import info.informatica.doc.style.css.j2d.Java2DStyleDatabase;
import info.informatica.doc.style.css.visual.CSSBox;
import info.informatica.doc.style.css.visual.CSSContainerBox;
import info.informatica.doc.style.css.visual.CSSInlineBox;
import info.informatica.doc.style.css.visual.CSSTableCellBox;
import info.informatica.doc.style.css.visual.box.ListItemBox;
import info.informatica.doc.style.css.visual.box.ListItemBox.ListItemMarkerBox;

import java.awt.Color;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;

import org.apache.log4j.Logger;
import org.w3c.dom.css.CSSPrimitiveValue;

import com.itextpdf.text.BaseColor;
import com.itextpdf.text.Chunk;
import com.itextpdf.text.Document;
import com.itextpdf.text.DocumentException;
import com.itextpdf.text.Element;
import com.itextpdf.text.Font;
import com.itextpdf.text.Image;
import com.itextpdf.text.List;
import com.itextpdf.text.ListItem;
import com.itextpdf.text.Paragraph;
import com.itextpdf.text.Phrase;
import com.itextpdf.text.Rectangle;
import com.itextpdf.text.pdf.BaseFont;
import com.itextpdf.text.pdf.PdfPCell;
import com.itextpdf.text.pdf.PdfPTable;

/**
 * Utility class for helping in the format of iText PDF documents based on CSS
 * style sheets.
 * <p>
 * TODO: work in progress.
 * </p>
 * 
 * @author Carlos Amengual (amengual at informatica.info)
 * 
 */
public class PDFStyleFormatter {

	private UserAgent<Image> userAgent;

	static Logger log = Logger.getLogger(PDFStyleFormatter.class.getName());

	public PDFStyleFormatter(UserAgent<Image> userAgent) {
		super();
		this.userAgent = userAgent;
	}

	/**
	 * Gets the style database associated to this formatter.
	 * 
	 * @return the style database.
	 */
	public StyleDatabase getStyleDatabase() {
		return userAgent.getStyleDatabase();
	}

	protected UserAgent<Image> getUserAgent() {
		return userAgent;
	}

	/**
	 * Create an iText font from the properties specified in the given style.
	 * 
	 * @param styledecl
	 *            the style declaration.
	 * @return the Font specified in the style sheet, or the closest one that could be created.
	 */
	public Font createFont(CSS2ComputedProperties styledecl) throws DocumentException {
		String fontfamily = styledecl.getFontFamily();
		float sz = styledecl.getFontSize();
		// Font style
		String stylename = styledecl.getPropertyValue("font-style");
		int style = Font.NORMAL;
		if (stylename.length() > 0) {
			stylename = stylename.toLowerCase();
			if (stylename.equals("italic")) {
				style = Font.ITALIC;
			}
		}
		String fontweight = styledecl.getFontWeight();
		if (fontweight != null) {
			fontweight = fontweight.toLowerCase();
			if (fontweight.equals("bold")) {
				if (style != Font.ITALIC) {
					style = Font.BOLD;
				} else {
					style = Font.BOLDITALIC;
				}
			}
		}
		String decoration = styledecl.getPropertyValue("text-decoration");
		if (decoration.length() > 0) {
			decoration = decoration.toLowerCase();
			if (decoration.equals("underline")) {
				style = Font.UNDERLINE;
			} else if (decoration.equals("line-through")) {
				style = Font.STRIKETHRU;
			}
		}
		//
		Font font = null;
		if (fontfamily != null) {
			try {
				BaseFont bf = BaseFont.createFont(fontfamily, BaseFont.WINANSI,
						BaseFont.NOT_EMBEDDED);
				font = new Font(bf, sz, style);
			} catch (Exception e) {
				log.error(e);
				try {
					font = new Font(BaseFont.createFont(), sz, style);
				} catch (IOException e1) {
					// iText documentation says "This shouldn't occur ever"
					log.error(e1);
				}
			}
		} else {
			try {
				font = new Font(BaseFont.createFont(), sz, style);
			} catch (IOException e) {
				// iText documentation says "This shouldn't occur ever"
				log.error(e);
			}
		}
		// Font color
		Color color = Java2DStyleDatabase.getAWTColor(styledecl.getColor());
		if (color != null) {
			font.setColor(new BaseColor(color));
		}
		return font;
	}

	/**
	 * Translates the alignment as provided by CSS/XHTML, to alignment as
	 * understood by the iText package.
	 * 
	 * @param align
	 *            a string with the alignment identifier (left, right, center,
	 *            justify).
	 * @return the alignment according to <code>com.itextpdf.text.Element</code>.
	 */
	public static int translateAlignment(String align) {
		if (align == null) {
			return Element.ALIGN_UNDEFINED;
		}
		String lcalign = align.toLowerCase();
		if (lcalign.equals("left")) {
			return Element.ALIGN_LEFT;
		} else if (lcalign.equals("right")) {
			return Element.ALIGN_RIGHT;
		} else if (lcalign.equals("center")) {
			return Element.ALIGN_CENTER;
		} else if (lcalign.equals("justify")) {
			return Element.ALIGN_JUSTIFIED;
		} else {
			return Element.ALIGN_UNDEFINED;
		}
	}

	/**
	 * Translates the vertical alignment as provided by CSS/XHTML, to same
	 * alignment as understood by the iText package.
	 * 
	 * @return the vertical alignment according to
	 *         <code>com.itextpdf.text.Element</code>.
	 */
	public static int translateVerticalAlignment(String align) {
		// TODO: review table height algorithms
		if (align.equals("middle")) {
			return Element.ALIGN_MIDDLE;
		} else if (align.equals("baseline")) {
			return Element.ALIGN_BASELINE;
		} else if (align.equals("bottom")) {
			return Element.ALIGN_BOTTOM;
		} else if (align.equals("top")) {
			return Element.ALIGN_TOP;
		}
		return Element.ALIGN_UNDEFINED;
	}

	/**
	 * Creates a chunk with the text and font specified by the given box.
	 * 
	 * @param box
	 *            the inline box.
	 * @return the chunk.
	 */
	public Chunk createChunk(CSSInlineBox box) {
		return createChunk(box, box.getText());
	}

	/**
	 * Creates a chunk with the text and font specified by the given box.
	 * 
	 * @param box
	 *            the inline box.
	 * @return the chunk.
	 */
	public Chunk createChunk(CSSInlineBox box, String text) {
		Chunk ch;
		CSS2ComputedProperties styledecl = box.getComputedStyle();
		if(box instanceof DownloadListener) {
			ch = createChunk((DownloadListener<Image>)box);
		} else if(styledecl.getBackgroundImage() != null) {
			ResourceDownloader<Image> downloader;
			try {
				downloader = getUserAgent().getResourceDownloader(
						new URL(styledecl.getBackgroundImage()));
				// FIXME deadlock
				while(!downloader.isDone());
				ch = new Chunk(downloader.getNativeContent(), 0, 0);
			} catch (MalformedURLException e) {
				log.error(e);
				ch = new Chunk("");
			}
		} else {
			ch = new Chunk(text);
		}
		formatChunk(styledecl, ch);
		return ch;
	}

	/**
	 * Creates a chunk with an image.
	 * 
	 * @param box
	 *            the box containing an image.
	 * @return the chunk.
	 */
	public Chunk createChunk(DownloadListener<Image> box) {
		return new Chunk(box.getNativeContent(), 0, 0);
	}

	/**
	 * Creates a chunk with the given text and the font specified by the style
	 * applying to the given box.
	 * 
	 * @param styledecl
	 *            the style applying to the chunk box.
	 * @param ch
	 *            the chunk to format.
	 */
	public void formatChunk(CSS2ComputedProperties styledecl, Chunk ch) {
		Font font = null;
		try {
			font = createFont(styledecl);
		} catch (Exception e) {
			log.error("Unable to create font from given properties", e);
		}
		if (font != null) {
			ch.setFont(font);
		}
		// Background color
		Color color = Java2DStyleDatabase.getAWTColor(styledecl.getBackgroundColor());
		if (color != null) {
			ch.setBackground(new BaseColor(color));
		}
	}

	/**
	 * Creates a Phrase object according to the style of an inline box.
	 * 
	 * @param box the inline box.
	 * @return the phrase.
	 */
	public Phrase createPhrase(CSSInlineBox box) {
		Phrase phrase = new Phrase();
		// Leading
		float float_value = box.getLeading();
		phrase.setLeading(float_value);
		return phrase;
	}

	public void formatDocument(CSSBox box, Document pdf) {
		pdf.setMargins(box.getMarginLeft(), box.getMarginRight(), 
				box.getMarginTop(), box.getMarginBottom());
	}

	public void formatRectangle(CSSBox box, Rectangle rect) {
		ComputedCSSStyle styledecl = (ComputedCSSStyle) box.getComputedStyle();
		// Background color
		Color color = Java2DStyleDatabase.getAWTColor(styledecl.getBackgroundColor());
		if (color != null) {
			rect.setBackgroundColor(new BaseColor(color));
		}
		// borders
		// border color
		color = Java2DStyleDatabase.getAWTColor(box.getBorderTopColor());
		if (color != null) {
			rect.setBorderColorTop(new BaseColor(color));
		}
		color = Java2DStyleDatabase.getAWTColor(box.getBorderRightColor());
		if (color != null) {
			rect.setBorderColorRight(new BaseColor(color));
		}
		color = Java2DStyleDatabase.getAWTColor(box.getBorderBottomColor());
		if (color != null) {
			rect.setBorderColorBottom(new BaseColor(color));
		}
		color = Java2DStyleDatabase.getAWTColor(box.getBorderLeftColor());
		if (color != null) {
			rect.setBorderColorLeft(new BaseColor(color));
		}
		// border width
		rect.setBorderWidthTop(box.getBorderTopWidth());
		rect.setBorderWidthRight(box.getBorderRightWidth());
		rect.setBorderWidthBottom(box.getBorderBottomWidth());
		rect.setBorderWidthLeft(box.getBorderLeftWidth());
		// border-style
		String s = styledecl.getPropertyValue("border-top-style").toLowerCase();
		if (s.equals("none") || s.equals("hidden")) {
			rect.disableBorderSide(Rectangle.TOP);
		}
		s = styledecl.getPropertyValue("border-right-style").toLowerCase();
		if (s.equals("none") || s.equals("hidden")) {
			rect.disableBorderSide(Rectangle.RIGHT);
		}
		s = styledecl.getPropertyValue("border-bottom-style").toLowerCase();
		if (s.equals("none") || s.equals("hidden")) {
			rect.disableBorderSide(Rectangle.BOTTOM);
		}
		s = styledecl.getPropertyValue("border-left-style").toLowerCase();
		if (s.equals("none") || s.equals("hidden")) {
			rect.disableBorderSide(Rectangle.LEFT);
		}
	}
	
	/**
	 * Formats the given paragraph according to the style that applies to the
	 * given peer element.
	 * 
	 * @param box
	 *            the box to be rendered.
	 * @param par
	 *            the paragraph to be formatted.
	 */
	public void formatParagraph(CSSBox box, Paragraph par) {
		ComputedCSSStyle styledecl = (ComputedCSSStyle) box.getComputedStyle();
		styledecl.setStyleDatabase(getStyleDatabase());
		// margins
		par.setIndentationLeft(box.getMarginLeft());
		par.setIndentationRight(box.getMarginRight());
		par.setSpacingBefore(box.getMarginTop());
		par.setSpacingAfter(box.getMarginBottom());
		// Alignment
		String alignTxt = ((CSSPrimitiveValue)styledecl.getCSSValue(
				"text-align")).getStringValue();
		par.setAlignment(translateAlignment(alignTxt));
	}

	/**
	 * Formats a PdfPTable according to the style that applies to the given peer
	 * element.
	 * 
	 * @param box
	 *            the box to be rendered.
	 * @param table
	 *            the table to be formatted.
	 */
	public void formatPTable(CSSContainerBox box, PdfPTable table) {
		ComputedCSSStyle styledecl = (ComputedCSSStyle) box.getComputedStyle();
		styledecl.setStyleDatabase(getStyleDatabase());
		table.setSpacingBefore(box.getMarginTop());
		table.setSpacingAfter(box.getMarginBottom());
	}

	/**
	 * Formats a PdfPCell according to the style that applies to the given peer
	 * element.
	 * 
	 * @param box
	 *            the box to be rendered.
	 * @param cell
	 *            the cell to be formatted.
	 */
	public void formatPCell(CSSTableCellBox box, PdfPCell cell) {
		ComputedCSSStyle styledecl = (ComputedCSSStyle) box.getComputedStyle();
		styledecl.setStyleDatabase(getStyleDatabase());
		// Padding
		cell.setPaddingTop(box.getPaddingTop());
		cell.setPaddingRight(box.getPaddingRight());
		cell.setPaddingBottom(box.getPaddingBottom());
		cell.setPaddingLeft(box.getPaddingLeft());
		// Alignment
		String alignTxt = ((CSSPrimitiveValue)styledecl.getCSSValue("text-align")).getStringValue();
		cell.setHorizontalAlignment(translateAlignment(alignTxt));
		// Vertical alignment
		int align = translateVerticalAlignment(((CSSPrimitiveValue)styledecl.getCSSValue("vertical-align")).getStringValue());
		if (Element.ALIGN_UNDEFINED != align) {
			cell.setVerticalAlignment(align);
		}
	}

	public void formatList(CSSContainerBox box, List list) {
		// margins
		list.setIndentationLeft(box.getMarginLeft());
		list.setIndentationRight(box.getMarginRight());
	}

	public void formatListItem(CSSContainerBox box, ListItem item) {
		ListItemMarkerBox markerBox = (ListItemMarkerBox)box.getGeneratedSibling();
		if(markerBox instanceof ListItemBox.ListItemNumberedMarkerBox) {
			Chunk ch = new Chunk(((ListItemBox.ListItemNumberedMarkerBox)markerBox).itemSymbol());
			formatChunk(box.getComputedStyle(), ch);
			item.setListSymbol(ch);
		} else {
			// TODO
			//item.setListSymbol(symbol);
		}
	}

}
