/*

 Copyright (c) 2005-2007, Carlos Amengual.

 Licensed under a BSD-style License. You can find the license here:
 http://www.informatica.info/projects/css/LICENSE.txt

 */

package info.informatica.doc.style.css;

import org.w3c.dom.DOMException;
import org.w3c.dom.css.CSSPrimitiveValue;

/**
 * Abstract base class for CSS Style databases.
 * 
 * @author Carlos Amengual (amengual at informatica.info)
 */
abstract public class AbstractStyleDatabase implements StyleDatabase {

	protected final String DEFAULT_GENERIC_FONT_FAMILY = "serif";

	public static final CSSPrimitiveValue DEFAULT_INITIAL_COLOR = (CSSPrimitiveValue) 
						StyleDeclarationFactory.parseProperty("#000000");

	private CSSPrimitiveValue initialColor;
	
	private float scrollbarWidth = 0;

	public AbstractStyleDatabase() {
		super();
		initialColor = DEFAULT_INITIAL_COLOR;
	}

	/**
	 * Converts from centimeters to pixels.
	 * 
	 * @param cm the value, in centimeters, to be converted.
	 * @return the value, expressed in pixels.
	 */
	abstract protected float cmToPixels(float cm);

	/**
	 * Converts from pixels to centimeters.
	 * 
	 * @param px the value, in pixels, to be converted.
	 * @return the value, expressed in centimeters.
	 */
	abstract protected float pxTocm(int px);

	public int getExSizeInPt(String familyName, int size) {
		return Math.round(0.5f * size);
	}

	public float floatValueConversion(float initialValue,
			short initialUnitType, short destUnitType) throws DOMException {
		/*
		 * This method is a hack.
		 */
		if (initialUnitType == destUnitType
				|| initialUnitType == CSSPrimitiveValue.CSS_NUMBER) {
			return initialValue;
		}
		if (initialValue == 0f) {
			return 0f;
		}
		switch (initialUnitType) {
		case CSSPrimitiveValue.CSS_PT:
			switch (destUnitType) {
			case CSSPrimitiveValue.CSS_IN:
				return initialValue / 72f;
			case CSSPrimitiveValue.CSS_CM:
				return initialValue / 28.35f;
			case CSSPrimitiveValue.CSS_PX:
				return cmToPixels(initialValue / 28.35f);
			}
			break;
		case CSSPrimitiveValue.CSS_CM:
			switch (destUnitType) {
			case CSSPrimitiveValue.CSS_PT:
				return initialValue * 28.35f;
			case CSSPrimitiveValue.CSS_IN:
				return initialValue / 2.54f;
			case CSSPrimitiveValue.CSS_PX:
				return cmToPixels(initialValue);
			}
			break;
		case CSSPrimitiveValue.CSS_IN:
			switch (destUnitType) {
			case CSSPrimitiveValue.CSS_PT:
				return initialValue * 72f;
			case CSSPrimitiveValue.CSS_CM:
				return initialValue * 2.54f;
			case CSSPrimitiveValue.CSS_PX:
				return cmToPixels(initialValue * 2.54f);
			}
			break;
		case CSSPrimitiveValue.CSS_PX:
			switch (destUnitType) {
			case CSSPrimitiveValue.CSS_PT:
				return pxTocm((int) initialValue) * 28.35f;
			case CSSPrimitiveValue.CSS_CM:
				return pxTocm((int) initialValue);
			case CSSPrimitiveValue.CSS_IN:
				return pxTocm((int) initialValue) / 2.54f;
			}
			break;
		case CSSPrimitiveValue.CSS_PERCENTAGE:
		case CSSPrimitiveValue.CSS_NUMBER:
			// Do nothing
			return initialValue;
		}
		throw new DOMException(DOMException.INVALID_ACCESS_ERR,
				"Unable to do unit conversion from " + initialUnitType + " to "
						+ destUnitType);
	}

	public float floatValueConversion(float initialValue,
			short initialUnitType) throws DOMException {
		return floatValueConversion(initialValue, initialUnitType, 
				getNaturalUnit());
	}

	/* (non-Javadoc)
	 * @see info.informatica.doc.style.css.StyleDatabase#getInitialColor()
	 */
	public CSSPrimitiveValue getInitialColor() {
		return initialColor;
	}

	/* (non-Javadoc)
	 * @see info.informatica.doc.style.css.StyleDatabase#setInitialColor(String)
	 */
	public void setInitialColor(String initialColor) {
		this.initialColor = (CSSPrimitiveValue) StyleDeclarationFactory
			.parseProperty(initialColor);
	}

	/* (non-Javadoc)
	 * @see info.informatica.doc.style.css.StyleDatabase#getScrollbarWidth()
	 */
	public float getScrollbarWidth() {
		return scrollbarWidth;
	}

	public String getDefaultGenericFontFamily() {
		return getDefaultGenericFontFamily(DEFAULT_GENERIC_FONT_FAMILY);
	}

}
