/*

 Copyright (c) 2005-2011, Carlos Amengual.

 Licensed under a BSD-style License. You can find the license here:
 http://www.informatica.info/projects/css/LICENSE.txt

 */

package info.informatica.doc.style.css;

import org.apache.log4j.Logger;
import org.w3c.css.sac.Parser;
import org.w3c.dom.DOMException;

/**
 * SAC Parser Factory.
 * <p>
 * 
 * @author Carlos Amengual (amengual at informatica.info)
 * 
 */
public class SACParserFactory {

	public static final String DEFAULT_PARSER = "org.apache.batik.css.parser.Parser";

	static Logger log = Logger.getLogger(SACParserFactory.class.getName());

	public SACParserFactory() {
		super();
	}

	/**
	 * Create a SAC Parser specified by the system property
	 * <code>org.w3c.css.sac.parser</code>.
	 * <p>
	 * By default, uses the Batik SAC Parser.
	 * 
	 * @return the SAC parser.
	 * @throws DOMException NOT_SUPPORTED_ERR
	 *             if the Parser could not be instantiated.
	 */
	public static Parser createSACParser() throws DOMException {
		String parserClass = null;
		try {
			parserClass = java.security.AccessController
			.doPrivileged(new java.security.PrivilegedAction<String>() {
				public String run() {
					return System.getProperty("org.w3c.css.sac.parser");
				}
			});
		}catch(SecurityException e) {
			log.warn("Unable to read system property org.w3c.css.sac.parser", e);
		}
		if (parserClass != null) {
			if (log.isDebugEnabled()) {
				log.debug("Instantiating SAC parser " + parserClass);
			}
			try {
				return instantiateParser(parserClass);
			} catch (Exception e) {
				log.error("Could not instantiate chosen SAC parser", e);
				log.error("Instantiating default SAC parser" + DEFAULT_PARSER);
			}
		}
		Parser parser;
		try {
			parser = instantiateParser(DEFAULT_PARSER);
		} catch (Exception e) {
			log.fatal("Could not instantiate SAC parser" + DEFAULT_PARSER);
			throw new DOMException(DOMException.NOT_SUPPORTED_ERR, 
					"Could not instantiate SAC parser" + DEFAULT_PARSER);
		}
		return parser;
	}

	private static Parser instantiateParser(String parserClass) 
	throws InstantiationException, IllegalAccessException, ClassNotFoundException {
		return (Parser) Class.forName(parserClass).newInstance();
	}
}
