/*

 Copyright (c) 2005-2011, Carlos Amengual.

 Licensed under a BSD-style License. You can find the license here:
 http://www.informatica.info/projects/css/LICENSE.txt

 */

package info.informatica.doc.style.css.dom;

import info.informatica.doc.style.css.CSSMediaException;
import info.informatica.doc.style.css.CSSStyleSheetFactory;
import info.informatica.doc.style.css.SACParserFactory;
import info.informatica.doc.style.css.dom.DOMCSSStyleRule.Specifity;

import java.io.IOException;
import java.io.Reader;
import java.io.StringReader;
import java.net.URL;
import java.util.Iterator;
import java.util.SortedSet;
import java.util.TreeSet;

import org.apache.log4j.Logger;
import org.w3c.css.sac.CSSException;
import org.w3c.css.sac.DocumentHandler;
import org.w3c.css.sac.InputSource;
import org.w3c.css.sac.LexicalUnit;
import org.w3c.css.sac.Parser;
import org.w3c.css.sac.SACMediaList;
import org.w3c.css.sac.SelectorList;
import org.w3c.dom.DOMException;
import org.w3c.dom.Node;
import org.w3c.dom.css.CSSMediaRule;
import org.w3c.dom.css.CSSRule;
import org.w3c.dom.css.CSSRuleList;
import org.w3c.dom.css.CSSStyleSheet;
import org.w3c.dom.stylesheets.MediaList;
import org.w3c.dom.stylesheets.StyleSheet;

/**
 * CSS Style Sheet Object Model implementation base class.
 * 
 * @author Carlos Amengual (amengual at informatica.info)
 * 
 */
public class BaseCSSStyleSheet implements CSSStyleSheet {

	private CSSStyleSheetFactory styleSheetFactory;

	private String namespaceUri = null;
	
	private StyleSheet parent = null;

	private String href = null;

	private String advisoryTitle = null;

	private CSSRule ownerRule = null;

	protected CSSRuleArrayList cssRules = new CSSRuleArrayList();

	protected int currentInsertionIndex = 0;

	private DOMMediaList destinationMedia;
	
	protected String targetMedium = null;

	private boolean disabled = false;

	static Logger log = Logger.getLogger(BaseCSSStyleSheet.class.getName());

	protected BaseCSSStyleSheet(CSSStyleSheetFactory factory, MediaList media,
			CSSRule ownerRule) {
		super();
		this.styleSheetFactory = factory;
		this.ownerRule = ownerRule;
		this.destinationMedia = new MyDOMMediaList(media);
		if(destinationMedia.getLength() == 1 && !destinationMedia.isAllMedia()) {
			// We target only one specific media
			targetMedium = destinationMedia.item(0);
		}
	}

	protected BaseCSSStyleSheet(CSSStyleSheetFactory factory, MediaList media) {
		this(factory, media, null);
	}

	public CSSStyleSheetFactory getStyleSheetFactory() {
		return styleSheetFactory;
	}

	public CSSRule getOwnerRule() {
		return ownerRule;
	}

	public Node getOwnerNode() {
		return null;
	}

	public CSSRuleList getCssRules() {
		return cssRules;
	}

    /**
     * Used to insert a new rule into the style sheet. The new rule now 
     * becomes part of the cascade.
     * 
     * @param rule The parsable text representing the rule. For rule sets 
     *   this contains both the selector and the style declaration. For 
     *   at-rules, this specifies both the at-identifier and the rule 
     *   content. 
     * @param index The index within the style sheet's rule list of the rule 
     *   before which to insert the specified rule. If the specified index 
     *   is equal to the length of the style sheet's rule collection, the 
     *   rule will be added to the end of the style sheet. 
     * @return  The index within the style sheet's rule collection of the 
     *   newly inserted rule. 
     * @throws DOMException
     *   HIERARCHY_REQUEST_ERR: Raised if the rule cannot be inserted at the 
     *   specified index e.g. if an <code>@import</code> rule is inserted 
     *   after a standard rule set or other at-rule.
     *   <br>INDEX_SIZE_ERR: Raised if the specified index is not a valid 
     *   insertion point.
     *   <br>NO_MODIFICATION_ALLOWED_ERR: Raised if this style sheet is 
     *   readonly.
     *   <br>SYNTAX_ERR: Raised if the specified rule has a syntax error and 
     *   is unparsable.
     */
	public int insertRule(String rule, int index) throws DOMException {
		if(index > getCssRules().getLength() || index < 0) {
			throw new DOMException(DOMException.INDEX_SIZE_ERR,
					"Invalid index: " + index);
		}
		// XXX The following may cause an (undocumented) DOMException.NOT_SUPPORTED_ERR
		Parser psr = SACParserFactory.createSACParser();
		InputSource source = new InputSource();
		Reader re = new StringReader(rule);
		source.setCharacterStream(re);
		psr.setDocumentHandler(createDocumentHandler());
		currentInsertionIndex = index - 1;
		try {
			psr.parseRule(source);
		} catch (CSSException e) {
			throw new DOMException(DOMException.SYNTAX_ERR, e
					.getMessage());
		} catch (IOException e) {
			// This should never happen!
			throw new DOMException(DOMException.INVALID_STATE_ERR, e
					.getMessage());
		}
		return currentInsertionIndex;
	}

	/**
	 * Inserts a rule in the current insertion point (generally after the last
	 * rule).
	 * 
	 * @param cssrule
	 *            the rule to be inserted.
	 */
	public void addRule(BaseCSSRule cssrule) {
		cssrule.setParentStyleSheet(this);
		addLocalRule(cssrule);
	}

	/**
	 * Inserts a local rule in the current insertion point (generally after the last
	 * rule).
	 * 
	 * @param cssrule
	 *            the rule to be inserted.
	 */
	private void addLocalRule(CSSRule cssrule) {
		currentInsertionIndex = cssRules.insertRule(cssrule,
				++currentInsertionIndex);
	}

    /**
     * Deletes a rule from the style sheet.
     * 
     * @param index The index within the style sheet's rule list of the rule 
     *   to remove. 
     * @throws DOMException
     *   INDEX_SIZE_ERR: Raised if the specified index does not correspond to 
     *   a rule in the style sheet's rule list.
     *   <br>NO_MODIFICATION_ALLOWED_ERR: Raised if this style sheet is 
     *   readonly.
     */
	public void deleteRule(int index) throws DOMException {
		try {
			cssRules.remove(index);
		} catch(IndexOutOfBoundsException e) {
			throw new DOMException(DOMException.INDEX_SIZE_ERR, e.getMessage());
		}
	}
	
	public void addStyleSheet(BaseCSSStyleSheet sheet) {
		CSSRuleList otherRules = sheet.getCssRules();
		int orl = otherRules.getLength();
		for(int i=0; i<orl; i++) {
			addRule((BaseCSSRule) otherRules.item(i));
		}
	}

	public String getType() {
		return "text/css";
	}

	/**
	 * Gets the namespace URI to which this style applies.
	 * 
	 * @return the namespace URI string.
	 */
	public String getNamespaceURI() {
		return namespaceUri;
	}

	/**
	 * Sets the namespace URI to which this style applies.
	 * 
	 * @param uri
	 *            the namespace URI.
	 */
	public void setNamespaceURI(String uri) {
		namespaceUri = uri;
	}

	public boolean getDisabled() {
		return disabled;
	}

	public void setDisabled(boolean disabled) {
		this.disabled = disabled;
	}

	public StyleSheet getParentStyleSheet() {
		return parent;
	}

	public void setParentStyleSheet(StyleSheet parent) {
		this.parent = parent;
	}

	public String getHref() {
		return href;
	}

	public void setHref(String href) {
		this.href = href;
	}

	/**
	 * Gets the base URL for the source document of this sheet
	 * @return the base URL, or null if this style sheet is not embedded 
	 * into a source document (i.e. a standalone sheet), or is not known.
	 */
	public URL getDocumentBaseURL() {
		return null;
	}

	/**
	 * Gets the advisory title.
	 * 
	 * @return the title.
	 */
	public String getTitle() {
		return advisoryTitle;
	}

	/**
	 * Sets the advisory title.
	 * 
	 * @param title
	 *            the title.
	 */
	public void setTitle(String title) {
		advisoryTitle = title;
	}

	/**
	 * Sets the medium that is expected by the user agent, 
	 * and will be used for computed styles.<p>
	 * Must be one of the mediums in the media list returned by <code>getMedia()</code>.
	 * 
	 * @param medium the name of the medium, like 'screen' or 'print'.
	 */
	public void setTargetMedium(String medium) 
	throws CSSMediaException {
		medium = medium.intern();
		if("all".equals(medium)) {
			targetMedium = null;
		} else {
			if(!destinationMedia.match(medium)) {
				throw new CSSMediaException(
						"Style sheet does not match the medium " + medium);
			}
			targetMedium = medium;
		}
	}

	/**
	 * Gets the target medium for this sheet.
	 * 
	 * @return the taget medium, or null if has not been set.
	 */
	public String getTargetMedium() {
		return targetMedium;
	}

	public MediaList getMedia() {
		return destinationMedia;
	}

	/**
	 * Creates a SAC document handler implemented by this style sheet.
	 * 
	 * @return the new SAC document handler.
	 */
	DocumentHandler createDocumentHandler() {
		return new CSSDocumentHandler();
	}

	@Override
	public String toString() {
		return getCssRules().toString();
	}
	
	/**
	 * Compute the style for an element.
	 * 
	 * @param style a base, empty style to be filled with the computed style.
	 * @param matcher the selector matcher.
	 * @param inlineStyle the inline style for the element.
	 * @param pseudoElt the pseudo-element.
	 * @return the computed CSS style, or an empty style declaration if none applied
	 * or the sheet is disabled.
	 */
	protected ComputedCSSStyle computeStyle(ComputedCSSStyle style,
			SelectorMatcher matcher, ComputedCSSStyle inlineStyle,
			String pseudoElt) {
		// This check for the disabled attribute is required for spec compliance.
		if(disabled) {
			return style;
		}
		// Set the pseudo-element
		matcher.setPseudoElement(pseudoElt);
		/*
		 * We build a sorted set of styles that apply to the given element.
		 */
		SortedSet<DOMCSSStyleRule.Specifity> matchingStyles = 
			new TreeSet<DOMCSSStyleRule.Specifity>(
				new DOMCSSStyleRule.SpecificityComparator());
		Iterator<CSSRule> it = cssRules.iterator();
		while (it.hasNext()) {
			CSSRule rule = it.next();
			if (!(rule instanceof DOMCSSStyleRule)) {
				if(rule instanceof DOMCSSMediaRule) {
					DOMCSSMediaRule mediaRule = (DOMCSSMediaRule)rule;
					DOMMediaList mediaList = (DOMMediaList) mediaRule.getMedia();
					// If we target a specific media, account for matching @media rules, 
					// otherwise ignore them.
					if(targetMedium != null && mediaList.match(targetMedium)) {
						CSSRuleList ruleList = mediaRule.getCssRules();
						int rll = ruleList.getLength();
						for(int i=0; i<rll; i++) {
							if(ruleList.item(i) instanceof DOMCSSStyleRule) {
								DOMCSSStyleRule stylerule = (DOMCSSStyleRule) ruleList.item(i);
								int selIdx = matcher.match(stylerule.getSelectorList());
								if (selIdx >= 0) {
									matchingStyles.add(stylerule.getSpecifity(selIdx));
								}
							}
						}
					}
				} else {
					log.info("While computing style, found rule of type "
							+ rule.getType());
				}
				continue;
			}
			DOMCSSStyleRule stylerule = (DOMCSSStyleRule) rule;
			int selIdx = matcher.match(stylerule.getSelectorList());
			if (selIdx >= 0) {
				matchingStyles.add(stylerule.getSpecifity(selIdx));
			}
		}
		/*
		 * The styles are sorted according to its specificity, per the
		 * SpecificityComparator.
		 */
		Iterator<Specifity> styleit = matchingStyles.iterator();
		if (!styleit.hasNext()) {
			// Cascade is empty
			if(log.isDebugEnabled()){
				log.debug("Could not find styles matching selector "
						+ matcher.toString());
			}
			// No styles match!
			if(inlineStyle != null) {
				// We just have the inline style
				return inlineStyle;
			} else {
				// Return an empty style
				return style;
			}
		}
		/*
		 * Now we add all the styles to form a single declaration. We add them
		 * according to the order specified by the sorted set.
		 * 
		 * Each more specific style is added, starting with the less specific
		 * declaration.
		 */
		while (styleit.hasNext()) {
			style.addStyle((BaseCSSStyleDeclaration) styleit.next()
					.getCSSStyleRule().getStyle());
		}
		// The inline style has higher priority, so we add it at the end.
		if (inlineStyle != null) {
			style.addStyle(inlineStyle);
		}
		return style;
	}
	
	private class MyDOMMediaList extends DOMMediaList {

		private MyDOMMediaList(MediaList list) {
			super(list);
		}

		@Override
		public void setMediaText(String mediaText) throws DOMException {
			throw new DOMException(DOMException.NO_MODIFICATION_ALLOWED_ERR, 
			"Cannot modify target media: you must re-create the style sheet with a different media list.");
		}

		@Override
		public void appendMedium(String newMedium) throws DOMException {
			throw new DOMException(DOMException.NO_MODIFICATION_ALLOWED_ERR, 
			"Cannot modify target media: you must re-create the style sheet with a different media list.");
		}

		@Override
		public void deleteMedium(String oldMedium) throws DOMException {
			throw new DOMException(DOMException.NO_MODIFICATION_ALLOWED_ERR, 
			"Cannot modify target media: you must re-create the style sheet with a different media list.");
		}

	}

	protected static Parser createSACParser() throws DOMException {
		return SACParserFactory.createSACParser();
	}

	/**
	 * Parses a style sheet.
	 * 
	 * If the style sheet is not empty, the rules from the parsed source will 
	 * be added at the end of the rule list.
	 * @param source
	 *            the SAC input source.
	 * @throws DOMException if a DOM problem is found parsing the sheet.
	 * @throws CSSException if a non-DOM problem is found parsing the sheet.
	 * @throws IOException if a problem is found reading the sheet.
	 * @see info.informatica.doc.style.css.CSSStyleSheetFactory#createStyleSheet
	 */
	public void parseCSSStyleSheet(
			InputSource source) throws DOMException, IOException {
		Parser parser = createSACParser();
		parser.setDocumentHandler(createDocumentHandler());
		parser.parseStyleSheet(source);
	}

	class CSSDocumentHandler implements DocumentHandler {

		private BaseCSSRule currentRule = null;

		// switch for ignoring rules based on target media
		private boolean ignoreRulesForMedia = false;
		
		private boolean ignoreImports = false;

		CSSDocumentHandler() {
			super();
		}

		public void startDocument(InputSource source) throws CSSException {
			currentRule = null;
			ignoreRulesForMedia = false;
			ignoreImports = false;
			log.debug("Starting StyleSheet processing");
		}

		public void endDocument(InputSource source) throws CSSException {
			log.debug("Ending StyleSheet processing");
		}

		public void comment(String text) throws CSSException {

		}

		public void ignorableAtRule(String atRule) throws CSSException {
			log.debug("Ignorable @-rule: " + atRule);
		}

		public void namespaceDeclaration(String prefix, String uri)
				throws CSSException {
			if (log.isDebugEnabled()) {
				log.debug("Setting namespace uri: " + uri);
			}
			setNamespaceURI(uri);
		}

		public void importStyle(String uri, SACMediaList media,
				String defaultNamespaceURI) throws CSSException, DOMException {
			// Ignore any '@import' rule that occurs inside a block or after any 
			// non-ignored statement other than an @charset or an @import rule (CSS 2.1 §4.1.5)
			if(ignoreImports) {
				log.debug("Ignoring @import from " + uri
						+ ": must be at the beginning of the style sheet (CSS 2.1 §4.1.5)");
				return;
			}
			if (destinationMedia.match(media)) {
				if (log.isDebugEnabled()) {
					log.debug("Importing rule from uri: " + uri);
				}
				DOMCSSImportRule imp = new DOMCSSImportRule(
						BaseCSSStyleSheet.this);
				try {
					imp.loadStyleSheet(uri, getTitle(), media);
				} catch (CSSException e) {
					throw new CSSException(e);
				} catch (IOException e) {
					throw new CSSException(e);
				}
			} else {
				log.debug("Ignoring @import from " + uri
						+ ": target media mismatch");
			}
		}

		public void startMedia(SACMediaList media) throws CSSException {
			ignoreImports = true;
			if (media.getLength() > 0) {
				if(targetMedium == null) {
					if(destinationMedia.match(media)) { // Insert media rules into this sheet
						currentRule = new DOMCSSMediaRule(BaseCSSStyleSheet.this);
						int sz = media.getLength();
						for (int i = 0; i < sz; i++) {
							((CSSMediaRule) currentRule).getMedia().appendMedium(
									media.item(i));
						}
						if (log.isDebugEnabled()) {
							log.debug("Starting @media block for " + media.toString());
						}
					} else {
						if (log.isDebugEnabled()) {
							log.debug("Ignoring @media: target media mismatch. Expected: "
											+ destinationMedia.getMediaText());
						}
						ignoreRulesForMedia = true;
					}
				} // else: Insert styles from matching media rules as if they were top-level rules
			} else {
				log.warn("@media rule with empty media list.");
				ignoreRulesForMedia = true;
			}
		}

		public void endMedia(SACMediaList media) throws CSSException {
			if (ignoreRulesForMedia) {
				ignoreRulesForMedia = false;
			} else if(targetMedium == null) {
				if (log.isDebugEnabled()) {
					log.debug("Inserting @media rule "
							+ currentRule.getCssText() + " into sheet");
				}
				addLocalRule(currentRule);
			}
			currentRule = null;
		}

		public void startPage(String name, String pseudo_page)
				throws CSSException {
			ignoreImports = true;
			if (!ignoreRulesForMedia) {
				currentRule = new DOMCSSPageRule(BaseCSSStyleSheet.this);
			}
		}

		public void endPage(String name, String pseudo_page)
				throws CSSException {
			if (!ignoreRulesForMedia) {
				if (log.isDebugEnabled()) {
					log.debug("Inserting @page rule "
							+ currentRule.getCssText() + " into sheet");
				}
				addLocalRule(currentRule);
				currentRule = null;
			} else {
				log.debug("Ignored @page: target media mismatch");
			}
		}

		public void startFontFace() throws CSSException {
			ignoreImports = true;
			if (!ignoreRulesForMedia) {
				currentRule = new DOMCSSFontFaceRule(BaseCSSStyleSheet.this);
			} else {
				log.debug("Ignoring @font-face: target media mismatch");
			}
		}

		public void endFontFace() throws CSSException {
			if (!ignoreRulesForMedia) {
				if (log.isDebugEnabled()) {
					log.debug("Inserting @font-face rule "
							+ currentRule.getCssText() + " into sheet");
				}
				addLocalRule(currentRule);
				currentRule = null;
			}
		}

		public void startSelector(SelectorList selectors) throws CSSException {
			ignoreImports = true;
			if (!ignoreRulesForMedia) {
				DOMCSSStyleRule styleRule = new DOMCSSStyleRule(BaseCSSStyleSheet.this);
				if (currentRule != null) {
					styleRule.setParentRule(currentRule);
				}
				currentRule = styleRule;
				((CSSStyleDeclarationRule) currentRule)
						.setSelectorList(selectors);
				if (log.isDebugEnabled()) {
					String[] selectorTypes = { "CONDITIONAL SELECTOR: ",
							"ANY NODE SELECTOR: ", "ROOT NODE SELECTOR: ",
							"NEGATIVE SELECTOR: ", "ELEMENT NODE SELECTOR: ",
							"TEXT NODE SELECTOR: ",
							"CDATA SECTION NODE SELECTOR: ",
							"PROCESSING INSTRUCTION NODE SELECTOR: ",
							"COMMENT NODE SELECTOR: ",
							"PSEUDO ELEMENT SELECTOR: ",
							"DESCENDANT SELECTOR: ", "CHILD SELECTOR: ",
							"DIRECT ADJACENT SELECTOR: " };
					StringBuilder sb = new StringBuilder(36).append(
							"Found selectors: ").append(
							selectorTypes[selectors.item(0).getSelectorType()])
							.append(selectors.item(0));
					int sz = selectors.getLength();
					for (int i = 1; i < sz; i++) {
						sb.append(',').append(' ').append(
								selectorTypes[selectors.item(0)
										.getSelectorType()]).append(
								selectors.item(i));
					}
					log.debug(sb.toString());
				}
			} else {
				if (log.isDebugEnabled()) {
					log.debug("Ignoring rule for " + selectors.toString()
							+ ": target media mismatch");
				}
			}
		}

		public void endSelector(SelectorList selectors) throws CSSException {
			if (!ignoreRulesForMedia && currentRule instanceof DOMCSSStyleRule) {
				BaseCSSRule pRule = (BaseCSSRule) currentRule.getParentRule();
				if (((DOMCSSStyleRule) currentRule).getStyle().getLength() == 0) {
					if (log.isDebugEnabled()) {
						log.debug("Discarding empty Style rule for selector "
								+ ((DOMCSSStyleRule) currentRule)
										.getSelectorText());
					}
				} else {
					if (pRule == null) {
						if (log.isDebugEnabled()) {
							log.debug("Inserting rule " + currentRule.getCssText()
									+ " into sheet");
						}
						addLocalRule(currentRule);
					} else {
						((DOMCSSMediaRule)pRule).addRule(currentRule);
					}
				}
				currentRule = pRule;
			}
		}

		public void property(String name, LexicalUnit value, boolean important)
				throws CSSException {
			if (!ignoreRulesForMedia) {
				String importantString = null;
				if (important) {
					importantString = "important";
				} else {
					importantString = "";
				}
				((BaseCSSStyleDeclaration) ((CSSStyleDeclarationRule) currentRule)
						.getStyle()).setProperty(name, value, importantString);
			} else {
				if (log.isDebugEnabled()) {
					log.debug("Ignoring property " + name
							+ ": target media mismatch");
				}
			}
		}
	}
}
