/*

 Copyright (c) 2005-2008, Carlos Amengual.

 Licensed under a BSD-style License. You can find the license here:
 http://www.informatica.info/projects/css/LICENSE.txt

 */

package info.informatica.doc.style.css.dom;

import info.informatica.doc.style.css.CSS2ComputedProperties;
import info.informatica.doc.style.css.StyleDatabaseAware;
import info.informatica.doc.style.css.StyleDeclarationFactory;
import info.informatica.doc.style.css.property.CSSNumberValue;
import info.informatica.doc.style.css.property.CSSPercentageValue;
import info.informatica.doc.style.css.property.CSSPropertyException;
import info.informatica.doc.style.css.property.CSSStringValue;
import info.informatica.doc.style.css.property.PropertyDatabase;

import java.net.MalformedURLException;
import java.net.URL;

import org.w3c.css.sac.LexicalUnit;
import org.w3c.dom.DOMException;
import org.w3c.dom.Node;
import org.w3c.dom.css.CSSPrimitiveValue;
import org.w3c.dom.css.CSSValue;

/**
 * Style declaration that computes CSS properties.
 * <p>
 * See section 6.1 of the Document Object Model CSS spec.
 * </p>
 * <p>
 * Some of the methods require that a style database is set, for example to
 * verify the availability of font families.
 * </p>
 * 
 * @author Carlos Amengual (amengual at informatica.info)
 * 
 */
abstract public class ComputedCSSStyle extends BaseCSSStyleDeclaration
		implements CSS2ComputedProperties, StyleDatabaseAware {

	private Node node = null;

	protected ComputedCSSStyle() {
		super();
	}

	protected ComputedCSSStyle(BaseCSSStyleDeclaration copiedObject) {
		super(copiedObject);
	}

	public void setPeerNode(Node node) {
		this.node = node;
	}

	public Node getPeerNode() {
		return node;
	}

	/**
	 * Gets the absolute, primitive "computed" value for the given property.
	 * <p>
	 * The rendering context is not taken into account for this method.
	 * <p>
	 * See paragraph 6.1.2 of the Document Object Model CSS specification for
	 * the definition of "computed" values.
	 * 
	 * @param property
	 *            the property that we want to evaluate.
	 * @return the primitive value of the property, a CSSShorthandValue if the 
	 *         property is a shorthand, or null if the property is not known.
	 */
	@Override
	public CSSValue getCSSValue(String property) {
		CSSValue value = super.getCSSValue(property);
		boolean inherited = PropertyDatabase.getInstance()
				.isInherited(property);
		CSS2ComputedProperties ancStyle = this;
		/*
		 * We compute inherited value, if appropriate.
		 */
		while (value == null ? inherited
				: value.getCssValueType() == CSSValue.CSS_INHERIT) {
			ancStyle = ancStyle.getParentComputedStyle();
			if (ancStyle == null) {
				break;
			}
			value = ancStyle.getPropertyCSSValue(property);
		}
		// Still inheriting ?
		if (value != null && value.getCssValueType() == CSSValue.CSS_INHERIT) {
			value = null;
		}
		// Check for null, and apply initial values if appropriate
		if (value == null) {
			value = defaultPropertyValue(property);
		}
		// If value is null now, we have no idea about this property's value
		if (value != null && value instanceof StyleDatabaseAware) {
			// Set the style database
			((StyleDatabaseAware) value).setStyleDatabase(getStyleDatabase());
			if(property.equalsIgnoreCase("font-family")) {
				CSSStringValue fontFamily = (CSSStringValue)value;
				ancStyle = this;
				LexicalUnit lu = fontFamily.getLexicalUnit();
				String requestedFamily;
				if(lu != null) {
					requestedFamily = lu.getStringValue();
				} else {
					requestedFamily = fontFamily.getStringValue();
				}
				while (!getStyleDatabase().isFontFamilyAvailable(requestedFamily)) {
					fontFamily = (CSSStringValue) fontFamily.nextPrimitiveValue();
					if (fontFamily == null) {
						ancStyle = ancStyle.getParentComputedStyle();
						while(ancStyle != null) {
							fontFamily = (CSSStringValue) ((BaseCSSStyleDeclaration)ancStyle).getDeclaredCSSValue(property);
							if(fontFamily != null) {
								break;
							}
						}
						if(fontFamily == null) {
							break;
						}
					}
					requestedFamily = fontFamily.getLexicalUnit().getStringValue();
				}
				if (fontFamily == null) {
					requestedFamily = getStyleDatabase().getDefaultGenericFontFamily();
				}
				value = new CSSStringValue();
				((CSSStringValue)value).setStringValue(CSSPrimitiveValue.CSS_STRING, requestedFamily);
			} else {
				// Computed values of some properties are constrained by others
				value = applyConstrains(property, value);
			}
		}
		return value;
	}

	private CSSValue applyConstrains(String property, CSSValue value) {
		CSSValue computedValue = value;
		if ("display".equals(property)) {
			// CSS spec, sect. 9.7
			String strVal = ((CSSPrimitiveValue) value).getStringValue();
			if (!"none".equals(strVal)) {
				String position = ((CSSPrimitiveValue) getCSSValue("position"))
						.getStringValue();
				if ("absolute".equals(position) || "fixed".equals(position)) {
					computedValue = computeConstrainedDisplay(value);
				} else {
					String floatProp = ((CSSPrimitiveValue) getCSSValue("float"))
							.getStringValue();
					if (!"none".equals(floatProp)) {
						computedValue = computeConstrainedDisplay(value);
					} else {
						// Is this root element ?
						// TODO
						// if(yes) {computedValue =
						// computeConstrainedDisplay(value);}
					}
				}

			}
		}
		return computedValue;
	}

	/**
	 * Table of computed values of 'display' property, per CSS spec, sect. 9.7.
	 * 
	 * @param value
	 * @return
	 */
	private CSSValue computeConstrainedDisplay(CSSValue value) {
		String display = ((CSSPrimitiveValue) value).getStringValue();
		if ("inline-table".equals(display)) {
			return StyleDeclarationFactory.parseProperty("table");
		} else if ("inline".equals(display) || "run-in".equals(display)
				|| "table-row-group".equals(display)
				|| "table-column".equals(display)
				|| "table-column-group".equals(display)
				|| "table-header-group".equals(display)
				|| "table-footer-group".equals(display)
				|| "table-row".equals(display) || "table-cell".equals(display)
				|| "table-caption".equals(display)
				|| "inline-block".equals(display)) {
			return StyleDeclarationFactory.parseProperty("block");
		}
		return value;
	}

	@Override
	public CSSPrimitiveValue getColor() {
		return (CSSPrimitiveValue) getCSSValue("color");
	}

	public CSSPrimitiveValue getBackgroundColor() {
		return (CSSPrimitiveValue) getCSSValue("background-color");
	}

	public String getBackgroundImage() {
		CSSPrimitiveValue cssVal = ((CSSPrimitiveValue) getCSSValue("background-image"));
		if(cssVal == null) {
			return null;
		} else {
			String imgUri = cssVal.getCssText();
			if(!imgUri.contains("//")) {
				// Relative URL
				URL url;
				try {
					url = new URL(getBaseURL(), imgUri);
				} catch (MalformedURLException e) {
					return imgUri;
				}
				imgUri = url.toExternalForm();
			}
			return imgUri;
		}
	}

	/**
	 * Gets the base URL that should be used to resolve relative URLs 
	 * in property values.
	 * 
	 * @return the base URL, or null if could not be 
	 * determined.
	 */
	public URL getBaseURL() {
		String baseURL = getParentRule().getParentStyleSheet().getHref();
		if(baseURL == null) {
			// Use the base URL from the source document
			return ((BaseCSSStyleSheet)getParentRule().getParentStyleSheet())
				.getDocumentBaseURL();
		} else {
			try {
				return new URL(baseURL);
			} catch (MalformedURLException e) {
				log.error("Bad URL: " + baseURL, e);
				return null;
			}
		}
	}

	/**
	 * Gets the 'used' value for the font-family property.
	 * <p>This method requires a style database.</p>
	 * 
	 * @return the value of the font-family property.
	 * @throws IllegalStateException if the style database has not been set.
	 */
	public String getFontFamily() {
		if(getStyleDatabase() == null){
			throw new IllegalStateException("Style database not set");
		}
		CSSStringValue fontFamily = (CSSStringValue) getCSSValue("font-family");
		return fontFamily.getStringValue();
	}

	/**
	 * Gets the computed font weight.
	 * 
	 * @return the font weight.
	 */
	public String getFontWeight() {
		CSSStringValue fontWeight = (CSSStringValue) getCSSValue("font-weight");
		return fontWeight.getStringValue();
	}

	/**
	 * Gets the computed value of the font-size property.
	 * <p>
	 * May require a style database to work.
	 * </p>
	 * 
	 * @return the value of the font-size property, in typographic 
	 * points.
	 */
	public int getFontSize() {
		CSSPrimitiveValue cssSize = (CSSPrimitiveValue) getCSSValue("font-size");
		int sz = getStyleDatabase().getFontSizeFromIdentifier(null, "medium");
		if (cssSize == null) {
			return sz;
		}
		switch (cssSize.getPrimitiveType()) {
		case CSSPrimitiveValue.CSS_EMS:
			float factor = cssSize.getFloatValue(CSSPrimitiveValue.CSS_EMS);
			// Use parent element's size.
			sz = Math.round(getParentElementFontSize() * factor);
			break;
		case CSSPrimitiveValue.CSS_IDENT:
			String sizeIdentifier = cssSize.getStringValue();
			// relative size: larger, smaller.
			if ("larger".equalsIgnoreCase(sizeIdentifier)) {
				try {
					sz = getLargerFontSize(sz);
				} catch (CSSPropertyException e) {
					log.error("Do not know how to compute larger size");
					log.error(e);
				}
			} else if ("smaller".equalsIgnoreCase(sizeIdentifier)) {
				try {
					sz = getSmallerFontSize(sz);
				} catch (CSSPropertyException e) {
					log.error("Do not know how to compute smaller size");
					log.error(e);
				}
			} else {
				try {
					sz = getStyleDatabase().getFontSizeFromIdentifier(null,
							cssSize.getStringValue());
				} catch (DOMException e) {
					log.error("Unrecognized CSS identifier for element "
							+ getPeerXPath(), e);
				}
			}
			break;
		case CSSPrimitiveValue.CSS_PERCENTAGE:
			float pcnt = cssSize
					.getFloatValue(CSSPrimitiveValue.CSS_PERCENTAGE);
			// Use parent element's size.
			sz = Math.round((float)getParentElementFontSize() * pcnt / 100f);
			break;
		case CSSPrimitiveValue.CSS_PT:
			sz = (int) cssSize.getFloatValue(CSSPrimitiveValue.CSS_PT);
			break;
		}
		return sz;
	}

	protected int getLargerFontSize(int defaultSize)
			throws CSSPropertyException {
		float sz = defaultSize * 1.2f;
		ComputedCSSStyle parentCss = (ComputedCSSStyle) getParentComputedStyle();
		if (parentCss != null) {
			parentCss.setStyleDatabase(getStyleDatabase());
			CSSPrimitiveValue csssize = (CSSPrimitiveValue) parentCss
					.getCSSValue("font-size");
			if (csssize != null) {
				switch (csssize.getPrimitiveType()) {
				case CSSPrimitiveValue.CSS_IDENT:
					String baseFontSize = csssize.getStringValue();
					if (baseFontSize.equals("xx-small")) {
						sz = getStyleDatabase().getFontSizeFromIdentifier(null,
								"x-small");
					} else if (baseFontSize.equals("x-small")) {
						sz = getStyleDatabase().getFontSizeFromIdentifier(null,
								"small");
					} else if (baseFontSize.equals("small")) {
						sz = getStyleDatabase().getFontSizeFromIdentifier(null,
								"medium");
					} else if (baseFontSize.equals("medium")) {
						sz = getStyleDatabase().getFontSizeFromIdentifier(null,
								"large");
					} else if (baseFontSize.equals("large")) {
						sz = getStyleDatabase().getFontSizeFromIdentifier(null,
								"x-large");
					} else if (baseFontSize.equals("x-large")) {
						sz = getStyleDatabase().getFontSizeFromIdentifier(null,
								"xx-large");
					} else if (baseFontSize.equals("xx-large")) {
						sz = 2f * getStyleDatabase().getFontSizeFromIdentifier(
										null, "xx-large")
								- getStyleDatabase().getFontSizeFromIdentifier(
										null, "x-large");
					} else {
						throw new CSSPropertyException(
								"Unknown size identifier " + baseFontSize
										+ " for element "
										+ parentCss.getParentXPath());
					}
					break;
				default:
					sz = parentCss.getFontSize() * 1.2f;
				}
			}
		}
		return Math.round(sz);
	}

	protected int getSmallerFontSize(int defaultSize)
			throws CSSPropertyException {
		float sz = defaultSize * 0.82f;
		ComputedCSSStyle parentCss = (ComputedCSSStyle) getParentComputedStyle();
		if (parentCss != null) {
			parentCss.setStyleDatabase(getStyleDatabase());
			CSSPrimitiveValue csssize = (CSSPrimitiveValue) parentCss
					.getCSSValue("font-size");
			if (csssize != null) {
				switch (csssize.getPrimitiveType()) {
				case CSSPrimitiveValue.CSS_IDENT:
					String baseFontSize = csssize.getStringValue();
					if (baseFontSize.equals("xx-small")) {
						sz = 2f * getStyleDatabase().getFontSizeFromIdentifier(
										null, "xx-small")
								- getStyleDatabase().getFontSizeFromIdentifier(
										null, "x-small");
						// Safety check
						if (sz < 0.1f) {
							sz = getStyleDatabase().getFontSizeFromIdentifier(
									null, "xx-small");
						}
					} else if (baseFontSize.equals("x-small")) {
						sz = getStyleDatabase().getFontSizeFromIdentifier(null,
								"xx-small");
					} else if (baseFontSize.equals("small")) {
						sz = getStyleDatabase().getFontSizeFromIdentifier(null,
								"x-small");
					} else if (baseFontSize.equals("medium")) {
						sz = getStyleDatabase().getFontSizeFromIdentifier(null,
								"small");
					} else if (baseFontSize.equals("large")) {
						sz = getStyleDatabase().getFontSizeFromIdentifier(null,
								"medium");
					} else if (baseFontSize.equals("x-large")) {
						sz = getStyleDatabase().getFontSizeFromIdentifier(null,
								"large");
					} else if (baseFontSize.equals("xx-large")) {
						sz = getStyleDatabase().getFontSizeFromIdentifier(null,
								"x-large");
					} else {
						throw new CSSPropertyException(
								"Unknown size identifier " + baseFontSize
										+ " for element "
										+ parentCss.getParentXPath());
					}
					break;
				default:
					sz = parentCss.getFontSize() * 0.82f;
				}
			}
		}
		return Math.round(sz);
	}

	protected int getParentElementFontSize() {
		int sz = getStyleDatabase().getFontSizeFromIdentifier(null, "medium");
		CSS2ComputedProperties parentCss = getParentComputedStyle();
		if (parentCss != null) {
			sz = parentCss.getFontSize();
		}
		return sz;
	}

	/**
	 * Computes the value, in the device's natural unit, of the given 
	 * number value.
	 * 
	 * @param cssValue the CSS value representing a number.
	 * @return the float value in the device's natural unit, or the 
	 * value of a percentage if the value is a percentage.
	 */
	public float computeFloatValue(CSSNumberValue cssValue) {
		float value;
		if(cssValue instanceof CSSPercentageValue) {
			value = cssValue.getFloatValue(
				CSSPrimitiveValue.CSS_PERCENTAGE);
		} else {
			switch(cssValue.getPrimitiveType()) {
			case CSSPrimitiveValue.CSS_EMS:
				value = getFontSize() * cssValue.getFloatValue(CSSPrimitiveValue.CSS_PT);
			case CSSPrimitiveValue.CSS_EXS:
				value = getStyleDatabase().getExSizeInPt(getFontFamily(), getFontSize()) * 
					cssValue.getFloatValue(CSSPrimitiveValue.CSS_PT);
			default:
				value = ((CSSNumberValue)cssValue).getFloatValue();
			}
		}
		return value;
	}

	abstract public String getPeerXPath();

	abstract public String getParentXPath();

	/**
	 * Gets the computed style for the parent element.
	 * 
	 * @return the computed style for the parent element, or null if there is no
	 *         parent element, or has no style associated.
	 */
	abstract public CSS2ComputedProperties getParentComputedStyle();

	/**
	 * Gets the (whitespace-trimmed) text content of the 
	 * node associated to this style.
	 * 
	 * @return the text content, or the empty string if the box 
	 * has no text.
	 */
	abstract public String getText();
	
	abstract public ComputedCSSStyle clone();

}
