/*

 Copyright (c) 2005-2011, Carlos Amengual.

 Licensed under a BSD-style License. You can find the license here:
 http://www.informatica.info/projects/css/LICENSE.txt

 */

package info.informatica.doc.style.css.dom;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.StringTokenizer;

import org.w3c.css.sac.SACMediaList;
import org.w3c.dom.DOMException;
import org.w3c.dom.stylesheets.MediaList;

/**
 * DOM media list.
 * 
 * @author Carlos Amengual (amengual at informatica.info)
 * 
 */
public class DOMMediaList implements MediaList {

	private List<String> mediaList = new ArrayList<String>(5);
	
	private boolean allMedia;
	
	private static final List<String> mediaTypes;
	
	static {
		String[] mediaTypesArray = { "all", "braille", "embossed",
				"handheld", "print", "projection", "screen", "speech", "tty", "tv" };
		mediaTypes = Arrays.asList(mediaTypesArray);
	}
	
	public DOMMediaList() {
		super();
		allMedia = true;
	}

	public DOMMediaList(String media) {
		super();
		setMediaText(media);
	}

	public DOMMediaList(MediaList list) {
		super();
		allMedia = false;
		int mll = list.getLength();
		for(int i=0; i<mll; i++) {
			addMedium(list.item(i));
		}
	}

	@Override
	public String getMediaText() {
		if(allMedia) {
			return "all";
		}
		StringBuilder sb = new StringBuilder(mediaList.size() * 8 + 2);
		Iterator<String> it = mediaList.iterator();
		if (it.hasNext()) {
			sb.append(it.next());
		}
		while (it.hasNext()) {
			sb.append(',').append(it.next());
		}
		return sb.toString();
	}

	@Override
	public void setMediaText(String mediaText) throws DOMException {
		allMedia = false;
		StringTokenizer st = new StringTokenizer(mediaText, ",");
		while (st.hasMoreElements()) {
			appendMedium(st.nextToken().trim());
		}
	}
	
	@Override
	public int getLength() {
		if(allMedia) {
			return 1;
		}
		return mediaList.size();
	}

	@Override
	public String item(int index) {
		try {
			return mediaList.get(index);
		} catch(IndexOutOfBoundsException e) {
			if (allMedia && index == 0) {
				return "all";
			}
			return null;
		}
	}

	@Override
	public void deleteMedium(String oldMedium) throws DOMException {
		if(!mediaList.remove(oldMedium)) {
			throw new DOMException(DOMException.NOT_FOUND_ERR, 
					oldMedium + " not in media list.");
		}
	}

	@Override
	public void appendMedium(String newMedium) throws DOMException {
		String lcnm = newMedium.toLowerCase(Locale.US);
		addMedium(lcnm);
	}
	
	boolean isValidMedium(String lcmedia) {
		return mediaTypes.contains(lcmedia);
	}

	private void addMedium(String newMedium) {
		if("all".equals(newMedium)) {
			allMedia = true;
			mediaList.clear();
		} else {
			mediaList.add(newMedium.intern());
			allMedia = false;
		}
	}

	/**
	 * Is this an all-media list?
	 * 
	 * @return true if this list matches all media, false otherwise.
	 */
	boolean isAllMedia() {
		return allMedia;
	}

	/**
	 * Does the given SAC media list contain any media present in this list?
	 * 
	 * @param sacMedia
	 *            the SAC media list to test.
	 * @return true if the SAC media contains any media which applies to this
	 *         list, false otherwise.
	 */
	boolean match(SACMediaList sacMedia) {
		if(allMedia) {
			return true;
		}
		int sz = sacMedia.getLength();
		for (int i = 0; i < sz; i++) {
			String iitem = sacMedia.item(i).toLowerCase(Locale.US);
			if (mediaList.contains(iitem) || "all".equals(iitem)) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Does the given DOM media list contain any media present in this list?
	 * 
	 * @param domMedia
	 *            the DOM media list to test.
	 * @return true if the supplied media list contains any media which applies to this
	 *         list, false otherwise.
	 */
	public boolean match(MediaList domMedia) {
		if(allMedia) {
			return true;
		}
		int sz = domMedia.getLength();
		for (int i = 0; i < sz; i++) {
			String iitem = domMedia.item(i).toLowerCase(Locale.US);
			if (mediaList.contains(iitem) || "all".equals(iitem)) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Does this list match the given media?
	 * 
	 * @param media
	 *            the lowercase name of the media to test for.
	 * @return true if this list matches the supplied media name, false otherwise.
	 */
	public boolean match(String media) {
		if(allMedia) {
			return true;
		}
		return mediaList.contains(media);
	}

	/**
	 * Append the contents of the given SAC media list to this one.
	 * 
	 * @param sacMedia
	 *            the SAC media to add.
	 */
	void appendSACMediaList(SACMediaList sacMedia) {
		int sz = sacMedia.getLength();
		for (int i = 0; i < sz; i++) {
			appendMedium(sacMedia.item(i));
		}
	}
	
	@Override
	public String toString() {
		return getMediaText();
	}
}
