/*

 Copyright (c) 2005-2008, Carlos Amengual.

 Licensed under a BSD-style License. You can find the license here:
 http://www.informatica.info/projects/css/LICENSE.txt

 */

package info.informatica.doc.style.css.dom;

import java.util.StringTokenizer;

import org.w3c.css.sac.AttributeCondition;
import org.w3c.css.sac.Condition;
import org.w3c.css.sac.ConditionalSelector;
import org.w3c.css.sac.DescendantSelector;
import org.w3c.css.sac.ElementSelector;
import org.w3c.css.sac.LangCondition;
import org.w3c.css.sac.Selector;
import org.w3c.css.sac.SelectorList;
import org.w3c.css.sac.SiblingSelector;
import org.w3c.css.sac.SimpleSelector;

/**
 * CSS Selector matcher.
 * 
 * @author Carlos Amengual (amengual at informatica.info)
 * 
 */
abstract public class SelectorMatcher {

	private String tagname = null;

	private String parentTagName = null;

	private String classAttr = null;

	private String pseudoElt = null;

	public SelectorMatcher() {
		super();
	}

	/**
	 * Gets the name of the tag to which this selector applies.
	 * 
	 * @return the tagname.
	 */
	public String getTagName() {
		return tagname;
	}

	protected void setClassAttribute(String classAttr) {
		this.classAttr = classAttr;
	}

	protected void setTagname(String tagname) {
		this.tagname = tagname;
	}

	protected void setParentTagname(String parentTagname) {
		this.parentTagName = parentTagname;
	}

	public String getPseudoElement() {
		return pseudoElt;
	}

	public void setPseudoElement(String pseudoElt) {
		this.pseudoElt = pseudoElt;
	}

	/**
	 * Does this selector match the given selector list?
	 * 
	 * @param selist
	 *            the list of selectors to which this matcher will compare.
	 * 
	 * @return the index of the matching selector, or -1 if none matches.
	 */
	public int match(SelectorList selist) {
		int sz = selist.getLength();
		for (int i = 0; i < sz; i++) {
			if (match(selist.item(i))) {
				return i;
			}
		}
		return -1;
	}

	/**
	 * Does this matcher match the given selector?
	 * 
	 * @param selector
	 *            the selector to be tested.
	 * @return true if the given selector matches this object, false otherwise.
	 */
	public boolean match(Selector selector) {
		switch (selector.getSelectorType()) {
		case Selector.SAC_ELEMENT_NODE_SELECTOR:
			String elname = ((ElementSelector) selector).getLocalName();
			if (elname == null || tagname.equals(elname) || elname.equals("*")) {
				return true;
			}
			break;
		case Selector.SAC_CONDITIONAL_SELECTOR:
			return matchConditional((ConditionalSelector) selector);
		case Selector.SAC_CHILD_SELECTOR:
			SimpleSelector desc = ((DescendantSelector) selector).getSimpleSelector();
			if(match(desc)) {
				Selector ancestor = ((DescendantSelector) selector).getAncestorSelector();
				SelectorMatcher parentSM = getParentSelectorMatcher();
				if(parentSM != null && parentSM.match(ancestor)){
					return true;
				}
			}
			break;
		case Selector.SAC_DESCENDANT_SELECTOR:
			desc = ((DescendantSelector) selector).getSimpleSelector();
			if(match(desc)) {
				Selector ancestor = ((DescendantSelector) selector).getAncestorSelector();
				SelectorMatcher parentSM = getParentSelectorMatcher();
				while(parentSM != null) {
					if(parentSM.match(ancestor)){
						return true;
					}
					parentSM = parentSM.getParentSelectorMatcher();
				}
			}
			break;
		case Selector.SAC_PSEUDO_ELEMENT_SELECTOR:
			if (pseudoElt != null
					&& pseudoElt.equals(((ElementSelector)selector)
							.getLocalName())) {
				return true;
			}
			break;
		case Selector.SAC_DIRECT_ADJACENT_SELECTOR:
			if(match(((SiblingSelector) selector).getSiblingSelector())) {
				Selector sel = ((SiblingSelector) selector).getSelector();
				SelectorMatcher siblingSM = getPreviousSiblingSelectorMatcher();
				return siblingSM.match(sel);
			}
		}
		return false;
	}

	boolean matchConditional(ConditionalSelector selector) {
		Condition cond = selector.getCondition();
		switch (cond.getConditionType()) {
		case Condition.SAC_CLASS_CONDITION:
			String cond_value = ((AttributeCondition) cond).getValue();
			if (classAttr != null
					&& match(((ConditionalSelector) selector)
							.getSimpleSelector())) {
				return matchesClass(cond_value);
			}
			break;
		case Condition.SAC_ID_CONDITION:
			String idAttr = getId();
			if (((AttributeCondition) cond).getValue()
							.equals(idAttr)) {
				return true;
			}
			break;
		case Condition.SAC_PSEUDO_CLASS_CONDITION:
			cond_value = ((AttributeCondition) cond).getValue();
			if("first-child".equals(cond_value) && isFirstChild()) {
				SimpleSelector simple = ((ConditionalSelector) 
						selector).getSimpleSelector();
				return match(simple);
			}
			break;
		case Condition.SAC_ATTRIBUTE_CONDITION:
			String attrName = ((AttributeCondition) cond).getLocalName();
			if(hasAttribute(attrName)) {
				if(((AttributeCondition) cond).getSpecified()) {
					String attribValue = getAttributeValue(attrName);
					cond_value = ((AttributeCondition) cond).getValue();
					// SS parser returns getSpecified => true when cond_value is null
					if(cond_value == null) {
						return true;
					} else {
						return attribValue.equals(cond_value);
					}
				} else {
					return true;
				}
			}
			break;
		case Condition.SAC_ONE_OF_ATTRIBUTE_CONDITION:
			attrName = ((AttributeCondition) cond).getLocalName();
			if(hasAttribute(attrName)) {
				String attrValue = getAttributeValue(attrName);
				StringTokenizer tok = new StringTokenizer(attrValue, " ");
				while(tok.hasMoreElements()) {
					String token = tok.nextToken();
					if(token.equals(((AttributeCondition) cond).getValue())) {
						return true;
					}
				}
			}
			break;
		case Condition.SAC_BEGIN_HYPHEN_ATTRIBUTE_CONDITION:
			attrName = ((AttributeCondition) cond).getLocalName();
			if(hasAttribute(attrName)) {
				String attrValue = getAttributeValue(attrName);
				return attrValue.startsWith(((AttributeCondition) cond).getValue());
			}
			break;
		case Condition.SAC_LANG_CONDITION:
			attrName = ((LangCondition) cond).getLang();
			String lang = getLanguage();
			return lang.startsWith(attrName);
		}
		return false;
	}

	/**
	 * Verifies if the selector matches a given class name.
	 * <p>
	 * Per section 4.1.3 of CSS 2.1 spec, identifiers can contain only the characters 
	 * [a-z0-9] and ISO 10646 characters U+00A1 and higher, plus the hyphen (-) and 
	 * the underscore (_). This is confusing, since -for example- both upper and lower 
	 * case accented characters are accepted, but not uppercase ASCII letters. Then it 
	 * states that any ISO 10646 character is accepted if given as a numeric code.
	 * </p>
	 * <p>
	 * A case-insensitive comparison is performed.
	 * </p>
	 * 
	 * @param cond_value the class name.
	 * @return true if matches, false otherwise.
	 */
	private boolean matchesClass(String cond_value) {
		int oiws = 0, iws;
		while((iws = classAttr.indexOf(' ', oiws)) >= 0) {
			if(classAttr.substring(oiws, iws).equalsIgnoreCase(cond_value)) {
				return true;
			}
			oiws = iws + 1;
		}
		if(oiws == 0) {
			return classAttr.equalsIgnoreCase(cond_value);
		} else {
			if(oiws < classAttr.length()) {
				return classAttr.substring(oiws).equalsIgnoreCase(cond_value);
			} else {
				return false;
			}
		}
	}

	/**
	 * Gets the selector matcher for the parent element.
	 * 
	 * @return the selector matcher for the parent element, or null if 
	 * none.
	 */
	abstract public SelectorMatcher getParentSelectorMatcher();

	/**
	 * Gets the selector matcher for the previous sibling.
	 * 
	 * @return the selector matcher for the previous sibling, or null if 
	 * no previous sibling.
	 */
	abstract public SelectorMatcher getPreviousSiblingSelectorMatcher();

	/**
	 * The element in this matcher is the first child?
	 * 
	 * @return true if the element in this matcher is a first child, false if not.
	 */
	abstract public boolean isFirstChild();

	/**
	 * Gets the value of the given attribute in the element associated 
	 * to this selector matcher.
	 * 
	 * @param attrName the attribute name.
	 * @return the attribute value, or the empty string if the 
	 * attribute is defined but has no value, also if the 
	 * attribute is not defined. Never returns null.
	 */
	abstract public String getAttributeValue(String attrName);

	/**
	 * Checks if is defined the given attribute in the element associated 
	 * to this selector matcher.
	 * 
	 * @param attrName the attribute name.
	 * @return true if the attribute is defined, false if not.
	 */
	abstract public boolean hasAttribute(String attrName);

	/**
	 * Gets the 'id' attribute of the element associated 
	 * to this selector matcher.
	 * 
	 * @return the 'id' attribute value, or the empty string if the 
	 * element has no 'id'.
	 */
	abstract public String getId();

	/**
	 * Gets the language of the element associated 
	 * to this selector matcher.
	 * 
	 * @return the language, or the empty String if the 
	 * element has no language defined.
	 */
	/*
	 *  In (X)HTML, the lang attribute contains the language,
	 *  but that may not be true for other XML.
	 */
	abstract public String getLanguage();

	public String toString() {
		StringBuilder sb = new StringBuilder();
		if (parentTagName != null) {
			sb.append(parentTagName).append(' ').append('>').append(' ');
		}
		if (tagname != null) {
			sb.append(tagname);
		}
		if (classAttr != null) {
			sb.append('.').append(classAttr);
		} else if (getId() != "") {
			sb.append('#').append(getId());
		}
		if (pseudoElt != null) {
			sb.append(':').append(pseudoElt);
		}
		return sb.toString();
	}

}
