/*

 Copyright (c) 2005-2011, Carlos Amengual.

 Licensed under a BSD-style License. You can find the license here:
 http://www.informatica.info/projects/css/LICENSE.txt

 */

package info.informatica.doc.style.css.j2d;

import info.informatica.doc.style.css.AbstractStyleDatabase;
import info.informatica.doc.style.css.property.ColorIdentifiers;

import java.awt.Color;
import java.awt.GraphicsConfiguration;
import java.awt.GraphicsEnvironment;

import org.apache.log4j.Logger;
import org.w3c.dom.DOMException;
import org.w3c.dom.css.CSSPrimitiveValue;
import org.w3c.dom.css.RGBColor;

/**
 * CSS style database for use with Java2D objects.
 * <p>
 * 
 * @author Carlos Amengual (amengual at informatica.info)
 * 
 */
public class Java2DStyleDatabase extends AbstractStyleDatabase {
	
	private GraphicsConfiguration gConfiguration = null;
	
	private short naturalUnit = CSSPrimitiveValue.CSS_PX;

	static Logger log = Logger.getLogger(Java2DStyleDatabase.class.getName());

	public Java2DStyleDatabase() {
		super();
	}

	/**
	 * Gets the name of the default font used when a generic font family (serif,
	 * sans-serif, monospace, cursive, fantasy) is specified.
	 * <p>
	 * This class attempts to map the generic name to a "logical font" name.
	 * </p>
	 * <p>
	 * As, in Java, logical font names are internally mapped to physical fonts
	 * by the Java runtime environment, the name of the corresponding "logical
	 * font" is returned, and no further mapping is attempted.
	 * </p>
	 * 
	 * @param genericFamily
	 *            the name of the logical font.
	 * @return the name of the associated logical font, or null if none.
	 */
	public String getDefaultGenericFontFamily(String genericFamily) {
		String fontName = null;
		if (genericFamily.equals("serif")) {
			fontName = "Serif";
		} else if (genericFamily.equals("sans serif")) {
			fontName = "SansSerif";
		} else if (genericFamily.equals("monospace")) {
			fontName = "Monospaced";
		}
		return fontName;
	}

	public boolean isFontFamilyAvailable(String fontFamily) {
		String[] fontNames = GraphicsEnvironment.getLocalGraphicsEnvironment()
				.getAvailableFontFamilyNames();
		for (int i = 0; i < fontNames.length; i++) {
			if (fontNames[i].equalsIgnoreCase(fontFamily)) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Gets the GraphicsConfiguration for this style database.
	 * 
	 * @return the GraphicsConfiguration object previously set, of the default
	 *         one if none was set.
	 * @throws UnsupportedOperationException
	 *             if no appropriate GraphicsConfiguration could be found.
	 */
	protected GraphicsConfiguration getGraphicsConfiguration() {
		if (gConfiguration == null) {
			gConfiguration = GraphicsEnvironment.getLocalGraphicsEnvironment()
					.getDefaultScreenDevice().getDefaultConfiguration();
		}
		return gConfiguration;
	}

	public void setGraphicsConfiguration(GraphicsConfiguration configuration) {
		gConfiguration = configuration;
	}

	@Override
	protected float cmToPixels(float cm) {
		// XXX
		double width;
		double height;
		try {
			GraphicsConfiguration gcf = getGraphicsConfiguration();
			width = gcf.getBounds().getWidth();
			height = gcf.getBounds().getHeight();
		} catch (UnsupportedOperationException e) {
			// Headless device
			width = 1024;
			height = 768;
		}
		double diag = Math.sqrt(width * width + height * height);
		// Assume a screen of 17" = 43cm
		return (float) (cm * diag / 43.18);
	}

	@Override
	protected float pxTocm(int px) {
		// XXX
		double width;
		double height;
		try {
			GraphicsConfiguration gcf = getGraphicsConfiguration();
			width = gcf.getBounds().getWidth();
			height = gcf.getBounds().getHeight();
		} catch (UnsupportedOperationException e) {
			// Headless device
			width = 1024;
			height = 768;
		}
		double diag = Math.sqrt(width * width + height * height);
		// Assume a screen of 17" = 43cm
		return (float) (px * 43.18 / diag);
	}

	/**
	 * This method is used to normalize sizes to a 1024x768 screen.
	 * 
	 * @return the factor by which screensize-dependent quantities can be
	 *         multiplied to be normalized.
	 */
	protected float deviceResolutionFactor() {
		return (float) (getGraphicsConfiguration().getBounds().getHeight() / 768);
	}

	public int getFontSizeFromIdentifier(String familyName,
			String fontSizeIdentifier) throws DOMException {
		// Normalize to 1024x768
		float factor = Math.max(0.9f, deviceResolutionFactor());
		float sz;
		if (fontSizeIdentifier.equals("xx-small")) {
			sz = 8f * factor;
		} else if (fontSizeIdentifier.equals("x-small")) {
			sz = 9f * factor;
		} else if (fontSizeIdentifier.equals("small")) {
			sz = 10f * factor;
		} else if (fontSizeIdentifier.equals("medium")) {
			sz = 12f * factor;
		} else if (fontSizeIdentifier.equals("large")) {
			sz = 14f * factor;
		} else if (fontSizeIdentifier.equals("x-large")) {
			sz = 16f * factor;
		} else if (fontSizeIdentifier.equals("xx-large")) {
			sz = 18f * factor;
		} else {
			throw new DOMException(DOMException.INVALID_ACCESS_ERR,
					"Unknown size identifier: " + fontSizeIdentifier);
		}
		return Math.round(sz);
	}

	public float getWidthSize(String widthIdentifier)
			throws DOMException {
		// Normalize to 1024x768
		float factor = Math.max(0.62f, deviceResolutionFactor());
		if ("thin".equals(widthIdentifier)) {
			return 0.5f * factor;
		} else if ("thick".equals(widthIdentifier)) {
			return 2f * factor;
		} else if ("medium".equals(widthIdentifier)) {
			return 1f * factor;
		} else {
			throw new DOMException(DOMException.SYNTAX_ERR,
					"Unknown identifier " + widthIdentifier);
		}
	}

	public short getNaturalUnit() {
		return naturalUnit;
	}

	/**
	 * Sets the natural unit for this device.
	 * 
	 * @param naturalUnit the natural unit.
	 * @see org.w3c.dom.css.CSSPrimitiveValue
	 */
	public void setNaturalUnit(short naturalUnit) {
		this.naturalUnit = naturalUnit;
	}

	public float getDocumentHeight() {
		return floatValueConversion((float) getGraphicsConfiguration()
				.getBounds().getHeight(), CSSPrimitiveValue.CSS_PX, getNaturalUnit());
	}

	public float getDocumentWidth() {
		return floatValueConversion((float) getGraphicsConfiguration()
				.getBounds().getWidth(), CSSPrimitiveValue.CSS_PX, getNaturalUnit());
	}

	/**
	 * Gets the AWT color as obtained from the given CSS primitive value.
	 * 
	 * @param cssColor
	 *            the primitive color value, which can contain an RGB color, a
	 *            number or an identifier.
	 * @return the AWT color object, or null if the color was specified as an 
	 * unknown identifier or as 'transparent'.
	 */
	public static Color getAWTColor(CSSPrimitiveValue cssColor) {
		Color awtcolor = null;
		if (cssColor != null) {
			switch (cssColor.getPrimitiveType()) {
			case CSSPrimitiveValue.CSS_RGBCOLOR:
				RGBColor color = cssColor.getRGBColorValue();
				CSSPrimitiveValue red = color.getRed();
				CSSPrimitiveValue green = color.getGreen();
				CSSPrimitiveValue blue = color.getBlue();
				switch (red.getPrimitiveType()) {
				case CSSPrimitiveValue.CSS_PERCENTAGE:
					awtcolor = new Color(
							clipColorValue(red
									.getFloatValue(CSSPrimitiveValue.CSS_PERCENTAGE) / 100f),
							clipColorValue(green
									.getFloatValue(CSSPrimitiveValue.CSS_PERCENTAGE) / 100f),
							clipColorValue(blue
									.getFloatValue(CSSPrimitiveValue.CSS_PERCENTAGE) / 100f));
					break;
				case CSSPrimitiveValue.CSS_NUMBER:
					try {
						awtcolor = new Color(
								clipColorValue((int) red
										.getFloatValue(CSSPrimitiveValue.CSS_NUMBER)),
								clipColorValue((int) green
										.getFloatValue(CSSPrimitiveValue.CSS_NUMBER)),
								clipColorValue((int) blue
										.getFloatValue(CSSPrimitiveValue.CSS_NUMBER)));
					} catch (IllegalArgumentException e) {
						log.error("Unknown color: " + cssColor.getStringValue(), e);
					}
				}
				break;
			case CSSPrimitiveValue.CSS_IDENT:
				String sv = cssColor.getStringValue();
				String s = ColorIdentifiers.getInstance().getColor(sv);
				if (s != null) {
					try {
						awtcolor = Color.decode(s);
					} catch (NumberFormatException e) {
						log.error("Unknown color: " + sv, e);
					}
				} else {
					awtcolor = Color.getColor(sv);
					if(awtcolor == null) {
						log.error("Unknown color: " + sv);
					}
				}
				break;
			case CSSPrimitiveValue.CSS_STRING:
				String encoded = cssColor.getStringValue();
				if (encoded.charAt(0) == '#') {
					try {
						awtcolor = Color.decode(encoded);
					} catch (NumberFormatException e) {
						log.error("Unknown color: " + encoded, e);
					}
				} else {
					log.error("Unknown color: " + encoded);
				}
				break;
			case CSSPrimitiveValue.CSS_NUMBER:
				awtcolor = new Color((int) cssColor
						.getFloatValue(CSSPrimitiveValue.CSS_NUMBER));
				break;
			}
		}
		return awtcolor;
	}

	protected static int clipColorValue(int color) {
		return Math.max(Math.min(255, color), 0);
	}

	protected static float clipColorValue(float color) {
		return Math.max(Math.min(1f, color), 0f);
	}

}
