/*

 Copyright (c) 2005-2008, Carlos Amengual.

 Licensed under a BSD-style License. You can find the license here:
 http://www.informatica.info/projects/css/LICENSE.txt

 */

package info.informatica.doc.style.css.property;

import org.w3c.css.sac.LexicalUnit;
import org.w3c.dom.DOMException;
import org.w3c.dom.css.CSSValue;

/**
 * Implementation of CSSValue.
 * 
 * @author Carlos Amengual (amengual at informatica.info)
 * 
 */
public class AbstractCSSValue implements CSSValue {

	private final short valueType;
	
	private String cssText = null;

	protected AbstractCSSValue(short valueType) {
		super();
		this.valueType = valueType;
	}

	public String getCssText() {
		return cssText;
	}

	public void setCssText(String cssText) throws DOMException {
		this.cssText = cssText;
	}

	public short getCssValueType() {
		return valueType;
	}

	/**
	 * Creates a CSSValue according to the given lexical value.
	 * <p>
	 * The value is assumed to be stand-alone, independent of a shorthand 
	 * property.
	 * 
	 * @param lunit the lexical value.
	 * @return a CSSValue associated to the given lexical value.
	 */
	public static CSSValue createCSSValue(LexicalUnit lunit) {
		return createCSSValue(lunit, false);
	}

	/**
	 * Creates a CSSValue according to the given lexical value.
	 * 
	 * @param lunit the lexical value.
	 * @param subproperty true if the value is created under the umbrella of a shorthand set.
	 * @return a CSSValue associated to the given lexical value.
	 */
	public static CSSValue createCSSValue(LexicalUnit lunit, boolean subproperty) {
		short unitType = lunit.getLexicalUnitType();
		CSSValue value = null;
		switch (unitType) {
		case LexicalUnit.SAC_INHERIT:
			value = CSSInheritedValue.getValue();
			break;
		default:
			value = AbstractCSSPrimitiveValue.createCSSPrimitiveValue(lunit);
			((AbstractCSSPrimitiveValue)value).setSubproperty(subproperty);
		}
		return value;
	}
}
