/*

 Copyright (c) 2005-2007, Carlos Amengual.

 Licensed under a BSD-style License. You can find the license here:
 http://www.informatica.info/projects/css/LICENSE.txt

 */

package info.informatica.doc.style.css.property;

import info.informatica.doc.style.css.StyleDatabase;

import org.w3c.css.sac.LexicalUnit;
import org.w3c.dom.DOMException;
import org.w3c.dom.css.CSSPrimitiveValue;

/**
 * Float-specific CSSPrimitiveValue.
 * 
 * @author Carlos Amengual (amengual at informatica.info)
 *
 */
public class CSSNumberValue extends AbstractCSSPrimitiveValue {

	protected float realvalue = 0;

	private String dimensionUnitText = "";

	CSSNumberValue() {
		super();
	}

	@Override
	public String getCssText() {
		double rintValue = Math.rint(realvalue);
		if(realvalue == rintValue){
			return Integer.toString((int)rintValue) + dimensionUnitText;
		} else {
			return Float.toString(realvalue) + dimensionUnitText;
		}
	}

	@Override
	public void setFloatValue(short unitType, float floatValue)
			throws DOMException {
		setCSSUnitType(unitType);
		realvalue = floatValue;
		dimensionUnitText = dimensionUnitString(unitType);
		setCssText(Float.toString(realvalue) + dimensionUnitText);
	}

    /**
     *  Gets a float value in a specified unit. If this 
     * CSS value doesn't contain a float value or can't be converted into 
     * the specified unit, a <code>DOMException</code> is raised. 
     * @param unitType A unit code to get the float value. The unit code can 
     *   only be a float unit type (i.e. <code>CSS_NUMBER</code>, 
     *   <code>CSS_PERCENTAGE</code>, <code>CSS_EMS</code>, 
     *   <code>CSS_EXS</code>, <code>CSS_PX</code>, <code>CSS_CM</code>, 
     *   <code>CSS_MM</code>, <code>CSS_IN</code>, <code>CSS_PT</code>, 
     *   <code>CSS_PC</code>, <code>CSS_DEG</code>, <code>CSS_RAD</code>, 
     *   <code>CSS_GRAD</code>, <code>CSS_MS</code>, <code>CSS_S</code>, 
     *   <code>CSS_HZ</code>, <code>CSS_KHZ</code>, 
     *   <code>CSS_DIMENSION</code>). 
     * @return  The float value in the specified unit. 
     * @throws DOMException
     *   INVALID_ACCESS_ERR if the CSS value can't be converted into 
     *   the specified unit. 
     */
	@Override
	public float getFloatValue(short unitType) throws DOMException {
		if(unitType == getPrimitiveType() || unitType == CSSPrimitiveValue.CSS_NUMBER){
			return realvalue;
		} else {
			switch(getPrimitiveType()){
			case CSSPrimitiveValue.CSS_MS:
				if(unitType == CSSPrimitiveValue.CSS_S){
					return realvalue * 1000f; 
				}
				break;
			case CSSPrimitiveValue.CSS_DEG:
				if(unitType == CSSPrimitiveValue.CSS_RAD){
					return (float) Math.toRadians(realvalue);
				} else if(unitType == CSSPrimitiveValue.CSS_GRAD){
					return realvalue * 1.1111111111f;
				}
				break;
			case CSSPrimitiveValue.CSS_RAD:
				if(unitType == CSSPrimitiveValue.CSS_DEG){
					return (float) Math.toDegrees(realvalue);
				} else if(unitType == CSSPrimitiveValue.CSS_GRAD){
					return realvalue * 63.6619772368f;
				}
				break;
			}
			StyleDatabase deviceStyleDb = getStyleDatabase();
			if(deviceStyleDb == null){
				throw new DOMException(DOMException.INVALID_ACCESS_ERR, 
						"Cannot transform unit to " + dimensionUnitString(unitType));
			}
			return deviceStyleDb.floatValueConversion(realvalue, getPrimitiveType(), unitType);
		}
	}

	/**
	 * Gets a float value in the device's natural unit, as defined 
	 * by the style database.
	 * 
	 * @return the float value.
	 * @throws DOMException
     *   INVALID_ACCESS_ERR if the CSS value can't be converted into 
     *   the natural unit. 
	 */
	public float getFloatValue() throws DOMException {
		StyleDatabase deviceStyleDb = getStyleDatabase();
		if(deviceStyleDb == null){
			throw new DOMException(DOMException.INVALID_ACCESS_ERR, 
					"Cannot transform value to natural unit.");
		}
		return deviceStyleDb.floatValueConversion(realvalue, getPrimitiveType());
	}

	@Override
	void setLexicalUnit(LexicalUnit lunit) {
		super.setLexicalUnit(lunit);
		switch(lunit.getLexicalUnitType()){
		case LexicalUnit.SAC_INTEGER:
			realvalue = lunit.getIntegerValue();
			break;
		default:
			realvalue = lunit.getFloatValue();
			dimensionUnitText = lunit.getDimensionUnitText();
		}
	}

	/**
	 * Gives the dimension unit String associated to the given CSS unit type.
	 * 
	 * @param unitType the CSS primitive unit type.
	 * @return the unit String.
	 */
	static String dimensionUnitString(short unitType) {
		switch (unitType) {
		case CSSPrimitiveValue.CSS_EMS:
			return "em";
		case CSSPrimitiveValue.CSS_EXS:
			return "ex";
		case CSSPrimitiveValue.CSS_PX:
			return "px";
		case CSSPrimitiveValue.CSS_IN:
			return "in";
		case CSSPrimitiveValue.CSS_CM:
			return "cm";
		case CSSPrimitiveValue.CSS_MM:
			return "mm";
		case CSSPrimitiveValue.CSS_PT:
			return "pt";
		case CSSPrimitiveValue.CSS_PC:
			return "pc";
		case CSSPrimitiveValue.CSS_PERCENTAGE:
			return "%";
		case CSSPrimitiveValue.CSS_DEG:
			return "deg";
		case CSSPrimitiveValue.CSS_GRAD:
			return "grad";
		case CSSPrimitiveValue.CSS_RAD:
			return "rad";
		case CSSPrimitiveValue.CSS_MS:
			return "ms";
		case CSSPrimitiveValue.CSS_S:
			return "s";
		case CSSPrimitiveValue.CSS_HZ:
			return "Hz";
		case CSSPrimitiveValue.CSS_KHZ:
			return "kHz";
		case CSSPrimitiveValue.CSS_DIMENSION:
		default:
			return "";
		}
	}
}
