/*

 Copyright (c) 2005-2007, Carlos Amengual.

 Licensed under a BSD-style License. You can find the license here:
 http://www.informatica.info/projects/css/LICENSE.txt

 */

package info.informatica.doc.style.css.visual.box;

import info.informatica.doc.style.css.CSS2ComputedProperties;
import info.informatica.doc.style.css.property.CSSNumberValue;
import info.informatica.doc.style.css.property.CSSStringValue;
import info.informatica.doc.style.css.visual.CSSContainerBox;

import org.w3c.dom.css.CSSPrimitiveValue;

/**
 * Abstract block box.
 * 
 * @author Carlos Amengual (amengual at informatica.info)
 *
 */
abstract public class AbstractBlockBox extends AbstractCSSBox {

	private final float UNSET_FLOAT_VALUE = Float.MIN_VALUE;

	private float boxWidth = UNSET_FLOAT_VALUE;
	private float boxHeight = UNSET_FLOAT_VALUE;
	
	public AbstractBlockBox(CSS2ComputedProperties style) {
		super(style);
	}

	public float getWidth() {
		if(boxWidth == UNSET_FLOAT_VALUE){
			CSSPrimitiveValue cssValue = getCSSValue("width");
			short primiType = cssValue.getPrimitiveType();
			if(primiType == getStyleDatabase().getNaturalUnit()) {
				boxWidth = cssValue.getFloatValue(getStyleDatabase().getNaturalUnit());
			} else if(primiType == CSSPrimitiveValue.CSS_PERCENTAGE) {
				boxWidth = getContainerWidth() * 
					cssValue.getFloatValue(CSSPrimitiveValue.CSS_PERCENTAGE) / 100f;
			} else if(cssValue instanceof CSSNumberValue) {
				// Expressed in non-natural unit
				boxWidth = cssValue.getFloatValue(primiType);
			} else {
				// Even if this was not suppiled as 'auto', will be handled as such.
				boxWidth = getContainerWidth() - getMarginLeft() - getBorderLeftWidth() 
					- getMarginRight() - getBorderRightWidth();
				// Should we take scroll into account ?
				if(overflowsWithScroll()) {
					boxWidth -= getStyleDatabase().getScrollbarWidth(); 
				}
				boxWidth -= getPaddingLeft();
				boxWidth -= getPaddingRight();
			}
		}
		return boxWidth;
	}

	public float getHeight() {
		if(boxHeight == UNSET_FLOAT_VALUE){
			CSSPrimitiveValue cssValue = getCSSValue("height");
			short primiType = cssValue.getPrimitiveType();
			if(primiType == getStyleDatabase().getNaturalUnit()) {
				boxHeight = cssValue.getFloatValue(getStyleDatabase().getNaturalUnit());
			} else if(primiType == CSSPrimitiveValue.CSS_PERCENTAGE) {
				if(isContainerHeightAuto()) {
					boxHeight =  computeAutoHeight(getWidth());
				} else {
					// Compute container height
					float containerHeight;
					CSSContainerBox container = getContainingBlock();
					if(container != null){
						containerHeight = container.getHeight();
					} else {
						// root block
						containerHeight = getStyleDatabase().getDocumentHeight();
					}
					boxHeight = containerHeight * 
						cssValue.getFloatValue(CSSPrimitiveValue.CSS_PERCENTAGE) / 100f;
				}
			} else if(cssValue instanceof CSSNumberValue) {
				// Expressed in non-natural unit
				boxHeight = cssValue.getFloatValue(primiType);
			} else {
				// Even if this was not supplied as 'auto', will be handled as such.
				boxHeight = computeAutoHeight(getWidth());
			}
		}
		return boxHeight;
	}

	abstract protected float computeAutoHeight(float containerWidth);

	/**
	 * Gets the value of the margin-top property, expressed in the 
	 * unit given by the <code>StyleDatabase.getNaturalUnit()</code> method.
	 * 
	 * @return  the value of the margin-top property.
	 */
	public float getMarginTop() {
		return computeMarginSubproperty("margin-top", false);
	}

	/**
	 * Gets the value of the margin-bottom property, expressed in the 
	 * unit given by the <code>StyleDatabase.getNaturalUnit()</code> method.
	 * 
	 * @return  the value of the margin-bottom property.
	 */
	public float getMarginBottom() {
		return computeMarginSubproperty("margin-bottom", false);
	}

	/**
	 * Gets the value of the margin-right property, expressed in the 
	 * unit given by the <code>StyleDatabase.getNaturalUnit()</code> method.
	 * 
	 * @return  the value of the margin-right property.
	 */
	public float getMarginRight() {
		return computeMarginSubproperty("margin-right", false);
	}

	/**
	 * Gets the value of the margin-left property, expressed in the 
	 * unit given by the <code>StyleDatabase.getNaturalUnit()</code> method.
	 * 
	 * @return  the value of the margin-left property.
	 */
	public float getMarginLeft() {
		return computeMarginSubproperty("margin-left", false);
	}

	protected float computeMarginSubproperty(String marginProperty, boolean auto) {
		float margin;
		CSSPrimitiveValue cssval = getCSSValue(marginProperty);
		if (cssval.getPrimitiveType() == CSSPrimitiveValue.CSS_PERCENTAGE) {
			margin = getContainerWidth()
					* cssval.getFloatValue(CSSPrimitiveValue.CSS_PERCENTAGE) / 100f;
		} else if (cssval.getPrimitiveType() == CSSPrimitiveValue.CSS_EMS) {
			margin = getComputedStyle().getFontSize()
					* cssval.getFloatValue(CSSPrimitiveValue.CSS_EMS);
		} else if (cssval instanceof CSSNumberValue) {
			margin = cssval.getFloatValue(getStyleDatabase().getNaturalUnit());
		} else if (cssval instanceof CSSStringValue) {
			// Identifier: can only be auto (or an error, which also means auto)
			// Now, let's see if width is auto (or we are called with auto == true)
			cssval = getCSSValue("width");
			if((cssval instanceof CSSStringValue) || auto) {
				// width: auto, or this is an automatic computation: margin becomes 0
				margin = 0;
			} else {
				// Compute automatic margin
				if(marginProperty.equals("margin-right")){
					margin = automaticRightMargin();
				} else if(marginProperty.equals("margin-left")){
					margin = automaticLeftMargin();
				} else {
					margin = 0;
				}
			}
		} else {
			margin = 0;
		}
		return margin;
	}

	protected float automaticRightMargin() {
		// See the opposed margin
		CSSPrimitiveValue cssval = getCSSValue("margin-left");
		float margin;
		if (cssval instanceof CSSStringValue) {
			// auto: margin-right = margin-left
			margin = (getContainerWidth() - getBorderLeftWidth() 
					- getBorderRightWidth() - getWidth()) / 2f;
		} else {
			// Is width 'auto' too ?
			CSSPrimitiveValue cssWidth = getCSSValue("width");
			if(cssWidth instanceof CSSStringValue) {
				// width is 'auto'
				margin = getMarginLeft();
			} else {
				// width is number
				margin = getContainerWidth() - getMarginLeft() - getBorderLeftWidth() 
						- getBorderRightWidth() - getWidth();
				// Should we take scroll into account ?
				if(overflowsWithScroll()) {
					margin -= getStyleDatabase().getScrollbarWidth(); 
				}
			}
		}
		return margin;
	}

	protected float automaticLeftMargin() {
		// See the opposed margin
		CSSPrimitiveValue cssval = getCSSValue("margin-right");
		float margin;
		if (cssval instanceof CSSStringValue) {
			// auto: margin-right = margin-left
			margin = (getContainerWidth() - getBorderLeftWidth() 
					- getBorderRightWidth()) / 2f;
		} else {
			// Is width 'auto' too ?
			CSSPrimitiveValue cssWidth = getCSSValue("width");
			if(cssWidth instanceof CSSStringValue) {
				// width is 'auto'
				margin = getMarginRight();
			} else {
				// width is number
				margin = getContainerWidth() - getMarginRight() - getBorderLeftWidth() 
						- getBorderRightWidth() - getWidth();
				// Should we take scroll into account ?
				if(overflowsWithScroll()) {
					margin -= getStyleDatabase().getScrollbarWidth(); 
				}
			}
		}
		return margin;
	}

	/**
	 * Computes the possible scrollbar width, in case that this box overflows with 
	 * scroll (see the 'overflow' property).
	 * 
	 * @return the scrollbar width if this box overflows with scroll, or zero if 
	 * don't overflows with scroll.
	 */
	protected float computeScrollbarWidth() {
		if(overflowsWithScroll()) {
			return getStyleDatabase().getScrollbarWidth(); 
		} else {
			return 0f;
		}
	}

}
