/*

 Copyright (c) 2005-2007, Carlos Amengual.

 Licensed under a BSD-style License. You can find the license here:
 http://www.informatica.info/projects/css/LICENSE.txt

 */

package info.informatica.doc.style.css.visual.box;

import info.informatica.doc.style.css.CSS2ComputedProperties;
import info.informatica.doc.style.css.dom.ComputedCSSStyle;
import info.informatica.doc.style.css.visual.CSSInlineBox;

import org.w3c.dom.css.CSSPrimitiveValue;

/**
 * CSS inline Box.
 * 
 * @author Carlos Amengual (amengual at informatica.info)
 *
 */
public class InlineBox extends AbstractInlineBox implements CSSInlineBox {
	
	public InlineBox(CSS2ComputedProperties style) {
		super(style);
	}

	/**
	 * Gets the (whitespace-trimmed) text content of this box.
	 * 
	 * @return the text content, or the empty string if the box 
	 * has no text.
	 */
	public String getText() {
		return ((ComputedCSSStyle)getComputedStyle()).getText();
	}

	@Override
	protected float getWidth() {
		return getContentWidth();
	}

	public float getContentWidth() {
		return textLengthToNaturalUnit(getText().length());
	}

	public float getMinimumWidth() {
		String text = getText();
		// find size of the longest word in text
		int tlen = text.length();
		int maxsz = 0;
		int j=0;
		int i=0;
		while(i<tlen) {
			if(!Character.isLetterOrDigit(text.charAt(i))){
				int k = i - j;
				if(k > maxsz) {
					maxsz = k;
				}
				j = i;
				j++;
				while(j<tlen && 
						!Character.isLetterOrDigit(text.charAt(j))) {
					j++;
				}
				i = j;
			} else {
				i++;
			}
		}
		if(j > 0) {
			if(tlen-j > maxsz) {
				maxsz = tlen - j;
			}
		} else {
			maxsz = tlen;
		}
		return textLengthToNaturalUnit(maxsz);
	}

	@Override
	protected float getHeight() {
		return getContentHeight();
	}

	public float getContentHeight() {
		return getStyleDatabase().floatValueConversion(getFontSize(), CSSPrimitiveValue.CSS_PT);
	}

	public float computeHeight(float width) {
		// FIXME: specify what happens when the scrollbar is permanent
		int chWidth = (int)(Math.floor(getStyleDatabase().floatValueConversion(
				width, getStyleDatabase().getNaturalUnit(), CSSPrimitiveValue.CSS_PT) / 
			getStyleDatabase().getExSizeInPt(getFontFamily(), getFontSize())));
		// chWidth is the width of the box, expressed in chars.
		if(chWidth == 0) {
			return Float.NaN;
		}
		String text = getText();
		int lineCount = 1;
		int tLen = text.length();
		if(chWidth <= tLen) {
			int lbound = 0;
			int i = 0;
			int from;
			do {
				from = i + chWidth;
				if(from >= tLen) {
					break;
				}
				i = lastNonLetterOrDigitIndex(text, from, lbound);
				if(i == lbound) {
					// Overflows
					i = nonLetterOrDigitIndex(text, from, chWidth);
				}
				i++;
				lbound = i;
				lineCount++;
			} while(i < tLen);
		}
		return getStyleDatabase().floatValueConversion(getFontSize()
				* lineCount, CSSPrimitiveValue.CSS_PT);
	}

	static int nonLetterOrDigitIndex(CharSequence seq, int fromIndex, 
			int upperBound) {
		int i = fromIndex;
		while(i < upperBound && Character.isLetterOrDigit(seq.charAt(i))) {
			i++;
		}
		return i;
	}

	static int lastNonLetterOrDigitIndex(CharSequence seq, int fromIndex, 
			int lowerBound) {
		int i = fromIndex;
		while(i > lowerBound && Character.isLetterOrDigit(seq.charAt(i))) {
			i--;
		}
		return i;
	}

	public float textLengthToNaturalUnit(int length) {
		return super.textLengthToNaturalUnit(length);
	}

}
