/*

 Copyright (c) 2005-2007, Carlos Amengual.

 Licensed under a BSD-style License. You can find the license here:
 http://www.informatica.info/projects/css/LICENSE.txt

 */

package info.informatica.doc.style.css.visual.box;

import org.w3c.dom.css.CSSPrimitiveValue;

import info.informatica.doc.style.css.CSS2ComputedProperties;
import info.informatica.doc.style.css.visual.CSSBox;
import info.informatica.doc.style.css.visual.ItemMarkerFactory;
import info.informatica.doc.style.css.visual.NonStaticallyPositioned;

/**
 * CSS list item block box.
 * 
 * @author Carlos Amengual (amengual at informatica.info)
 *
 */
public class ListItemBox extends BlockBox {
	
	private int orderNumber = 0;
	
	public ListItemBox(CSS2ComputedProperties style) {
		super(style);
	}

	public void setOrderNumber(int index) {
		orderNumber = index;
	}

	public int getOrderNumber() {
		return orderNumber;
	}

	/**
	 * Gets the next block box generated by the element that generated 
	 * this box.
	 * 
	 * @return the generated list-item inline block box, or null if 
	 * no marker is generated.
	 */
	public CSSBox getGeneratedSibling() {
		String listStyleType = getCSSValue("list-style-type").getCssText();
		if(listStyleType.equals("disc") || listStyleType.equals("circle")
				|| listStyleType.equals("square")) {
			return new ListItemMarkerBox(getComputedStyle());
		} else if(!listStyleType.equals("none")) {
			// Numbered marker
			ListItemNumberedMarkerBox markerBox =  new ListItemNumberedMarkerBox(
					getComputedStyle());
			ItemMarkerFactory factory;
			if(listStyleType.equals("decimal")) {
				factory = new DecimalItemMarkerFactory();
			} else if(listStyleType.equals("decimal-leading-zero")) {
				factory = new DecimalLeadingZeroItemMarkerFactory();
			} else {
				// TODO: rest of number types
				factory = new DecimalItemMarkerFactory();
			}
			markerBox.setItemMarkerFactory(factory);
			return markerBox;
		} else {
			return null;
		}
	}

	/**
	 * CSS list item marker box.
	 * 
	 * @author Carlos Amengual (amengual at informatica.info)
	 *
	 */
	public class ListItemMarkerBox extends AbstractCSSBox implements 
	NonStaticallyPositioned {

		private float left;

		private float right;
		
		private boolean positionInside = true;

		public ListItemMarkerBox(CSS2ComputedProperties style) {
			super(style);
			String stylePosition = getCSSValue("list-style-position").getCssText();
			if(stylePosition.equals("outside")) {
				positionInside = false;
			}
			computeDimension();
		}

		public void computeDimension() {
			if(isPositionInside()) {
				left = 0;
				right = getContainerWidth() - ListItemBox.this.getWidth();
			} else {
				/*
				 *  FIXME: non-defined 'auto' behavior for list item markers 
				 *  positioned outside
				 */
				left = ListItemBox.super.getLeft();
				right = ListItemBox.super.getRight();
			}
		}

		public boolean isPositionInside() {
			return positionInside;
		}

		/**
		 * Gets the marker type according to the <code>list-style-type</code>
		 * property value.
		 * 
		 * @return a String with the value of the <code>list-style-type</code>
		 * property.
		 */
		public String getMarkerType() {
			return getCSSValue("list-style-type").getCssText();
		}

		public float getWidth() {
			return getStyleDatabase().floatValueConversion(
					getFontSize() * 0.6f, CSSPrimitiveValue.CSS_PT);
		}

		public float getHeight() {
			return getStyleDatabase().floatValueConversion(
					getFontSize(), CSSPrimitiveValue.CSS_PT);
		}

		@Override
		public float getLeft() {
			return left;
		}

		@Override
		public float getRight() {
			return right;
		}
	}

	/**
	 * CSS list item numbered marker box.
	 * 
	 * @author Carlos Amengual (amengual at informatica.info)
	 *
	 */
	public class ListItemNumberedMarkerBox extends ListItemMarkerBox {
		
		private ItemMarkerFactory factory;
	
		public ListItemNumberedMarkerBox(CSS2ComputedProperties style) {
			super(style);
		}

		public void setItemMarkerFactory(ItemMarkerFactory factory) {
			this.factory = factory;
		}
	
		public String itemSymbol() {
			return factory.marker(getOrderNumber());
		}
	
	}

}
