/*

 Copyright (c) 1998-2006, Carlos Amengual.

 Licensed under a BSD-style License. You can find the license here:
 http://www.informatica.info/projects/css/LICENSE.txt

 */

package info.informatica.doc.xml.dtd;

import java.io.BufferedReader;
import java.io.IOException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.apache.log4j.Logger;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;



/**
 * XML Content Model information.
 * 
 * @author amengual at informatica dot info
 */
public class ContentModel {
	private static Map<String,ContentModel> contentModelMap = new HashMap<String,ContentModel>();

	private Set<String> emptyElements;

	static Logger log = Logger.getLogger(ContentModel.class.getName());

	protected ContentModel(DocumentTypeDeclaration dtDecl) throws SAXException, IOException {
		super();
		DefaultEntityResolver resolver = new DefaultEntityResolver();
		InputSource isrc = resolver.resolveEntity(dtDecl);
		if(isrc == null){
			throw new NullPointerException("Unable to resolve declaration " + dtDecl.toString());
		}
		BufferedReader re = new BufferedReader(isrc.getCharacterStream());
		Set<String> emptySet = new HashSet<String>(60);
		String line;
		while ((line = re.readLine()) != null) {
			if (line.length() == 0) {
				continue;
			}
			// Trim?
			if(line.charAt(0) == ' ') {
				line = line.trim();
			}
			// Get past comments
			if (line.startsWith("<!-- ")) {
				if (line.contains(" -->")) {
					continue;
				}
				do {
					line = re.readLine();
					if(line == null){
						break;
					}
				} while(!line.contains(" -->"));
			}
			// Check for ELEMENT
			// <!ELEMENT e EMPTY>
			// 012345678901234567
			if(line.startsWith("<!ELEMENT ")){
				int igt = line.indexOf('>', 10);
				if(igt < 16){
					// No room for ELEMENT name.
					continue;
				}
				if(line.regionMatches(igt - 6, " EMPTY", 0, 6)){
					emptySet.add(line.substring(10, igt-6).trim());
				}
			}
		}
		emptyElements = new HashSet<String>(emptySet.size());
		emptyElements.addAll(emptySet);
		emptySet.clear();
	}

	/**
	 * Gets an instance of ContentModel.
	 * 
	 * @return the ContentModel for the given filename, or null if not found.
	 */
	public synchronized static ContentModel getModel(String docTypeDecl)
	throws SAXException {
		if (!contentModelMap.containsKey(docTypeDecl)) {
			try {
				contentModelMap.put(docTypeDecl, new ContentModel(DocumentTypeDeclaration.parse(docTypeDecl)));
			} catch (RuntimeException e) {
				log.error("Unable to get content models for " + docTypeDecl, e);
			} catch (IOException e) {
				log.error("Unable to get content models for " + docTypeDecl, e);
			}
		}
		return contentModelMap.get(docTypeDecl);
	}

	public synchronized static ContentModel getXHTML1TransitionalModel() {
		try {
			return getModel(DocumentTypeDeclaration.XHTML1_TRA_DTDECL);
		} catch (SAXException e) {
			// This should not happen
			throw new IllegalStateException();
		}
	}
			
	/**
	 * Checks if the content model of the given element is EMPTY.
	 * 
	 * @param name
	 *            the element name.
	 * @return true if the content model is EMPTY, false otherwise.
	 */
	public boolean isEmpty(String name) {
		return emptyElements.contains(name);
	}

}
