/*
 * Decompiled with CFR 0.152.
 */
package com.crashlytics.tools.utils.elf;

import com.crashlytics.reloc.com.google.common.base.Charsets;
import com.crashlytics.reloc.com.google.common.base.Optional;
import com.crashlytics.tools.android.DeveloperTools;
import com.crashlytics.tools.utils.dwarf.DebugLineEntry;
import com.crashlytics.tools.utils.dwarf.DebugLineStateMachine;
import com.crashlytics.tools.utils.elf.ElfData;
import com.crashlytics.tools.utils.elf.ElfFileHeader;
import com.crashlytics.tools.utils.elf.ElfFileIdent;
import com.crashlytics.tools.utils.elf.ElfSectionHeader;
import com.crashlytics.tools.utils.elf.ElfSymbol;
import com.crashlytics.tools.utils.io.ByteReader;
import com.crashlytics.tools.utils.io.SeekableByteArrayInputStream;
import java.io.IOException;
import java.nio.ByteOrder;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;

public class ElfDataParser {
    private static final String[] ARM_ARCH = new String[]{"Pre-v4", "4", "4T", "5T", "5TE", "5TEJ", "6", "6KZ", "6T2", "6K", "7", "6-M", "6S-M", "7E-M", "8"};
    private static final int SHT_ARM_ATTRIBUTES = 0x70000003;
    private static final String ARM_ATTR_PUBLIC_SECTION = "aeabi";
    private static final int ARM_ATTR_TAG_FILE_ATTRIBUTES = 1;
    private static final String SECTION_DEBUG_LINE = ".debug_line";
    private final ByteReader _fileReader;
    private int _wordSize;

    public ElfDataParser(ByteReader reader) {
        this._fileReader = reader;
    }

    public ElfData parse() throws IOException {
        ElfFileIdent fileIdent = this.readFileIdent();
        if (!fileIdent.isElf()) {
            throw new IllegalArgumentException("Input is not a valid ELF file.");
        }
        this._wordSize = fileIdent.getElfClass() == 2 ? 8 : 4;
        this._fileReader.setByteOrder(fileIdent.getDataEncoding() == 2 ? ByteOrder.BIG_ENDIAN : ByteOrder.LITTLE_ENDIAN);
        ElfFileHeader fileHeader = this.readFileHeader(fileIdent);
        List<ElfSectionHeader> sectionHeaders = this.readSectionHeaders(fileHeader);
        List<ElfSymbol> symbols = this.readSymbols(sectionHeaders, fileIdent.getElfClass());
        Optional<String> armVersion = Optional.absent();
        if (fileHeader.eMachine == 40) {
            DeveloperTools.logD("Crashlytics detected ARM target in ELF data file.");
            Optional<ElfSectionHeader> armAttrSection = this.findArmAttributesSection(sectionHeaders);
            if (armAttrSection.isPresent()) {
                DeveloperTools.logD("Crashlytics found .ARM.attributes section.");
                ElfSectionHeader hdr = armAttrSection.get();
                armVersion = this.readArmVersion(hdr.shOffset, hdr.shSize);
            }
        }
        Optional<List<DebugLineEntry>> debugLineEntries = this.readDebugLineData(sectionHeaders);
        return new ElfData(fileHeader, sectionHeaders, symbols, debugLineEntries, armVersion);
    }

    private ElfFileIdent readFileIdent() throws IOException {
        this._fileReader.seek(0L);
        byte[] identBuffer = this._fileReader.readBytes(16);
        return new ElfFileIdent(identBuffer);
    }

    private ElfFileHeader readFileHeader(ElfFileIdent fileIdent) throws IOException {
        ElfFileHeader hdr = new ElfFileHeader(fileIdent);
        this._fileReader.seek(16L);
        hdr.eType = this._fileReader.readInt(2);
        hdr.eMachine = this._fileReader.readInt(2);
        hdr.eVersion = this._fileReader.readLong(4);
        hdr.eEntry = this._fileReader.readLong(this._wordSize);
        hdr.ePhoff = this._fileReader.readLong(this._wordSize);
        hdr.eShoff = this._fileReader.readLong(this._wordSize);
        hdr.eFlags = this._fileReader.readLong(4);
        hdr.eEhsize = this._fileReader.readInt(2);
        hdr.ePhentsize = this._fileReader.readInt(2);
        hdr.ePhnum = this._fileReader.readInt(2);
        hdr.eShentsize = this._fileReader.readInt(2);
        hdr.eShnum = this._fileReader.readInt(2);
        hdr.eShstrndx = this._fileReader.readInt(2);
        return hdr;
    }

    private List<ElfSectionHeader> readSectionHeaders(ElfFileHeader fileHeader) throws IOException {
        long sectionHeaderOffset = fileHeader.eShoff;
        int numSections = fileHeader.eShnum;
        int shNamesIndex = fileHeader.eShstrndx;
        ArrayList<ElfSectionHeader> hdrs = new ArrayList<ElfSectionHeader>(numSections);
        this._fileReader.seek(sectionHeaderOffset);
        for (int i = 0; i < numSections; ++i) {
            ElfSectionHeader sh = new ElfSectionHeader();
            sh.shName = this._fileReader.readInt(4);
            sh.shType = this._fileReader.readInt(4);
            sh.shFlags = this._fileReader.readLong(this._wordSize);
            sh.shAddr = this._fileReader.readLong(this._wordSize);
            sh.shOffset = this._fileReader.readLong(this._wordSize);
            sh.shSize = this._fileReader.readLong(this._wordSize);
            sh.shLink = this._fileReader.readInt(4);
            sh.shInfo = this._fileReader.readInt(4);
            sh.shAddrAlign = this._fileReader.readLong(this._wordSize);
            sh.shEntSize = this._fileReader.readLong(this._wordSize);
            hdrs.add(sh);
        }
        ElfSectionHeader shNames = (ElfSectionHeader)hdrs.get(shNamesIndex);
        this._fileReader.seek(shNames.shOffset);
        ByteReader stringTableReader = new ByteReader(new SeekableByteArrayInputStream(this._fileReader.readBytes((int)shNames.shSize)));
        for (ElfSectionHeader sh : hdrs) {
            stringTableReader.seek(sh.shName);
            sh.shNameString = stringTableReader.readNullTerminatedString(Charsets.UTF_8);
        }
        return hdrs;
    }

    private List<ElfSymbol> readSymbols(List<ElfSectionHeader> sectionHeaders, int alignment) throws IOException {
        LinkedList<ElfSymbol> symbols = new LinkedList<ElfSymbol>();
        for (ElfSectionHeader sh : sectionHeaders) {
            if (sh.shType != 2) continue;
            int numSymbols = (int)sh.shSize / (int)sh.shEntSize;
            ElfSectionHeader shStrings = sectionHeaders.get(sh.shLink);
            symbols.addAll(this.readSymbolTable(sh.shOffset, numSymbols, shStrings.shOffset, (int)shStrings.shSize, alignment));
        }
        return symbols;
    }

    private List<ElfSymbol> readSymbolTable(long symTabOffset, int numSymbols, long stringsOffset, int stringsSize, int alignment) throws IOException {
        this._fileReader.seek(stringsOffset);
        ByteReader stringTableReader = new ByteReader(new SeekableByteArrayInputStream(this._fileReader.readBytes(stringsSize)));
        this._fileReader.seek(symTabOffset);
        ArrayList<ElfSymbol> symbols = new ArrayList<ElfSymbol>(numSymbols);
        for (int i = 0; i < numSymbols; ++i) {
            ElfSymbol sym = new ElfSymbol();
            switch (alignment) {
                case 2: {
                    sym.stName = this._fileReader.readInt(4);
                    sym.stInfo = this._fileReader.readByte();
                    sym.stOther = this._fileReader.readByte();
                    sym.stShndx = this._fileReader.readShort(2);
                    sym.stValue = this._fileReader.readLong(this._wordSize);
                    sym.stSize = this._fileReader.readLong(this._wordSize);
                    break;
                }
                default: {
                    sym.stName = this._fileReader.readInt(4);
                    sym.stValue = this._fileReader.readLong(this._wordSize);
                    sym.stSize = this._fileReader.readLong(this._wordSize);
                    sym.stInfo = this._fileReader.readByte();
                    sym.stOther = this._fileReader.readByte();
                    sym.stShndx = this._fileReader.readShort(2);
                }
            }
            stringTableReader.seek(sym.stName);
            sym.stNameString = stringTableReader.readNullTerminatedString(Charsets.UTF_8);
            symbols.add(sym);
        }
        return symbols;
    }

    private Optional<List<DebugLineEntry>> readDebugLineData(List<ElfSectionHeader> sectionHeaders) throws IOException {
        LinkedList<DebugLineEntry> debugLineEntries = null;
        ElfSectionHeader debugLineHeader = null;
        for (ElfSectionHeader sh : sectionHeaders) {
            if (!SECTION_DEBUG_LINE.equals(sh.shNameString)) continue;
            debugLineHeader = sh;
            break;
        }
        if (debugLineHeader != null) {
            DebugLineStateMachine stateMachine = new DebugLineStateMachine();
            this._fileReader.seek(debugLineHeader.shOffset);
            byte[] sectionData = this._fileReader.readBytes((int)debugLineHeader.shSize);
            ByteReader debugLineReader = new ByteReader(new SeekableByteArrayInputStream(sectionData));
            debugLineReader.setByteOrder(this._fileReader.getByteOrder());
            debugLineEntries = new LinkedList<DebugLineEntry>();
            long offset = debugLineReader.getCurrentOffset();
            while (offset < (long)sectionData.length) {
                debugLineEntries.addAll(stateMachine.run(debugLineReader));
                offset = debugLineReader.getCurrentOffset();
            }
        }
        return Optional.fromNullable(debugLineEntries);
    }

    private Optional<ElfSectionHeader> findArmAttributesSection(List<ElfSectionHeader> sectionHeaders) {
        for (ElfSectionHeader hdr : sectionHeaders) {
            if (hdr.shType != 0x70000003) continue;
            return Optional.of(hdr);
        }
        return Optional.absent();
    }

    private Optional<String> readArmVersion(long dataOffset, long dataSize) throws IOException {
        this._fileReader.seek(dataOffset);
        ByteReader dataReader = new ByteReader(new SeekableByteArrayInputStream(this._fileReader.readBytes((int)dataSize)));
        dataReader.setByteOrder(this._fileReader.getByteOrder());
        byte version = dataReader.readByte();
        if (version != 65) {
            throw new IllegalArgumentException(String.format("Invalid data found at offset %d.", dataOffset));
        }
        long dataRemaining = dataSize - 1L;
        while (dataRemaining > 0L) {
            long sectionRemaining = dataReader.readInt(4);
            if (sectionRemaining > dataRemaining) {
                String errorString = "Section size %d is greater than remaining data length %d.";
                throw new IOException(String.format(errorString, sectionRemaining, dataRemaining));
            }
            dataRemaining -= sectionRemaining;
            sectionRemaining -= 4L;
            String sectionName = dataReader.readNullTerminatedString(Charsets.UTF_8);
            sectionRemaining -= (long)(sectionName.length() - 1);
            if (sectionName.equals(ARM_ATTR_PUBLIC_SECTION)) {
                DeveloperTools.logD("Crashlytics found ARM public attributes section.");
                return this.findArmVersionInSection(dataReader, sectionRemaining);
            }
            dataReader.seek(dataReader.getCurrentOffset() + sectionRemaining);
        }
        return Optional.absent();
    }

    private Optional<String> findArmVersionInSection(ByteReader dataReader, long sectionRemaining) throws IOException {
        while (sectionRemaining > 0L) {
            byte tag = dataReader.readByte();
            long subSectionRemaining = dataReader.readInt(4);
            if (subSectionRemaining > sectionRemaining) {
                String errorString = "Subsection size %d is greater than parent section size %d.";
                throw new IOException(String.format(errorString, subSectionRemaining, sectionRemaining));
            }
            sectionRemaining -= subSectionRemaining;
            subSectionRemaining -= 5L;
            if (tag == 1) {
                DeveloperTools.logD("Crashlytics found ARM File Attributes subsection.");
                return this.findArmVersionInSubSection(dataReader, subSectionRemaining);
            }
            dataReader.seek(dataReader.getCurrentOffset() + subSectionRemaining);
        }
        return Optional.absent();
    }

    private Optional<String> findArmVersionInSubSection(ByteReader dataReader, long subSectionRemaining) throws IOException {
        long nextSubSection = dataReader.getCurrentOffset() + subSectionRemaining;
        block4: while (dataReader.getCurrentOffset() < nextSubSection) {
            int attrTag = dataReader.readULEB128();
            switch (attrTag) {
                case 4: 
                case 5: 
                case 32: 
                case 65: 
                case 67: {
                    dataReader.readNullTerminatedString(Charsets.UTF_8);
                    continue block4;
                }
                case 6: {
                    int attrVal = dataReader.readULEB128();
                    DeveloperTools.logD("Crashlytics found ARM CPU archictecture field: " + attrVal);
                    return Optional.of(ARM_ARCH[attrVal]);
                }
            }
            dataReader.readULEB128();
        }
        return Optional.absent();
    }
}

