/*
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 * SPDX-License-Identifier: Apache-2.0
 * Copyright (c) 2023-2025 Jeremy Long. All Rights Reserved.
 */
package io.github.jeremylong.openvulnerability.client.nvd;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonPropertyDescription;
import com.fasterxml.jackson.annotation.JsonPropertyOrder;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;

import java.io.Serializable;
import java.util.List;
import java.util.Objects;

/**
 * Metric scores for a vulnerability as found on NVD.
 */
@JsonInclude(JsonInclude.Include.NON_NULL)
@JsonIgnoreProperties(ignoreUnknown = true)
@JsonPropertyOrder({"cvssMetricV40", "cvssMetricV31", "cvssMetricV30", "cvssMetricV2"})
public class Metrics implements Serializable {

    /**
     * Serialization version UID.
     */
    private static final long serialVersionUID = 3506888424662802743L;
    /**
     * CVSS V4.0 score.
     */
    @JsonProperty("cvssMetricV40")
    @JsonPropertyDescription("CVSS V4.0 score.")
    private List<CvssV4> cvssMetricV40;
    /**
     * CVSS V3.1 score.
     */
    @JsonProperty("cvssMetricV31")
    @JsonPropertyDescription("CVSS V3.1 score.")
    private List<CvssV3> cvssMetricV31;
    /**
     * CVSS V3.0 score.
     */
    @JsonProperty("cvssMetricV30")
    @JsonPropertyDescription("CVSS V3.0 score.")
    private List<CvssV3> cvssMetricV30;
    /**
     * CVSS V2.0 score.
     */
    @JsonProperty("cvssMetricV2")
    @JsonPropertyDescription("CVSS V2.0 score.")
    private List<CvssV2> cvssMetricV2;

    public Metrics() {
    }

    @SuppressFBWarnings(value = {"EI_EXPOSE_REP",
            "EI_EXPOSE_REP2"}, justification = "I prefer to suppress these FindBugs warnings")
    public Metrics(List<CvssV4> cvssMetricV40, List<CvssV3> cvssMetricV31, List<CvssV3> cvssMetricV30,
            List<CvssV2> cvssMetricV2) {
        this.cvssMetricV40 = cvssMetricV40;
        this.cvssMetricV31 = cvssMetricV31;
        this.cvssMetricV30 = cvssMetricV30;
        this.cvssMetricV2 = cvssMetricV2;
    }

    /**
     * CVSS V4.0 score.
     *
     * @return cvssMetricV40
     */
    @JsonProperty("cvssMetricV40")
    @SuppressFBWarnings(value = {"EI_EXPOSE_REP",
            "EI_EXPOSE_REP2"}, justification = "I prefer to suppress these FindBugs warnings")
    public List<CvssV4> getCvssMetricV40() {
        return cvssMetricV40;
    }

    /**
     * CVSS V3.1 score.
     *
     * @return cvssMetricV31
     */
    @JsonProperty("cvssMetricV31")
    @SuppressFBWarnings(value = {"EI_EXPOSE_REP",
            "EI_EXPOSE_REP2"}, justification = "I prefer to suppress these FindBugs warnings")
    public List<CvssV3> getCvssMetricV31() {
        return cvssMetricV31;
    }

    /**
     * CVSS V3.0 score.
     *
     * @return cvssMetricV30
     */
    @JsonProperty("cvssMetricV30")
    @SuppressFBWarnings(value = {"EI_EXPOSE_REP",
            "EI_EXPOSE_REP2"}, justification = "I prefer to suppress these FindBugs warnings")
    public List<CvssV3> getCvssMetricV30() {
        return cvssMetricV30;
    }

    /**
     * CVSS V2.0 score.
     *
     * @return cvssMetricV2
     */
    @JsonProperty("cvssMetricV2")
    @SuppressFBWarnings(value = {"EI_EXPOSE_REP",
            "EI_EXPOSE_REP2"}, justification = "I prefer to suppress these FindBugs warnings")
    public List<CvssV2> getCvssMetricV2() {
        return cvssMetricV2;
    }

    @Override
    public String toString() {
        return "Metrics{" + "cvssMetricV40=" + cvssMetricV40 + "cvssMetricV31=" + cvssMetricV31 + ", cvssMetricV30="
                + cvssMetricV30 + ", cvssMetricV2=" + cvssMetricV2 + '}';
    }

    @Override
    public boolean equals(Object o) {
        if (this == o)
            return true;
        if (o == null || getClass() != o.getClass())
            return false;
        Metrics metrics = (Metrics) o;
        return Objects.equals(cvssMetricV40, metrics.cvssMetricV40)
                && Objects.equals(cvssMetricV31, metrics.cvssMetricV31)
                && Objects.equals(cvssMetricV30, metrics.cvssMetricV30)
                && Objects.equals(cvssMetricV2, metrics.cvssMetricV2);
    }

    @Override
    public int hashCode() {
        return Objects.hash(cvssMetricV40, cvssMetricV31, cvssMetricV30, cvssMetricV2);
    }
}
