"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var Promise = require("bluebird");
require("isomorphic-fetch");
var jwt_1 = require("./jwt");
var utils_1 = require("./utils");
var config_1 = require("./config");
var dedupe_calls_1 = require("./utils/dedupe-calls");
var urlconfig_1 = require("./urlconfig");
var scopes_1 = require("./scopes");
var LoadingIndicator_1 = require("./LoadingIndicator");
var index_1 = require("./capability/index");
var refreshToken = null;
function isGetRequest(fetchOptions) {
    return !fetchOptions || !fetchOptions.method || 'get'.localeCompare(fetchOptions.method) === 0;
}
var FetchFunctions = /** @class */ (function () {
    function FetchFunctions() {
    }
    /**
     * Ensures the URL starts with jenkins path if not an absolute URL.
     * @param url
     * @returns {string}
     */
    FetchFunctions.prefixUrl = function (url) {
        if (url.indexOf('http') === 0) {
            return url;
        }
        if (urlconfig_1.UrlConfig.getJenkinsRootURL() !== '' && !url.startsWith(urlconfig_1.UrlConfig.getJenkinsRootURL())) {
            return "" + urlconfig_1.UrlConfig.getJenkinsRootURL() + url;
        }
        return url;
    };
    FetchFunctions.checkRefreshHeader = function (response) {
        var _refreshToken = response.headers.get('X-Blueocean-Refresher');
        // No token in response, lets just ignore.
        if (!_refreshToken) {
            return response;
        }
        // First time we have seen a refresh token, early exit.
        if (!refreshToken) {
            refreshToken = _refreshToken;
            return response;
        }
        // We need to refresh the page now!
        if (refreshToken !== _refreshToken) {
            utils_1.Utils.refreshPage();
            throw new Error('refreshing apge');
        }
        return response;
    };
    /**
     * This method checks for for 2XX http codes. Throws error it it is not.
     * This should only be used if not using fetch or fetchJson.
     */
    FetchFunctions.checkStatus = function (response) {
        if (response.status >= 300 || response.status < 200) {
            var message = "fetch failed: " + response.status + " for " + response.url;
            var error = new Error(message); //FIXME
            error.response = response;
            throw error;
        }
        return response;
    };
    FetchFunctions.stopLoadingIndicator = function (response) {
        LoadingIndicator_1.loadingIndicator.hide();
        return response;
    };
    /**
     * Adds same-origin option to the fetch.
     */
    FetchFunctions.sameOriginFetchOption = function (options) {
        if (options === void 0) { options = {}; }
        var newOpts = utils_1.Utils.clone(options);
        newOpts.credentials = newOpts.credentials || 'same-origin';
        return newOpts;
    };
    /**
     * Enhances the fetchOptions with the JWT bearer token. Will only be needed
     * if not using fetch or fetchJson.
     */
    FetchFunctions.jwtFetchOption = function (token, options) {
        if (options === void 0) { options = {}; }
        var newOpts = utils_1.Utils.clone(options);
        newOpts.headers = newOpts.headers || {};
        newOpts.headers['Authorization'] = newOpts.headers['Authorization'] || "Bearer " + token;
        return newOpts;
    };
    /**
     * REturns the json body from the response. It is only needed if
     * you are using FetchUtils.fetch
     *
     * Usage:
     * FetchUtils.fetch(..).then(FetchUtils.parseJSON)
     */
    FetchFunctions.parseJSON = function (response) {
        return (response
            .json()
            .catch(function (error) {
            if (error.message.indexOf('Unexpected end of JSON input') !== -1) {
                return {};
            }
            throw error;
        }));
    };
    /* eslint-disable no-param-reassign */
    /**
     * Parses the response body for the error generated in checkStatus.
     */
    FetchFunctions.parseErrorJson = function (error) {
        return error.response.json().then(function (body) {
            error.responseBody = body;
            throw error;
        }, function () {
            error.responseBody = null;
            throw error;
        });
    };
    /* eslint-enable no-param-reassign */
    /**
     * Error function helper to log errors to console.
     *
     * Usage;
     * fetchJson(..).catch(FetchUtils.consoleError)
     */
    FetchFunctions.consoleError = function (error) {
        console.error(error); // eslint-disable-line no-console
    };
    /**
     * Error function helper to call a callback on a rejected promise.
     * if callback is null, log to console). Use .catch() if you know it
     * will not be null though.
     *
     * Usage;
     * fetchJson(..).catch(FetchUtils.onError(error => //do something)
     */
    FetchFunctions.onError = function (errorFunc) {
        return function (error) {
            if (errorFunc) {
                errorFunc(error);
            }
            else {
                FetchFunctions.consoleError(error);
            }
        };
    };
    /**
     * Raw fetch that returns the json body.
     *
     * This method is semi-private, under normal conditions it should not be
     * used as it does not include the JWT bearer token
     *
     * @param {string} url - The URL to fetch from.
     * @param {Object} [options]
     * @param {function} [options.onSuccess] - Optional callback success function.
     * @param {function} [options.onError] - Optional error callback.
     * @param {Object} [options.fetchOptions] - Optional isomorphic-fetch options.
     * @param {boolean} [options.disableDedupe] - Optional flag to disable dedupe for this request.
     * @param {boolean} [options.disableLoadingIndicator] - Optional flag to disable loading indicator for this request.
     * @returns JSON body
     */
    FetchFunctions.rawFetchJSON = function (url, _a) {
        var _b = _a === void 0 ? {} : _a, onSuccess = _b.onSuccess, onError = _b.onError, fetchOptions = _b.fetchOptions, disableDedupe = _b.disableDedupe, disableLoadingIndicator = _b.disableLoadingIndicator, ignoreRefreshHeader = _b.ignoreRefreshHeader;
        var request = function () {
            var future = getPrefetchedDataFuture(url); // eslint-disable-line no-use-before-define
            if (!disableLoadingIndicator) {
                LoadingIndicator_1.loadingIndicator.show();
            }
            if (!future) {
                future = fetch(url, FetchFunctions.sameOriginFetchOption(fetchOptions));
                if (!ignoreRefreshHeader) {
                    future = future.then(FetchFunctions.checkRefreshHeader);
                }
                future = future.then(FetchFunctions.checkStatus).then(FetchFunctions.parseJSON, FetchFunctions.parseErrorJson);
                if (!disableLoadingIndicator) {
                    future = future.then(FetchFunctions.stopLoadingIndicator, function (err) {
                        FetchFunctions.stopLoadingIndicator(err);
                        throw err;
                    });
                }
            }
            else if (!disableLoadingIndicator) {
                LoadingIndicator_1.loadingIndicator.hide();
            }
            if (onSuccess) {
                return future.then(onSuccess).catch(FetchFunctions.onError(onError));
            }
            return future;
        };
        if (disableDedupe || !isGetRequest(fetchOptions)) {
            return request();
        }
        return dedupe_calls_1.dedupe(url, request);
    };
    /**
     * Raw fetch.
     *
     * This method is semi-private, under normal conditions it should not be
     * used as it does not include the JWT bearer token
     *
     * @param {string} url - The URL to fetch from.
     * @param {Object} [options]
     * @param {function} [options.onSuccess] - Optional callback success function.
     * @param {function} [options.onError] - Optional error callback.
     * @param {Object} [options.fetchOptions] - Optional isomorphic-fetch options.
     * @param {boolean} [options.disableDedupe] - Optional flag to disable dedupe for this request.
     * @param {boolean} [options.disableLoadingIndicator] - Optional flag to disable loading indicator for this request.
     * @returns fetch response
     */
    FetchFunctions.rawFetch = function (url, _a) {
        var _b = _a === void 0 ? {} : _a, onSuccess = _b.onSuccess, onError = _b.onError, fetchOptions = _b.fetchOptions, disableDedupe = _b.disableDedupe, disableLoadingIndicator = _b.disableLoadingIndicator, ignoreRefreshHeader = _b.ignoreRefreshHeader;
        var request = function () {
            var future = getPrefetchedDataFuture(url); // eslint-disable-line no-use-before-define
            if (!future) {
                if (!disableLoadingIndicator) {
                    LoadingIndicator_1.loadingIndicator.show();
                }
                future = fetch(url, FetchFunctions.sameOriginFetchOption(fetchOptions));
                if (!ignoreRefreshHeader) {
                    future = future.then(FetchFunctions.checkRefreshHeader);
                }
                future = future.then(FetchFunctions.checkStatus);
                if (!disableLoadingIndicator) {
                    future = future.then(FetchFunctions.stopLoadingIndicator, function (err) {
                        FetchFunctions.stopLoadingIndicator(err);
                        throw err;
                    });
                }
            }
            if (onSuccess) {
                return future.then(onSuccess).catch(FetchFunctions.onError(onError));
            }
            return future;
        };
        if (disableDedupe || !isGetRequest(fetchOptions)) {
            return request();
        }
        return dedupe_calls_1.dedupe(url, request);
    };
    return FetchFunctions;
}());
exports.FetchFunctions = FetchFunctions;
var Fetch = /** @class */ (function () {
    function Fetch() {
    }
    /**
     * Fetch JSON data.
     * <p>
     * Utility function that can be mocked for testing.
     *
     * @param {string} url - The URL to fetch from.
     * @param {Object} [options]
     * @param {function} [options.onSuccess] - Optional callback success function.
     * @param {function} [options.onError] - Optional error callback.
     * @param {Object} [options.fetchOptions] - Optional isomorphic-fetch options.
     * @returns JSON body.
     */
    Fetch.fetchJSON = function (url, _a) {
        var _b = _a === void 0 ? {} : _a, onSuccess = _b.onSuccess, onError = _b.onError, fetchOptions = _b.fetchOptions, disableCapabilities = _b.disableCapabilities, disableLoadingIndicator = _b.disableLoadingIndicator, ignoreRefreshHeader = _b.ignoreRefreshHeader;
        var fixedUrl = FetchFunctions.prefixUrl(url);
        var future;
        var crumbHeaderName = urlconfig_1.UrlConfig.getCrumbHeaderName();
        if (crumbHeaderName && fetchOptions && fetchOptions.headers) {
            fetchOptions.headers[crumbHeaderName] = urlconfig_1.UrlConfig.getCrumbToken();
        }
        if (!config_1.AppConfig.isJWTEnabled()) {
            future = FetchFunctions.rawFetchJSON(fixedUrl, { onSuccess: onSuccess, onError: onError, fetchOptions: fetchOptions, disableLoadingIndicator: disableLoadingIndicator, ignoreRefreshHeader: ignoreRefreshHeader });
        }
        else {
            future = jwt_1.JWT.getToken().then(function (token) {
                return FetchFunctions.rawFetchJSON(fixedUrl, {
                    onSuccess: onSuccess,
                    onError: onError,
                    fetchOptions: FetchFunctions.jwtFetchOption(token, fetchOptions),
                });
            });
        }
        if (!disableCapabilities) {
            return future.then(function (data) { return index_1.capabilityAugmenter.augmentCapabilities(utils_1.Utils.clone(data)); });
        }
        return future;
    };
    /**
     * Fetch data.
     * <p>
     * Utility function that can be mocked for testing.
     *
     * @param {string} url - The URL to fetch from.
     * @param {Object} [options]
     * @param {function} [options.onSuccess] - Optional callback success function.
     * @param {function} [options.onError] - Optional error callback.
     * @param {Object} [options.fetchOptions] - Optional isomorphic-fetch options.
     * @returns fetch body.
     */
    Fetch.fetch = function (url, _a) {
        var _b = _a === void 0 ? {} : _a, onSuccess = _b.onSuccess, onError = _b.onError, fetchOptions = _b.fetchOptions, disableLoadingIndicator = _b.disableLoadingIndicator, ignoreRefreshHeader = _b.ignoreRefreshHeader;
        var fixedUrl = FetchFunctions.prefixUrl(url);
        var crumbHeaderName = urlconfig_1.UrlConfig.getCrumbHeaderName();
        if (crumbHeaderName && fetchOptions && fetchOptions.headers) {
            fetchOptions.headers[crumbHeaderName] = urlconfig_1.UrlConfig.getCrumbToken();
        }
        if (!config_1.AppConfig.isJWTEnabled()) {
            return FetchFunctions.rawFetch(fixedUrl, { onSuccess: onSuccess, onError: onError, fetchOptions: fetchOptions, disableLoadingIndicator: disableLoadingIndicator, ignoreRefreshHeader: ignoreRefreshHeader });
        }
        return jwt_1.JWT.getToken().then(function (token) {
            return FetchFunctions.rawFetch(fixedUrl, {
                onSuccess: onSuccess,
                onError: onError,
                fetchOptions: FetchFunctions.jwtFetchOption(token, fetchOptions),
            });
        });
    };
    return Fetch;
}());
exports.Fetch = Fetch;
function trimRestUrl(url) {
    var REST_PREFIX = 'blue/rest/';
    var prefixOffset = url.indexOf(REST_PREFIX);
    if (prefixOffset !== -1) {
        return url.substring(prefixOffset);
    }
    return url;
}
function getPrefetchedDataFuture(url) {
    var trimmedUrl = trimRestUrl(url);
    for (var prop in scopes_1.prefetchdata) {
        if (scopes_1.prefetchdata.hasOwnProperty(prop)) {
            var preFetchEntry = scopes_1.prefetchdata[prop];
            if (preFetchEntry.restUrl && preFetchEntry.data) {
                // If the trimmed/normalized rest URL matches the url arg supplied
                // to the function, construct a pre-resolved future object containing
                // the prefetched data as the value.
                if (trimRestUrl(preFetchEntry.restUrl) === trimmedUrl) {
                    try {
                        return Promise.resolve(JSON.parse(preFetchEntry.data));
                    }
                    finally {
                        // Delete the preFetchEntry i.e. we only use these entries once. So, this
                        // works only for the first request for the data at that URL. Subsequent
                        // calls on that REST endpoint will result in a proper fetch. A local
                        // store needs to be used (redux/mobx etc) if you want to avoid multiple calls
                        // for the same data. This is not a caching layer/mechanism !!!
                        delete scopes_1.prefetchdata[prop];
                    }
                }
            }
        }
    }
    return undefined;
}
