'use strict';

Object.defineProperty(exports, "__esModule", {
    value: true
});
exports.JWT = undefined;

var _bluebird = require('bluebird');

var _bluebird2 = _interopRequireDefault(_bluebird);

var _isomorphicFetch = require('isomorphic-fetch');

var _isomorphicFetch2 = _interopRequireDefault(_isomorphicFetch);

var _jsonwebtoken = require('jsonwebtoken');

var _jsonwebtoken2 = _interopRequireDefault(_jsonwebtoken);

var _fetch = require('./fetch');

var _pemJwk = require('pem-jwk');

var _config = require('./config');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

var storedToken = null;
var publicKeyStore = null;
var tokenFetchPromise = null;

var CLOCK_SKEW_SECONDS = 60;
var JWT = exports.JWT = {
    /**
     * Fetches the JWT token. This token is cached for a default of 25mins.
     * If it is within 5mins or expiry it will fetch a new one.
     */
    fetchJWT: function fetchJWT() {
        if (storedToken && storedToken.exp) {
            var diff = storedToken.exp - Math.trunc(new Date().getTime() / 1000);

            // refetch token if we are within 60s of it exp
            if (diff < CLOCK_SKEW_SECONDS) {
                tokenFetchPromise = null;
            }
        }

        if (!tokenFetchPromise) {
            tokenFetchPromise = (0, _isomorphicFetch2.default)(_config.AppConfig.getJWTServiceHostUrl() + '/jwt-auth/token', { credentials: 'include', mode: 'cors' }).then(this.checkStatus).then(function (response) {
                var token = response.headers.get('X-BLUEOCEAN-JWT');
                if (token) {
                    return token;
                }

                throw new Error('Could not fetch jwt_token');
            });
        }

        return tokenFetchPromise;
    },


    /**
     * Verifies the token using the public key.
     */
    verifyToken: function verifyToken(token, certObject) {
        return new _bluebird2.default(function (resolve, reject) {
            return _jsonwebtoken2.default.verify(token, (0, _pemJwk.jwk2pem)(certObject), { algorithms: [certObject.alg], clockTolerance: CLOCK_SKEW_SECONDS }, function (err, payload) {
                if (err) {
                    reject(err);
                } else {
                    resolve(payload);
                }
            });
        });
    },


    /**
     * Fetches the public key that is used to verify tokens.
     */
    fetchJWTPublicKey: function fetchJWTPublicKey(token) {
        var _this = this;

        var decoded = _jsonwebtoken2.default.decode(token, { complete: true });
        var url = _config.AppConfig.getJWTServiceHostUrl() + '/jwt-auth/jwks/' + decoded.header.kid + '/';
        if (!publicKeyStore) {
            publicKeyStore = (0, _isomorphicFetch2.default)(url, { credentials: 'same-origin' }).then(_fetch.FetchFunctions.checkStatus).then(_fetch.FetchFunctions.parseJSON).then(function (cert) {
                return _this.verifyToken(token, cert).then(function (payload) {
                    return {
                        token: token,
                        payload: payload
                    };
                });
            });
        }

        return publicKeyStore;
    },


    /**
     * Puts the token into global storage for later use.
     */
    storeToken: function storeToken(data) {
        storedToken = data.payload;
        return data;
    },


    /**
     * Use this function if you want the payload from the token.
     */
    getTokenWithPayload: function getTokenWithPayload() {
        var _this2 = this;

        return this.fetchJWT().then(_fetch.FetchFunctions.checkStatus).then(function (token) {
            return _this2.fetchJWTPublicKey(token);
        }).then(function (data) {
            return _this2.storeToken(data);
        });
    },


    /**
     * Gets the token from te server and verifies it.
     */
    getToken: function getToken() {
        return this.getTokenWithPayload().then(function (token) {
            return token.token;
        });
    }
};
//# sourceMappingURL=jwt.js.map
