'use strict';

Object.defineProperty(exports, "__esModule", {
    value: true
});

var _createClass = function () { function defineProperties(target, props) { for (var i = 0; i < props.length; i++) { var descriptor = props[i]; descriptor.enumerable = descriptor.enumerable || false; descriptor.configurable = true; if ("value" in descriptor) descriptor.writable = true; Object.defineProperty(target, descriptor.key, descriptor); } } return function (Constructor, protoProps, staticProps) { if (protoProps) defineProperties(Constructor.prototype, protoProps); if (staticProps) defineProperties(Constructor, staticProps); return Constructor; }; }();

function _classCallCheck(instance, Constructor) { if (!(instance instanceof Constructor)) { throw new TypeError("Cannot call a class as a function"); } }

var DefaultSSEHandler = exports.DefaultSSEHandler = function () {
    function DefaultSSEHandler(pipelineService, activityService, pagerService) {
        var _this = this;

        _classCallCheck(this, DefaultSSEHandler);

        this.handleEvents = function (event) {
            switch (event.jenkins_event) {
                case 'job_run_paused':
                case 'job_run_unpaused':
                    _this.updateJob(event);
                    break;
                case 'job_crud_created':
                    // Refetch pagers here. This will pull in the newly created pipeline into the bunker.
                    _this.pipelineService.refreshPagers();
                    break;
                case 'job_crud_deleted':
                    // Remove directly from bunker. No need to refresh bunkers as it will just show one less item.
                    _this.pipelineService.removeItem(event.blueocean_job_rest_url);
                    break;
                case 'job_crud_renamed':
                    // TODO: Implement this.
                    // Seems to be that SSE fires an updated event for the old job,
                    // then a rename for the new one. This is somewhat confusing for us.
                    break;
                case 'job_run_queue_buildable':
                case 'job_run_queue_enter':
                case 'job_run_queue_blocked':
                    _this.queueEnter(event);
                    break;
                case 'job_run_queue_left':
                    _this.queueLeft(event);
                    break;
                case 'job_run_started':
                    {
                        _this.updateJob(event);
                        break;
                    }
                case 'job_run_ended':
                    {
                        _this.updateJob(event);
                        break;
                    }
                default:
                // Else ignore the event.
            }
        };

        this.pipelineService = pipelineService;
        this.activityService = activityService;
        this.pagerService = pagerService;
        this.loggingEnabled = false;
    }

    _createClass(DefaultSSEHandler, [{
        key: 'branchPagerKeys',
        value: function branchPagerKeys(event) {
            if (!event.blueocean_job_branch_name) {
                return [this.activityService.pagerKey(event.jenkins_org, event.blueocean_job_pipeline_name)];
            }
            return [this.activityService.pagerKey(event.jenkins_org, event.blueocean_job_pipeline_name), this.activityService.pagerKey(event.jenkins_org, event.blueocean_job_pipeline_name, event.blueocean_job_branch_name)];
        }
    }, {
        key: 'updateJob',
        value: function updateJob(event) {
            // const queueId = event.job_run_queueId;
            // const queueSelf = `${event.blueocean_job_rest_url}queue/${queueId}/`;
            var href = event.blueocean_job_rest_url + 'runs/' + event.jenkins_object_id + '/';
            this._updateRun(event, href);
        }
    }, {
        key: 'queueCancel',
        value: function queueCancel(event) {
            if (event.job_run_status === 'CANCELLED') {
                var id = event.blueocean_queue_item_expected_build_number;
                var href = event.blueocean_job_rest_url + 'runs/' + id + '/';
                this._removeRun(event, href);
            }
        }
    }, {
        key: 'queueEnter',
        value: function queueEnter(event) {
            // Ignore the event if there's no branch name. Usually indicates
            // that the event is wrt MBP indexing.
            if (event.job_ismultibranch && !event.blueocean_job_branch_name) {
                return;
            }
            // don't care about indexing events
            if (event.job_multibranch_indexing_status === 'INDEXING') {
                return;
            }
            // If we have a queued item but the branch isn't present, we need to refresh the pager
            // this happens, for example, when you create a new pipeline in a repo that did not have one
            var pipeline = this.pipelineService.getPipeline('/blue/rest/organizations/' + event.jenkins_org + '/pipelines/' + encodeURIComponent(event.blueocean_job_pipeline_name) + '/');
            if (pipeline && pipeline.branchNames.indexOf(event.blueocean_job_branch_name) === -1) {
                this.pipelineService.pipelinesPager(event.jenkins_org, event.blueocean_job_pipeline_name).refresh();
            }
            // Sometimes we can't match the queue item so we have to skip this event
            if (!event.blueocean_queue_item_expected_build_number) {
                return;
            }
            var id = event.blueocean_queue_item_expected_build_number;
            var href = event.blueocean_job_rest_url + 'runs/' + id + '/';
            this._updateRun(event, href);
        }
    }, {
        key: 'queueLeft',
        value: function queueLeft(event) {
            // ignore the event if there's no build number
            // it's not related to a run, rather something like repo or branch indexing
            if (!event.blueocean_queue_item_expected_build_number) {
                return;
            }

            var id = event.blueocean_queue_item_expected_build_number;
            var href = event.blueocean_job_rest_url + 'runs/' + id + '/';

            if (event.job_run_status === 'CANCELLED') {
                // Cancelled runs are removed from the stores. They are gone *poof*.
                this._removeRun(event, href);
            } else {
                // If not cancelled then the state may be leaving the queue to execute and should be updated with latest
                this._updateRun(event, href);
            }
        }

        /**
         * Removes the run from the activity service and any branch pagers
         * @param event triggering the removal
         * @param href of the run to remove
         * @private
         */

    }, {
        key: '_removeRun',
        value: function _removeRun(event, href) {
            this.activityService.removeItem(href);
            var _iteratorNormalCompletion = true;
            var _didIteratorError = false;
            var _iteratorError = undefined;

            try {
                for (var _iterator = this.branchPagerKeys(event)[Symbol.iterator](), _step; !(_iteratorNormalCompletion = (_step = _iterator.next()).done); _iteratorNormalCompletion = true) {
                    var key = _step.value;

                    var pager = this.pagerService.getPager({ key: key });
                    if (pager) {
                        pager.remove(href);
                    }
                }
            } catch (err) {
                _didIteratorError = true;
                _iteratorError = err;
            } finally {
                try {
                    if (!_iteratorNormalCompletion && _iterator.return) {
                        _iterator.return();
                    }
                } finally {
                    if (_didIteratorError) {
                        throw _iteratorError;
                    }
                }
            }
        }

        /**
         * Fetches the latest activity for this run, updates activity service and any branch pagers
         * @param event triggering the fetch
         * @param href of the run to add
         * @private
         */

    }, {
        key: '_updateRun',
        value: function _updateRun(event, href) {
            var _this2 = this;

            var pipelineHref = this._computePipelineHref(event);
            var logMessage = event.jenkins_event + ' for pipeline ' + pipelineHref + ' with run ' + href;

            if (!this.pipelineService.hasItem(pipelineHref)) {
                this.loggingEnabled && console.log('aborting fetch for ' + logMessage);
                return;
            }

            this.loggingEnabled && console.log('fetch ' + logMessage);

            this.activityService.fetchActivity(href, { useCache: false, disableLoadingIndicator: true }).then(function (run) {
                var _iteratorNormalCompletion2 = true;
                var _didIteratorError2 = false;
                var _iteratorError2 = undefined;

                try {
                    for (var _iterator2 = _this2.branchPagerKeys(event)[Symbol.iterator](), _step2; !(_iteratorNormalCompletion2 = (_step2 = _iterator2.next()).done); _iteratorNormalCompletion2 = true) {
                        var key = _step2.value;

                        var pager = _this2.pagerService.getPager({ key: key });
                        if (pager && !pager.has(href)) {
                            pager.insert(href);
                        }
                    }
                } catch (err) {
                    _didIteratorError2 = true;
                    _iteratorError2 = err;
                } finally {
                    try {
                        if (!_iteratorNormalCompletion2 && _iterator2.return) {
                            _iterator2.return();
                        }
                    } finally {
                        if (_didIteratorError2) {
                            throw _iteratorError2;
                        }
                    }
                }

                if (run) {
                    _this2.pipelineService.updateLatestRun(run);

                    /*
                        Check to see if the TestSummary has been loaded and if so then reload it. Otherwise don't because
                        it's expensive to calculate.
                     */

                    var testResultUrl = run._links.blueTestSummary && run._links.blueTestSummary.href;
                    if (_this2.activityService.hasItem(testResultUrl)) {
                        _this2.activityService.fetchTestSummary(testResultUrl, { useCache: false, disableLoadingIndicator: true });
                    }
                }
            });
        }

        /**
         * Compute the REST URL / href for the job referenced in the supplied server side event.
         * @param event
         * @returns {string}
         * @private
         */

    }, {
        key: '_computePipelineHref',
        value: function _computePipelineHref(event) {
            var jobRestUrl = event.blueocean_job_rest_url;

            if (event.blueocean_job_branch_name) {
                // trim the last two path segments (e.g. 'branches/branch-name')
                jobRestUrl = jobRestUrl.split('/').filter(function (p) {
                    return p;
                }).slice(0, -2).join('/');
            }
            // ensure leading / trailing slashes
            if (jobRestUrl.slice(0, 1) !== '/') {
                jobRestUrl = '/' + jobRestUrl;
            }
            if (jobRestUrl.slice(-1) !== '/') {
                jobRestUrl += '/';
            }
            return jobRestUrl;
        }
    }]);

    return DefaultSSEHandler;
}();
//# sourceMappingURL=DefaultSSEHandler.js.map
