'use strict';

Object.defineProperty(exports, "__esModule", {
    value: true
});
exports.AppConfig = undefined;

var _scopes = require('./scopes');

var config = _scopes.blueocean.config || {}; /**
                                              * This config object comes from blueocean-config.
                                              */

var features = config.features || {};
var organization = _scopes.blueocean.organization || {};

// any all features added by ?features=SOMETHING,SOMETHING_ELSE
var pfx = 'features=';
var pfxlen = pfx.length;
var condition = window && window.location && window.location.href && window.location.href.split instanceof Function && window.location.href.split('?').length > 0;
var query = condition ? window.location.href.split('?')[1] : undefined;
if (query) {
    query.split('&').forEach(function (p) {
        return p.startsWith(pfx) && p.substring(pfxlen).split(',').forEach(function (f) {
            features[f] = true;
        });
    });
}
var AppConfig = exports.AppConfig = {
    loadUrls: function loadUrls() {
        try {
            var headElement = document.getElementsByTagName('head')[0];

            // Look up where the Blue Ocean app is hosted
            config.blueoceanAppURL = headElement.getAttribute('data-appurl');

            // load server skewTime
            config.serverBrowserTimeSkewMillis = headElement.getAttribute('data-servertime') - Date.now();
            if (typeof config.blueoceanAppURL !== 'string') {
                config.blueoceanAppURL = '/';
            }

            config.jenkinsRootURL = headElement.getAttribute('data-rooturl');
            config.resourceUrl = headElement.getAttribute('data-resurl');

            config.isLoaded = true;
        } catch (e) {
            // headless escape
            config.jenkinsRootURL = '/jenkins';
            config.serverBrowserTimeSkewMillis = 0;
        }
    },
    getConfig: function getConfig() {
        return config;
    },
    getJenkinsConfig: function getJenkinsConfig() {
        return config.jenkinsConfig || {};
    },
    getOrganizationName: function getOrganizationName() {
        var encoded = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : true;

        return encoded ? encodeURIComponent(organization.name) : organization.name;
    },
    getOrganizationDisplayName: function getOrganizationDisplayName() {
        return organization.displayName;
    },
    getOrganizationGroup: function getOrganizationGroup() {
        return organization.organizationGroup;
    },
    getSecurityConfig: function getSecurityConfig() {
        return this.getJenkinsConfig().security || {};
    },
    getAnalyticsEnabled: function getAnalyticsEnabled() {
        return this.getJenkinsConfig().analytics || false;
    },
    isJWTEnabled: function isJWTEnabled() {
        return !!this.getSecurityConfig().enableJWT;
    },
    getJWTServiceHostUrl: function getJWTServiceHostUrl() {
        return this.getSecurityConfig().jwtServiceHostUrl;
    },
    getLoginUrl: function getLoginUrl() {
        return this.getSecurityConfig().loginUrl;
    },
    getPluginInfo: function getPluginInfo(pluginId) {
        return _scopes.blueocean.jsExtensions.find(function (pluginInfo) {
            return pluginInfo.hpiPluginId === pluginId;
        });
    },
    isFeatureEnabled: function isFeatureEnabled(name, defaultValue) {
        var value = features[name];
        if (typeof value === 'boolean') {
            return value;
        }
        if (typeof defaultValue === 'boolean') {
            return defaultValue;
        }
        return false;
    },
    showOrg: function showOrg() {
        return this.isFeatureEnabled('organizations.enabled', false);
    },
    getJenkinsRootURL: function getJenkinsRootURL() {
        if (!config.isLoaded) {
            this.loadUrls();
        }
        return typeof config.jenkinsRootURL === 'string' ? config.jenkinsRootURL : '/jenkins';
    },
    getResourceURL: function getResourceURL() {
        if (!config.isLoaded) {
            this.loadUrls();
        }
        return typeof config.resourceUrl === 'string' ? config.resourceUrl : '/jenkins/static';
    },
    getServerBrowserTimeSkewMillis: function getServerBrowserTimeSkewMillis() {
        if (!config.isLoaded) {
            this.loadUrls();
        }
        return typeof config.serverBrowserTimeSkewMillis === 'number' ? config.serverBrowserTimeSkewMillis : 0;
    },
    getRestRoot: function getRestRoot() {
        return this.getJenkinsRootURL() + '/blue/rest';
    },


    /**
     * Set a new "jenkinsConfig" object.
     * Useful for testing in a headless environment.
     * @param newConfig
     * @private
     */
    _setJenkinsConfig: function _setJenkinsConfig(newConfig) {
        config.jenkinsConfig = newConfig;
    }
};
//# sourceMappingURL=config.js.map
