/*
 * This file is part of the Meeds project (https://meeds.io/).
 * Copyright (C) 2020 - 2025 Meeds Association contact@meeds.io
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package io.meeds.billing.service;

import com.stripe.model.Subscription;
import io.meeds.billing.storage.HubBillingSettingsStorage;
import io.meeds.billing.utils.Utils;
import org.exoplatform.commons.ObjectAlreadyExistsException;
import org.exoplatform.social.core.identity.model.Identity;
import org.exoplatform.social.core.space.model.Space;
import org.exoplatform.social.core.space.spi.SpaceService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.exoplatform.commons.exception.ObjectNotFoundException;

import io.meeds.billing.model.HubBillingSettings;

import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Optional;

@Service
public class HubSettingService {

  @Autowired
  private HubBillingSettingsStorage hubBillingSettingsStorage;

  @Autowired
  private HubBillingPlanService     hubBillingPlanService;

  @Autowired
  private SpaceService              spaceService;


  /**
   * Save the settings of the billable hub
   */
  public void createSettings(Space space, Subscription subscription, Identity identity) throws ObjectAlreadyExistsException {
    HubBillingSettings hubBillingSettings = getSettingsBySpaceId(space.getId());
    if (hubBillingSettings != null) {
      throw new ObjectAlreadyExistsException("Hub billing settings already exists for space " + space.getDisplayName());
    }
    hubBillingSettings = new HubBillingSettings();
    hubBillingSettings.setSpaceId(space.getSpaceId());
    hubBillingSettings.setUserIdentityId(Long.parseLong(identity.getId()));
    hubBillingSettings.setCreatedDate(new Date());
    hubBillingSettings.setUpdatedDate(new Date());
    hubBillingSettings.setCustomerId(subscription.getCustomer());
    hubBillingSettings.setSubscriptionId(subscription.getId());
    hubBillingSettings.setSubscriptionStatus(subscription.getStatus());
    hubBillingSettings.setSubscriptionPreviousStatus(subscription.getStatus());
    hubBillingSettings.setSubscriptionCurrentPeriodEnd(Utils.getCurrentPeriodEndDate(subscription));
    hubBillingSettings.setHubBillingPlan(hubBillingPlanService.getOrCreateHubBillingPlan(subscription));
    hubBillingSettingsStorage.save(hubBillingSettings);
  }

  /**
   * Save the settings of the billable hub
   */
  public void updateSettings(HubBillingSettings hubBillingSettings) {
    hubBillingSettingsStorage.save(hubBillingSettings);
  }
  
  /**
   * Get the billable hub settings
   */
  public HubBillingSettings getSettingsBySpaceId(Long spaceId, String authenticatedUser) throws ObjectNotFoundException, IllegalAccessException {
    Space space = spaceService.getSpaceById(spaceId);
    if (space == null) {
      throw new ObjectNotFoundException("Space not found for Space Id: " + spaceId);
    }
    if (!spaceService.canManageSpace(space, authenticatedUser)) {
      throw new IllegalAccessException("User" + authenticatedUser + " is not allowed to mange space " + spaceId);
    }
    return hubBillingSettingsStorage.getHubBillingSettingsBySpaceId(spaceId);
  }

  /**
   * Get the billable hub settings
   */
  public HubBillingSettings getSettingsBySubscriptionId(String subscriptionId) throws  ObjectNotFoundException {
    return hubBillingSettingsStorage.getHubBillingSettingBySubscriptionId(subscriptionId);
  }

  /**
   * Get the billable hub settings
   */
  public HubBillingSettings getSettingsBySpaceId(String spaceId) {
    try {
      return hubBillingSettingsStorage.getHubBillingSettingsBySpaceId(Long.parseLong(spaceId));
    } catch (Exception e) {
      return null;
    }
  }

  /**
   * Update the subscription status of the billable hub settings
   */
  public void updateSubscriptionStatus(Long billingHubId, String newSubscriptionStatus) {
    hubBillingSettingsStorage.updateSubscriptionStatus(billingHubId, newSubscriptionStatus);
  }
  
  public void updateHuBillingSetting(HubBillingSettings hubBillingSettings,
                                     Subscription subscription,
                                     Map<String, Object> previousAttributes) {
    String previousStatus = Optional.ofNullable(previousAttributes)
                                    .map(attributes -> (String) attributes.get("status"))
                                    .orElse(null);
    String newStatus = subscription.getStatus();
    Date currentPeriodEnd = Utils.getCurrentPeriodEndDate(subscription);
    boolean keepTrialing = Utils.SubscriptionStatus.TRIALING.name().equalsIgnoreCase(previousStatus)
        && Utils.SubscriptionStatus.ACTIVE.name().equalsIgnoreCase(newStatus);
    if (keepTrialing) {
      // Keep trialing status to allow proper trial expiration handling
      // Stripe temporarily switches to "active" for 1 hour right after the trial ends;
      // it will be updated by the invoice payment event.
      newStatus = Utils.SubscriptionStatus.TRIALING.name();
      currentPeriodEnd = hubBillingSettings.getSubscriptionCurrentPeriodEnd();
    }
    hubBillingSettings.setSubscriptionPreviousStatus(hubBillingSettings.getSubscriptionStatus());
    hubBillingSettings.setSubscriptionStatus(newStatus);
    hubBillingSettings.setSubscriptionCurrentPeriodEnd(currentPeriodEnd);
    hubBillingSettings.setHubBillingPlan(hubBillingPlanService.getOrCreateHubBillingPlan(subscription));
    hubBillingSettingsStorage.save(hubBillingSettings);
  }

  /**
   * Retrieves all hub billing settings that are associated with a Stripe meter.
   * 
   * @return a list of {@link HubBillingSettings} objects that have an associated
   *         meter ID; an empty list if none are found
   */
  public List<HubBillingSettings> getMeteredHubBillingSettingList() {
    return this.hubBillingSettingsStorage.getMeteredHubBillingSettingList();
  }
}
