/*
 * This file is part of the Meeds project (https://meeds.io/).
 * Copyright (C) 2020 - 2025 Meeds Association contact@meeds.io
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package io.meeds.billing.rest;

import io.meeds.billing.service.WebhookHandlingService;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.tags.Tag;
import org.exoplatform.commons.exception.ObjectNotFoundException;
import org.exoplatform.services.log.ExoLogger;
import org.exoplatform.services.log.Log;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestHeader;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.server.ResponseStatusException;

@RestController
@RequestMapping("/webhooks")
@Tag(name = "webhooks", description = "An endpoint to receive stripe webhooks")
public class BillingWebHookRest {

  private static final Log LOG = ExoLogger.getLogger(BillingWebHookRest.class);

  @Autowired
  WebhookHandlingService   webhookHandlingService;

  @PostMapping(path = "stripe")
  @Operation(summary = "Stripe Webhook Endpoint", description = "Receives and processes webhook events from Stripe.")
  @ApiResponse(responseCode = "200", description = "Ok")
  @ApiResponse(responseCode = "500", description = "Internal Server Error")
  @ResponseStatus(HttpStatus.OK)
  public void stripeEvent(@RequestBody String payload,
                          @RequestHeader("Stripe-Signature") String sigHeader) {
    try {
      webhookHandlingService.handleWebHookEvent(payload, sigHeader);
    } catch (ObjectNotFoundException objectNotFoundException) {
      LOG.warn("Received a Stripe webhook event not associated with any hub, Event ignored: {}", objectNotFoundException.getMessage());
    } catch (Exception e) {
      LOG.error("Unexpected error while handling Stripe webhook event");
      throw new ResponseStatusException(HttpStatus.INTERNAL_SERVER_ERROR, e.getMessage());
    }
  }
}
